// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   core/conformation/symmetry/SymmetricConformation.hh
/// @brief  symmetry conformation container.
//					Contains overloaded functions needed to
//					make changes in conformation symmetric
/// @author Phil Bradley, Ingemar Andre

// Unit headers
#include <core/conformation/symmetry/SymmetricConformation.hh>

#include <core/util/Tracer.hh>


namespace core {
namespace conformation {
namespace symmetry {

static core::util::Tracer TR("core.conformation.symmetry.Conformation");

/// @brief copy constructor
SymmetricConformation::SymmetricConformation( SymmetricConformation const & src ):
	Conformation( src ),
	symm_info_( src.symm_info_ )
{}

/// @brief operator=
Conformation &
SymmetricConformation::operator=( SymmetricConformation const & src )
{
	// will this work?
	Conformation::operator=( src );
	symm_info_ = src.symm_info_;
	return *this;
}

///@details make a copy of this conformation( allocate actual memory for it )
ConformationOP
SymmetricConformation::clone() const
{
  return new SymmetricConformation( *this );
}

/// DOF
void
SymmetricConformation::set_dof( DOF_ID const & id, Real const setting )
{
	typedef SymmetryInfo::DOF_IDs DOF_IDs;

	core::Size parent_rsd;

	if ( !symm_info_.dof_is_independent( id, *this ) ) {
		TR.Debug << "SymmetricConformation:: directly setting a dependent DOF!, try to set its parent" << std::endl;
		if (id.type() >= id::RB1 && id.type() <= id::RB6) {
			int parent_jump = symm_info_.jump_follows( fold_tree().get_jump_that_builds_residue( id.rsd() ) );
			parent_rsd = fold_tree().downstream_jump_residue( parent_jump );
		} else {
			parent_rsd = symm_info_.bb_follows( id.rsd() ) ;
		}
	} else {
		parent_rsd = id.rsd();
	}

	id::DOF_ID parent_id ( id::AtomID( id.atomno(), parent_rsd ), id.type() );

	// set this DOF using base-class method
	Conformation::set_dof( parent_id, setting );
	{
		DOF_IDs const & dofs( symm_info_.dependent_dofs( parent_id, *this ) );
		for ( DOF_IDs::const_iterator dof =dofs.begin(), dofe= dofs.end(); dof != dofe; ++dof ) {
			Conformation::set_dof( *dof, setting );
		}
	}


}

/// BONDS/TORSIONS
void
SymmetricConformation::set_torsion( id::TorsionID const & id, Real const setting )
{
	typedef SymmetryInfo::TorsionIDs TorsionIDs;

	TR.Trace << "SymmetricConformation: set_torsion: " << id << ' ' << setting << std::endl;

	core::Size parent_rsd;

	if ( !symm_info_.torsion_is_independent( id ) ) {
		TR.Debug << "[3] SymmetricConformation:: directly setting a dependent TORSION!, try to set its parent" << std::endl;
		parent_rsd = symm_info_.bb_follows( id.rsd() ) ;
	} else {
		parent_rsd = id.rsd();
	}

	id::TorsionID parent_id ( parent_rsd, id.type(), id.torsion() );

	// set this DOF using base-class method
	Conformation::set_torsion( parent_id, setting );
	{
		TorsionIDs const & tors( symm_info_.dependent_torsions( parent_id ) );
		for ( TorsionIDs::const_iterator tor =tors.begin(), tore= tors.end(); tor != tore; ++tor ) {
			Conformation::set_torsion( *tor, setting );
		}
	}
}


	/// JUMPS
void
SymmetricConformation::set_jump( int const jump_number, kinematics::Jump const & new_jump )
{
	typedef SymmetryInfo::AtomIDs AtomIDs;
	typedef SymmetryInfo::Clones Clones;

	 TR.Trace << "SymmetricConformation: set_jump jump_number: " << jump_number << std::endl;

   id::AtomID const id( Conformation::jump_atom_id( jump_number ) );
   Conformation::set_jump( id, new_jump );

 if ( !symm_info_.jump_is_independent( jump_number ) ) {
   TR.Warning << "SymmetricConformation:: directly setting a dependent ATOM!" << std::endl;
 } else {
		for ( Clones::const_iterator pos= symm_info_.jump_clones( jump_number ).begin(),
			epos=symm_info_.jump_clones( jump_number ).end(); pos != epos; ++pos ) {
				id::AtomID const id_clone( Conformation::jump_atom_id( *pos ) );
   			Conformation::set_jump( id_clone, new_jump );
		}
 }
}

void
SymmetricConformation::set_jump_now( int const jump_number, kinematics::Jump const & new_jump )
{
	typedef SymmetryInfo::AtomIDs AtomIDs;
	typedef SymmetryInfo::Clones Clones;

	TR.Trace << "SymmetricConformation: set_jump jump_number: " << jump_number << std::endl;

   id::AtomID const id( Conformation::jump_atom_id( jump_number ) );
   Conformation::set_jump_now( jump_number, new_jump );

 if ( !symm_info_.jump_is_independent( jump_number ) ) {
   TR.Warning << "SymmetricConformation:: directly setting a dependent ATOM!" << std::endl;
 } else {
		for ( Clones::const_iterator pos= symm_info_.jump_clones( jump_number ).begin(),
			epos=symm_info_.jump_clones( jump_number ).end(); pos != epos; ++pos ) {
				id::AtomID const id_clone( Conformation::jump_atom_id( *pos ) );
   			Conformation::set_jump( id_clone, new_jump );
		}
 }
}

// This doesn't work with
void
SymmetricConformation::set_jump( id::AtomID const & id, kinematics::Jump const & new_jump )
{
		 typedef SymmetryInfo::AtomIDs AtomIDs;
		 typedef SymmetryInfo::Clones Clones;

    TR.Trace << "SymmetricConformation: set_jump id:" << id << std::endl;

    Conformation::set_jump( id, new_jump );

	int const jump_number ( fold_tree().get_jump_that_builds_residue( id.rsd() ) );
 	if ( !symm_info_.jump_is_independent( jump_number ) ) {
   TR.Warning << "SymmetricConformation:: directly setting a dependent ATOM!" << std::endl;
 } else {
		for ( Clones::const_iterator pos= symm_info_.jump_clones( jump_number ).begin(),
			epos=symm_info_.jump_clones( jump_number ).end(); pos != epos; ++pos ) {
				id::AtomID const id_clone( Conformation::jump_atom_id( *pos ) );
   			Conformation::set_jump( id_clone, new_jump );
		}
 }
}

///////////////////////////////////////////////////////////////////////////////
/// @details Returns a mask of residues over which scoring is restricted.
/// Only these residues will be used in constructing the neighbor list
utility::vector1<bool>
SymmetricConformation::get_residue_mask() const {
	return symm_info_.independent_residues();
}

core::Real
SymmetricConformation::get_residue_weight(core::Size resid1, core::Size resid2) const {
	return symm_info_.score_multiply(resid1,resid2);
}

/// @details symmetry-safe replace residue
void
SymmetricConformation::replace_residue( Size const seqpos, Residue const & new_rsd_in, bool const orient_backbone ) {
	TR.Trace << "SymmetricConformation: replace_residue: " << seqpos << std::endl;

	core::Size parent_rsd;
	Residue new_rsd = new_rsd_in;

	if ( !symm_info_.bb_is_independent( seqpos ) ) {
		TR.Debug << "[1] SymmetricConformation:: directly setting a dependent TORSION!, try to set its parent" << std::endl;

		parent_rsd = symm_info_.bb_follows( seqpos ) ;

		// if we're not orienting backbone, we need to transform the coord frame to that of the ind. subunit
		if (!orient_backbone) {
			// find transform src->tgt
			//TR.Error << "  MAPPING VRT " << get_upstream_vrt( seqpos ) << " to " << get_upstream_vrt( parent_rsd ) << std::endl;
			Residue const &src_vrt_res = residue( get_upstream_vrt( seqpos ) );
			Residue const &tgt_vrt_res = residue( get_upstream_vrt( parent_rsd ) );

			numeric::xyzMatrix< core::Real > R_ij = numeric::alignVectorSets(
				src_vrt_res.atom(1).xyz() - src_vrt_res.atom(2).xyz(),
				src_vrt_res.atom(3).xyz() - src_vrt_res.atom(2).xyz(),
				tgt_vrt_res.atom(1).xyz() - tgt_vrt_res.atom(2).xyz(),
				tgt_vrt_res.atom(3).xyz() - tgt_vrt_res.atom(2).xyz());
			numeric::xyzVector< core::Real > T_i = -src_vrt_res.atom(2).xyz();
			numeric::xyzVector< core::Real > T_j = tgt_vrt_res.atom(2).xyz();

			// make the new res
			for (int i=1; i<=(int)new_rsd.natoms(); ++i) {
				new_rsd.set_xyz(i , R_ij * (new_rsd_in.xyz(i) + T_i) + T_j );
			}
		}
	} else {
		parent_rsd = seqpos;
	}

	// set this DOF using base-class method
	Conformation::replace_residue( parent_rsd, new_rsd, orient_backbone );

	// now ... if we re _not_ orienting backbone, we update symm copies using
	//    the local coord frame for each
	int parent_vrt=0, clone_vrt=0;
	if (!orient_backbone) {
		parent_vrt = get_upstream_vrt( parent_rsd );
	}

	for ( SymmetryInfo::Clones::const_iterator pos=symm_info_.bb_clones( parent_rsd ).begin(),
	      epos=symm_info_.bb_clones( parent_rsd ).end(); pos != epos; ++pos ) {

		if (!orient_backbone) {
			// find the VRT that builds this res
			clone_vrt = get_upstream_vrt( *pos );

			// find transform parent->clone
			//TR.Error << "  MAPPING VRT " << parent_vrt << " to " << clone_vrt << std::endl;
			Residue const &parent_vrt_res = residue( parent_vrt );
			Residue const &clone_vrt_res = residue( clone_vrt );

			numeric::xyzMatrix< core::Real > R_ij = numeric::alignVectorSets(
				parent_vrt_res.atom(1).xyz() - parent_vrt_res.atom(2).xyz(),
				parent_vrt_res.atom(3).xyz() - parent_vrt_res.atom(2).xyz(),
				clone_vrt_res.atom(1).xyz() - clone_vrt_res.atom(2).xyz(),
				clone_vrt_res.atom(3).xyz() - clone_vrt_res.atom(2).xyz());
			numeric::xyzVector< core::Real > T_i = -parent_vrt_res.atom(2).xyz();
			numeric::xyzVector< core::Real > T_j = clone_vrt_res.atom(2).xyz();

			// make the new res
			Residue new_new_rsd = new_rsd;
			for (int i=1; i<=(int)new_new_rsd.natoms(); ++i) {
				new_new_rsd.set_xyz(i , R_ij * (new_rsd.xyz(i) + T_i) + T_j );
			}
			Conformation::replace_residue( *pos, new_new_rsd, false );
		} else {
			Conformation::replace_residue( *pos, new_rsd, orient_backbone );
		}
	}
}

/// @details symmetry-safe replace residue
void
SymmetricConformation::replace_residue( Size const seqpos,
                 Residue const & new_rsd,
                 utility::vector1< std::pair< std::string, std::string > > const & atom_pairs )  {
	TR.Trace << "SymmetricConformation: replace_residue: " << seqpos << std::endl;

	core::Size parent_rsd;

	if ( !symm_info_.bb_is_independent( seqpos ) ) {
		TR.Debug << "[2] SymmetricConformation:: directly setting a dependent TORSION!, try to set its parent" << std::endl;
		parent_rsd = symm_info_.bb_follows( seqpos ) ;
	} else {
		parent_rsd = seqpos;
	}

	// set this DOF using base-class method
	Conformation::replace_residue( parent_rsd, new_rsd, atom_pairs );
	for ( SymmetryInfo::Clones::const_iterator pos=symm_info_.bb_clones( parent_rsd ).begin(),
	      epos=symm_info_.bb_clones( parent_rsd ).end(); pos != epos; ++pos ) {
		Conformation::replace_residue( *pos, new_rsd, atom_pairs );
	}
}


core::Size
SymmetricConformation::get_upstream_vrt( Size seqpos ) {
	if (this->residue( seqpos ).aa() == core::chemical::aa_vrt)
		return seqpos;

	// find peptide segment that contains this res (?)
	core::kinematics::Edge const &e = fold_tree().get_residue_edge( seqpos );
	core::Size curr_res = e.start();

	while ( this->residue( curr_res ).aa() != core::chemical::aa_vrt ) {
		core::kinematics::Edge const &e_i = fold_tree().get_residue_edge( curr_res );
		curr_res = e_i.start();
	}

	return curr_res;
}


/*
	/// XYZ
	virtual
	void
	set_xyz( AtomID const & id, PointPosition const & position );


	/// TRANSFORMS
	virtual
	void
	set_stub_transform( id::StubID const & stub_id1, id::StubID const & stub_id2, kinematics::RT const & target_rt );
**/

}
} // conformation
} // core


