// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   core/conformation/symmetry/SymmetricConformation.hh
/// @brief  symmetry conformation container.
//					Contains overloaded functions needed to
//					make changes in conformation symmetric
/// @author Phil Bradley, Ingemar Andre


#ifndef INCLUDED_core_conformation_symmetry_SymmetricConformation_HH
#define INCLUDED_core_conformation_symmetry_SymmetricConformation_HH


// Unit headers
#include <core/conformation/symmetry/SymmetricConformation.fwd.hh>

#include <core/conformation/Conformation.hh>
#include <core/conformation/symmetry/SymmetryInfo.hh>

namespace core {
namespace conformation {
namespace symmetry {

/// @brief  A symmetric conformation: has an additional data member "SymmetryInfo" class
/// @details  Handles symmetrizing of *some* of the conformation-changing methods of Conformation

class SymmetricConformation : public Conformation {

public:

	/////////////////////////////////////////
	//// Construction / Copying
	/////////////////////////////////////////

	/// @brief  Default CTOR
	SymmetricConformation():
		Conformation(),
		symm_info_()
	{}

	/// @brief  Default CTOR
	SymmetricConformation( Conformation const & conf, SymmetryInfo const & symm_info ):
		Conformation( conf ),
		symm_info_( symm_info )
	{}

	/// @brief copy constructor
	SymmetricConformation( SymmetricConformation const & src );

	/// @brief operator
	Conformation &
	operator=( SymmetricConformation const & src );

  ConformationOP
  clone() const;

  SymmetryInfo const &
	Symmetry_Info() const
  {
    return symm_info_;
  };

  SymmetryInfo &
	Symmetry_Info()
  {
    return symm_info_;
  };

	/////////////////////////////////////////
	//// Setters
	/////////////////////////////////////////

	/// DOF
	virtual
	void
	set_dof( DOF_ID const & id, Real const setting );

	/// BONDS/TORSIONS
	virtual
	void
	set_torsion( TorsionID const & id, Real const setting );

	/// JUMPS
	/// @brief set a jump
  virtual
  void
  set_jump(
    int const jump_number,
    Jump const & new_jump
  );

  /// @brief set a jump and force immediate calculation of affected XYZ coords
  virtual
  void
  set_jump_now(
    int const jump_number,
    Jump const & new_jump
  );

	/// @brief set a jump
  virtual
  void
  set_jump(
    AtomID const & id,
    Jump const & new_jump
  );

	virtual utility::vector1<bool>
	get_residue_mask() const;

	virtual Real
	get_residue_weight(core::Size resid1, core::Size resid2) const;

	/// @brief replace residue
	virtual void
	replace_residue(
		Size const seqpos,
		Residue const & new_rsd,
		bool const orient_backbone
	);

	virtual void
	replace_residue(
		Size const seqpos,
		Residue const & new_rsd,
		utility::vector1< std::pair< std::string, std::string > > const & atom_pairs
	);

	/**
	virtual
	void
	set_torsion_angle( AtomID const & atom1, AtomID const & atom2, AtomID const & atom3, AtomID const & atom4,
										 Real const setting 	);

	virtual
	void
	set_bond_angle( AtomID const & atom1, AtomID const & atom2, AtomID const & atom3, Real const setting );

	virtual
	void
	set_bond_length( AtomID const & atom1, AtomID const & atom2, Real const setting );

	/// JUMPS
	virtual
	void
	set_jump( int const jump_number, Jump const & new_jump );

	virtual
	void
	set_jump_now( int const jump_number, Jump const & new_jump );

	virtual
	void
	set_jump( AtomID const & id, Jump const & new_jump );

	/// XYZ
	virtual
	void
	set_xyz( AtomID const & id, PointPosition const & position );


	/// TRANSFORMS
	virtual
	void
	set_stub_transform( id::StubID const & stub_id1, id::StubID const & stub_id2, kinematics::RT const & target_rt );
	**/

	/////////////////////////////////////////////////////////////////////////////
	///
	virtual
	~SymmetricConformation() { clear(); }


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// private methods
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////

private:

	// utility function gets the nearest upstream virtual
	core::Size
	get_upstream_vrt( core::Size seqpos );



private:
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// data
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////

	SymmetryInfo symm_info_;

};

} // symmetry
} // conformation
} // core



#endif
