// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file
/// @brief
/// @author Ingemar Andre, Phil Bradley

// Unit Headers
#include <core/conformation/symmetry/SymmetryInfo.hh>
#include <core/id/types.hh>
#include <core/id/DOF_ID.hh>
#include <core/id/TorsionID.hh>
#include <core/id/AtomID.hh>

#include <core/conformation/Conformation.hh>

// Utility headers
#include <utility/exit.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// C++ headers
#include <cassert>
#include <iostream>

// core utilities
#include <core/util/Tracer.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("core.conformation.SymmetryInfo");

namespace core {
namespace conformation {
namespace symmetry {

/// @details  This is a helper function for some of the DOF_ID routines below. Really just a best guess...
/// this is a little tricky: the mapping from a DOF_ID to a TorsionID is not straightforward to
/// construct (see kinematics/util.cc:setup_dof_to_torsion_map)
/// So we don't really know whether a dof_id is a bb degree of freedom or a chi degree of freedom...
/// or even which residue it should be attached to, eg the phi of residue i might be dof_id with rsd i-1
/// So we take a guess based on whether id.atomno is a backbone or sidechain atom

id::TorsionType
guess_torsion_type_of_dof_id( id::DOF_ID const & id, Conformation const & conf )
{
	if ( id::RB1 <= id.type() && id.type() <= id::RB6 ) {
		return id::JUMP;
	} else {
//		Residue const & rsd( conf.residue( id.rsd() ) );
		if ( conf.atom_is_backbone_norefold( id.rsd(), id.atomno() ) ) {
			return id::BB;
		} else {
			return id::CHI;
		}
	}
}


SymmetryInfo::SymmetryInfo() { use_symmetry_ = false; score_multiply_factor_ = 1; }
SymmetryInfo::~SymmetryInfo() {}


bool SymmetryInfo::operator==( SymmetryInfo const & s )
{
	return ( npseudo_ == s.npseudo_ &&
           bb_clones_ ==s.bb_clones_ &&
           chi_clones_ == s.chi_clones_ &&
           jump_clones_ == s.jump_clones_ );
}

bool SymmetryInfo::operator!=( SymmetryInfo const & s )
{
	return !( *this == s );
}

SymmetryInfo::SymmetryInfo( SymmData const & symm_data )
{
	if (  symm_data.get_jump_clones().size() > 0 ) {
		initialize( symm_data.get_nres_subunit(), symm_data.get_njump_subunit(),
              symm_data.get_subunits(), symm_data.get_num_virtual(),
							symm_data.get_jump_clones(), symm_data.get_dofs(),
							symm_data.get_score_subunit(), symm_data.get_score_multiply(),
							symm_data.get_interfaces() );
	} else {
		initialize( symm_data.get_nres_subunit(), symm_data.get_njump_subunit(),
							symm_data.get_subunits(), symm_data.get_num_virtual(),
							symm_data.get_dofs(),symm_data.get_score_subunit(),
							symm_data.get_score_multiply(), symm_data.get_interfaces() );
	}
	TR<<*this<<std::endl;
}

SymmetryInfo::SymmetryInfo(
                           Size const nres_monomer,
                           Size const njump_monomer,
                           Size const N,
                           std::map< Size, SymDof > dofs,
													 Size const score_subunit,
                           utility::vector1< Size > score_multiply,
                           Size const num_interfaces,
                           std::string const & type
                           )
{
	initialize( nres_monomer, njump_monomer, N, N, dofs, score_subunit,
							score_multiply, num_interfaces, type );
}

// This is an old style constructor. Should change soon...
void
SymmetryInfo::initialize(
													 Size const nres_monomer,
													 Size const njump_monomer,
													 Size const N,
													 Size const num_virtual,
													 std::map< Size, SymDof > dofs,
													 Size const score_subunit,
													 utility::vector1< Size > score_multiply,
													 Size const num_interfaces,
													 std::string const & type
													 )
{

	// set number of interfaces
	interfaces_ = num_interfaces;
	// store the score multiplication factors
	score_multiply_ = score_multiply;
	// store the allowed dofs
	dofs_ = dofs;
	// set use symmetry
	use_symmetry_ = true;
	// set cp_weighting_during_minimization_ to false
	cp_weighting_during_minimization_ = false;
	// store type
	type_ = type;
	// setup bb,chi clones
	bb_clones_.clear();
	chi_clones_.clear();
	jump_clones_.clear();

	//check that score_monomer makes sense...
	if ( score_subunit > N || score_subunit < 1 ) {
      utility_exit_with_message("score_subunit must be in the range 1-N");
   }

	//special case of no symmetry
	if ( type == "c1" ) {
		npseudo_ = num_virtual;

		//we need to map to an empty array in order for
		for ( Size i=1; i<= nres_monomer; ++i ) {
			Clones clones;
			clones.clear();
			bb_clones_.insert( std::make_pair( i, clones ) );
			chi_clones_.insert( std::make_pair( i, clones ) );
		}
		for ( Size i=1; i<= njump_monomer; ++i ) {
			Clones clones;
			clones.clear();
			jump_clones_.insert( std::make_pair( i, clones ) );
		}
		return;
	}//end c1 symmetry

	for ( Size i=1; i<= nres_monomer; ++i ) {
		Clones clones;
      int base ( i + ( score_subunit - 1 ) * nres_monomer );
      for ( Size k=0; k<N; ++k ) {
        if ( k+1 != score_subunit ) {
          clones.push_back( i + k * nres_monomer );
					add_bb_clone( base, i + k * nres_monomer );
		      add_chi_clone( base, i + k * nres_monomer );
        }
      }
      bb_clones_.insert( std::make_pair( base, clones ) );
      chi_clones_.insert( std::make_pair( base, clones ) );
    }

	// the N*njump_monomer internal jumps
	for ( Size i=1; i<= njump_monomer; ++i ) {
		//Clones clones;
		for ( Size k=1; k<N; ++k ) {
			//clones.push_back( i + k * njump_monomer );
			add_jump_clone( i, i + k * njump_monomer, 0.0 );
		}
		//jump_clones_.insert( std::make_pair( i, clones ) );
	}

	if ( type == "no_pseudo" ) {
		npseudo_ = num_virtual;

	} else if ( type == "simple" ) {
		// 1                 --> N*njump_monomer  : the internal jumps
		// N*njump_monomer+1 --> N*njump_monomer+N: the pseudo-rsd--monomer jumps
		// last N-1 jumps                         : jumps between pseudo-rsds

		npseudo_ = num_virtual;

		// the N jumps from pseudo-residues to monomers
		{
			Size const base_jump( N*njump_monomer + 1 );
			//Clones clones;
			for ( Size k=1; k<N; ++k ) {
				//clones.push_back( base_jump + k );
				add_jump_clone( base_jump, base_jump + k, 0.0 );
			}
			//jump_clones_.insert( std::make_pair( base_jump, clones ) );
		}


		// the N-1 jumps between pseudo-residues
		{
			Size const base_jump( N*njump_monomer + N + 1 );
			//Clones clones;
			for ( Size k=1; k<N-1; ++k ) {
				//clones.push_back( base_jump + k );
				add_jump_clone( base_jump, base_jump + k, 0.0 );
			}
			//jump_clones_.insert( std::make_pair( base_jump, clones ) );
		}
	} else {
		std::cerr << "unrecognized type: " << type << std::endl;
		utility_exit();
	}

	update_score_multiply_factor();
}


//
void
SymmetryInfo::update_score_multiply_factor()
{
	// compute the score_multiply_factor
	utility::vector1< bool > indep_res = independent_residues();
	for (int i=1; i<=(int)indep_res.size(); ++i) {
		if (indep_res[i]) {
			score_multiply_factor_ = score_multiply_[ i ];
			//std::cout<< "score_multiply_factor "<< score_multiply_factor_ <<std::endl;
			break;
		}
	}
}

// This is an old style constructor. Should change soon...
void
SymmetryInfo::initialize(
													 Size const nres_monomer,
													 Size const njump_monomer,
													 Size const N,
													 Size const num_virtual,
													 std::map< Size, WtedClones > jump_clones,
													 std::map< Size, SymDof > dofs,
													 Size const score_subunit,
													 utility::vector1< Size > score_multiply,
													 Size const num_interfaces,
													 std::string const & type
													 )
{

	// set number of interfaces
	interfaces_ = num_interfaces;
	// store the score multiplication factors
	score_multiply_ = score_multiply;
	// store the allowed dofs
	dofs_ = dofs;
	// set use symmetry
	use_symmetry_ = true;
	// set cp_weighting_during_minimization_ to false
	cp_weighting_during_minimization_ = false;
	// store type
	type_ = type;
	// setup bb,chi clones
	bb_clones_.clear();
	chi_clones_.clear();
	jump_clones_.clear();

		// 1                 --> N*njump_monomer  : the internal jumps
		// N*njump_monomer+1 --> N*njump_monomer+N: the pseudo-rsd--monomer jumps
		// last N-1 jumps                         : jumps between pseudo-rsds

		npseudo_ = num_virtual;

	for ( Size i=1; i<= nres_monomer; ++i ) {
		Clones clones;
      int base ( i + ( score_subunit - 1 ) * nres_monomer );
      for ( Size k=0; k<N; ++k ) {
        if ( k+1 != score_subunit ) {
          clones.push_back( i + k * nres_monomer );
					add_bb_clone( base, i + k * nres_monomer );
		      add_chi_clone( base, i + k * nres_monomer );
        }
      }
      bb_clones_.insert( std::make_pair( base, clones ) );
      chi_clones_.insert( std::make_pair( base, clones ) );
	}

	// the N*njump_monomer internal jumps
	for ( Size i=1; i<= njump_monomer; ++i ) {
		//Clones clones;
		for ( Size k=1; k<N; ++k ) {
			//clones.push_back( i + k * njump_monomer );
			add_jump_clone( i, i + k * njump_monomer, 0.0 );
		}
		//jump_clones_.insert( std::make_pair( i, clones ) );
	}

	std::map< Size,WtedClones >::const_iterator it, it_start=jump_clones.begin(), it_end=jump_clones.end();
	for ( it=it_start; it != it_end; ++it ) {
		//Clones clones;
		for ( Size i = 1; i<= it->second.size(); ++i ) {
			//clones.push_back( it->second[i] + N*njump_monomer );
			add_jump_clone( it->first + N*njump_monomer, it->second[i].first + N*njump_monomer, it->second[i].second );
		}
		//jump_clones_.insert( std::make_pair( it->first + N*njump_monomer, clones ) );
	}

	// compute the score_multiply_factor
	update_score_multiply_factor();
}

	/////////////////////////////////////////////////////////////////////////////
	void
	comma_strings_to_vector_map(
		std::istream & is,
		Size const nbase,
		std::map< Size, utility::vector1< Size > > & clones,
		std::string tag=""
	)
	{
		bool fail( false );
		std::string tag0;
		if( tag != "" ) {
			is >> tag0;
			if( tag0 != tag ) {
				std::cout << "Input failed: tag mismatch " << tag << " " << tag0 << std::endl;
				return;
			}
		}
		for ( Size i=1; !fail && i<= nbase; ++i ) {
			std::string jump_string;
			is >> jump_string;
			if ( is.fail() ) {
				fail = true;
				break;
			}
			std::replace( jump_string.begin(), jump_string.end(), ',', ' ' );
			std::istringstream l( jump_string );
			Size base_jump;
			l >> base_jump;
			if ( l.fail() ) {
				fail = true;
				break;
			}
			while ( true ) {
				Size j;
				l >> j;
				if ( l.fail() ) break;
				clones[ base_jump ].push_back( j );
			}
			if ( clones[ base_jump ].size() < 1 ) {
				fail = true;
				break;
			}
		}
		if ( fail ) {
			is.setstate( std::ios_base::failbit );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	comma_strings_to_map(
		std::istream & is,
		Size const nbase,
		std::map< Size, Size > & clones,
		std::string tag=""
	)
	{
		bool fail( false );
		std::string tag0;
		if( tag != "" ) {
			is >> tag0;
			if( tag0 != tag ) {
				std::cout << "Input failed: tag mismatch " << tag << " " << tag0 << std::endl;
				return;
			}
		}
		for ( Size i=1; !fail && i<= nbase; ++i ) {
			std::string jump_string;
			is >> jump_string;
			if ( is.fail() ) {
				fail = true;
				break;
			}
			std::replace( jump_string.begin(), jump_string.end(), ',', ' ' );
			std::istringstream l( jump_string );
			Size base_jump;
			l >> base_jump;
			if ( l.fail() ) {
				fail = true;
				break;
			}
			l >> clones[ base_jump ];
		}
		if ( fail ) {
			is.setstate( std::ios_base::failbit );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	comma_strings_to_map(
		std::istream & is,
		Size const nbase,
		std::map< Size, SymDof > & clones,
		std::string tag=""
	)
	{
		bool fail( false );
		std::string tag0;
		if( tag != "" ) {
			is >> tag0;
			if( tag0 != tag ) {
				std::cout << "Input failed: tag mismatch " << tag << " " << tag0 << std::endl;
				return;
			}
		}
		for ( Size i=1; !fail && i<= nbase; ++i ) {
			std::string jump_string;
			is >> jump_string;
			if ( is.fail() ) {
				fail = true;
				break;
			}
			std::replace( jump_string.begin(), jump_string.end(), ',', ' ' );
			std::istringstream l( jump_string );
			Size base_jump;
			l >> base_jump;
			std::string dof_line;
			l >> dof_line;
			clones[base_jump].read(dof_line);
			if ( l.fail() ) {
				fail = true;
				break;
			}
		}
		if ( fail ) {
			is.setstate( std::ios_base::failbit );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	comma_strings_to_vector(
		std::istream & is,
		Size const nbase,
		utility::vector1< Size > & clones,
		std::string tag=""
	)
	{
		bool fail( false );
		std::string tag0;
		if( tag != "" ) {
			is >> tag0;
			if( tag0 != tag ) {
				std::cout << "Input failed: tag mismatch " << tag << " " << tag0 << std::endl;
				return;
			}
		}

		std::string jump_string;
		is >> jump_string;
		if ( is.fail() ) fail = true;
		std::replace( jump_string.begin(), jump_string.end(), ',', ' ' );
		std::istringstream l( jump_string );
		while ( true ) {
			Size j;
			l >> j;
			if ( l.fail() ) break;
			clones.push_back( j );
		}
		if ( clones.size() != nbase ) {
			fail = true;
		}

		if ( fail ) {
			is.setstate( std::ios_base::failbit );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	vector_map_to_comma_strings(
		std::ostream & out,
		std::map< Size, utility::vector1< Size > > clones,
		std::string tag=""
	)
	{
		if( tag != "" ) out << ' ' << tag ;
		for ( std::map< Size ,utility::vector1< Size > >::const_iterator
						it = clones.begin(); it != clones.end(); ++it ) {
			out << ' ' << it->first;
			utility::vector1< Size > const & l( it->second );
			for ( Size i=1; i<= l.size(); ++i ) {
				out << ',' << l[i];
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	map_to_comma_strings(
		std::ostream & out,
		std::map< Size, Size > clones,
		std::string tag=""
	)
	{
		if( tag != "" ) out << ' ' << tag ;
		for ( std::map< Size , Size >::const_iterator
						it = clones.begin(); it != clones.end(); ++it ) {
			out << ' ' << it->first << ',' << it->second ;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	map_to_comma_strings(
		std::ostream & out,
		std::map< Size, SymDof > clones,
		std::string tag=""
	)
	{
		if( tag != "" ) out << ' ' << tag ;
		for ( std::map< Size , SymDof >::const_iterator
						it = clones.begin(); it != clones.end(); ++it ) {
			//Dof const & dof (it->second);
			out << " " << it->first << "," << it->second ;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	vector_to_comma_strings(
		std::ostream & out,
		utility::vector1 < Size > clones,
		std::string tag=""
	)
	{
		if( tag != "" ) out << ' ' << tag ;
		for ( Size i=1; i<= clones.size(); ++i ) {
			if( i == 1 ) {
				out << " " << clones[i] ;
			} else {
				out << ',' << clones[i] ;
			}
		}

	}

	/////////////////////////////////////////////////////////////////////////////
std::istream& operator>> ( std::istream & s, SymmetryInfo & symminfo )
{
		bool fail( false );

		std::string tag;
		Size num_bb_indep, num_chi_indep, num_jump_indep;
		Size num_bb_dep, num_chi_dep, num_jump_dep;
		Size num_dof, num_score_multiply;

		symminfo.set_use_symmetry(true);
		symminfo.set_cp_weighting_during_minimization(false);

		s >> tag ;
		if ( tag != "SYMMETRY_INFO" || s.fail() ) {
			fail = true;
		} else {
      s >> tag >> tag >> tag >> symminfo.npseudo_ >> tag >> symminfo.interfaces_ >> tag >> symminfo.type_
				>> tag >> num_bb_indep >> tag >> num_chi_indep >> tag >> num_jump_indep
				>> tag >> num_bb_dep >> tag >> num_chi_dep >> tag >> num_jump_dep
				>> tag >> num_dof >> tag >> num_score_multiply ;

			if ( s.fail() ) fail = true;
			//std::cout<< num_bb_indep << " " << num_chi_indep << " " << num_jump_indep
			//				 << " " << num_bb_dep << " " << num_chi_dep << " " << num_jump_dep
			//				 << " " << num_dof << " " << num_score_multiply << std::endl;


			// clones
			comma_strings_to_vector_map( s,   num_bb_indep,  symminfo.bb_clones_, "BB_CLONES" );
			comma_strings_to_vector_map( s,  num_chi_indep,  symminfo.chi_clones_, "CHI_CLONES" );
			comma_strings_to_vector_map( s, num_jump_indep,  symminfo.jump_clones_, "JUMP_CLONES" );

			// follows
			comma_strings_to_map( s,   num_bb_dep,  symminfo.bb_follows_, "BB_FOLLOWS" );
			comma_strings_to_map( s,  num_chi_dep,  symminfo.chi_follows_, "CHI_FOLLOWS" );
			comma_strings_to_map( s, num_jump_dep,  symminfo.jump_follows_, "JUMP_FOLLOWS" );

			//dof_
			comma_strings_to_map( s, num_dof,  symminfo.dofs_, "DOFS" );

			//score_multiply_
			comma_strings_to_vector( s, num_score_multiply,  symminfo.score_multiply_, "SCORE_MULTIPLY" );
			symminfo.update_score_multiply_factor();

			if ( fail ) {
				std::cout << "Symmetry_info operator>>: Input failed" << std::endl;
				s.setstate( std::ios_base::failbit );
				return s;
			}
		}

		return s;
}

	/////////////////////////////////////////////////////////////////////////////
std::ostream& operator<< ( std::ostream & s, const SymmetryInfo & symminfo )
{
		s << "SYMMETRY_INFO " <<
			"N " << symminfo.subunits() << ' ' <<
      "N_VIRT " << symminfo.npseudo_ << ' ' <<
			"N_INTERFACE " << symminfo.num_interfaces() << ' ' <<
			"TYPE " << symminfo.type_ << ' ' <<
			"BB_CLONES_SIZE " << symminfo.bb_clones_.size() << ' ' <<
			"CHI_CLONES_SIZE " << symminfo.chi_clones_.size() << ' ' <<
			"JUMP_CLONES_SIZE " << symminfo.jump_clones_.size() << ' ' <<
			"BB_FOLLOWS_SIZE " << symminfo.bb_follows_.size() << ' ' <<
			"CHI_FOLLOWS_SIZE " << symminfo.chi_follows_.size() << ' ' <<
			"JUMP_FOLLOWS_SIZE " << symminfo.jump_follows_.size() << ' ' <<
			"DOFS_SIZE " << symminfo.dofs_.size() << ' ' <<
			"SCORE_MULTIPLY_SIZE " << symminfo.score_multiply_.size() ;

		// clones
		vector_map_to_comma_strings( s, symminfo.bb_clones_, "BB_CLONES" );
		vector_map_to_comma_strings( s, symminfo.chi_clones_, "CHI_CLONES" );
		vector_map_to_comma_strings( s, symminfo.jump_clones_, "JUMP_CLONES" );

		// follows
		map_to_comma_strings( s, symminfo.bb_follows_, "BB_FOLLOWS" );
		map_to_comma_strings( s, symminfo.chi_follows_, "CHI_FOLLOWS" );
		map_to_comma_strings( s, symminfo.jump_follows_, "JUMP_FOLLOWS" );

		//dof
		map_to_comma_strings( s, symminfo.dofs_, "DOFS" );

		//score_multiply_
		vector_to_comma_strings( s, symminfo.score_multiply_, "SCORE_MULTIPLY" );

		return s;
}

bool
SymmetryInfo::write_silent_struct(
std::string const & filename
)
{
   bool success = false;

   utility::io::ozstream output;
   if ( !utility::file::file_exists( filename ) ) {
      output.open( filename );
   } else {
      output.open_append( filename );
   }

   output << *this << '\n';

   output.close();

   success = true;
   return success;
}


bool
SymmetryInfo::read_silent_struct(
std::string const & filename
)
{
   bool success = false;

   utility::io::izstream input ( filename.c_str() );
   std::istringstream line_stream;
   std::string line("");
   if ( !input ) {
      std::cerr << "ERROR:: Unable to open symmetry info file: "
            << filename << std::endl;
      return success;
   }

   while( !input.eof() ) {
      getline(input,line);
      line_stream.clear();
      line_stream.str(line);
      line_stream >> *this;
	 }

   input.close();

   success = true;
   return success;
}

Size
SymmetryInfo::bb_follows( Size const seqpos ) const
{
	std::map< Size, Size >::const_iterator it( bb_follows_.find( seqpos ) );
	return ( it == bb_follows_.end() ? 0 : it->second );
}

Size
SymmetryInfo::chi_follows( Size const seqpos ) const
{
	std::map< Size, Size >::const_iterator it( chi_follows_.find( seqpos ) );
	return ( it == chi_follows_.end() ? 0 : it->second );
}

Size
SymmetryInfo::jump_follows( Size const seqpos ) const
{
	std::map< Size, Size >::const_iterator it( jump_follows_.find( seqpos ) );
	return ( it == jump_follows_.end() ? 0 : it->second );
}

std::vector < std::pair < Size, Size > >
SymmetryInfo::map_symmetric_res_pairs( Size res1, Size res2 )
{
	std::vector < std::pair < Size, Size > > map;
	int delta ( res2 - res1 );
	int mapped_res;
	for ( std::vector< Size>::const_iterator
            clone     = bb_clones( res1 ).begin(),
            clone_end = bb_clones( res1 ).end();
            clone != clone_end; ++clone ){
							if ( *clone + delta > num_total_residues() ) {
								mapped_res = (*clone + delta)%num_total_residues();
							} else {
								mapped_res = *clone + delta;
							}
							if ( mapped_res < 0 ) mapped_res += num_independent_residues();
							map.push_back( std::make_pair( *clone, mapped_res ) );
	}
	return map;
}

bool
SymmetryInfo::bb_is_independent( Size const seqpos ) const
{
	return bb_follows(seqpos) == 0;
}

bool
SymmetryInfo::chi_is_independent( Size const seqpos ) const
{
  return chi_follows(seqpos) == 0;
}

bool
SymmetryInfo::fa_is_independent( Size const seqpos ) const
{
	return ( bb_is_independent(seqpos) && chi_is_independent(seqpos) );
}

bool
SymmetryInfo::jump_is_independent( Size const seqpos ) const
{
  return jump_follows(seqpos) == 0;
}

Size
SymmetryInfo::subunits() const
{
	return num_bb_clones() + 1;
}

utility::vector1< bool >
SymmetryInfo::independent_residues() const
{
    utility::vector1 < bool > residues;
    for ( Size i=1; i <=num_total_residues_with_pseudo(); ++i ){
      if ( bb_is_independent(i) ) residues.push_back(true);
      else residues.push_back(false);
    }
  return residues;
}

Size
SymmetryInfo::num_bb_clones() const
{
	// all these lists have the same size
	return bb_clones_.begin()->second.size();
}

Size
SymmetryInfo::num_chi_clones() const
{
  // all these lists have the same size
  return chi_clones_.begin()->second.size();
}


Size
SymmetryInfo::num_jump_clones() const
{
  // all these lists have the same size
  return jump_clones_.begin()->second.size();
}

Size
SymmetryInfo::num_independent_residues() const
{
	return bb_clones_.size();
}

Size
SymmetryInfo::num_total_residues() const
{
	return num_independent_residues()*( num_bb_clones() + 1 );
}

Size
SymmetryInfo::num_total_residues_with_pseudo() const
{
	return num_independent_residues()*( num_bb_clones() + 1 ) + npseudo_;
}

Size
SymmetryInfo::num_total_residues_without_pseudo() const
{
  return num_independent_residues()*( num_bb_clones() + 1 );
}

Size
SymmetryInfo::num_interfaces() const
{
	return interfaces_;
}

Size
SymmetryInfo::score_multiply_factor() const
{
	return score_multiply_factor_;
}

// bool
// SymmetryInfo::scoring_residue(Size residue ) const
// {
// 	if ( score_multiply( residue ) == 0 ) return false;
// 	return true;
// }

SymmetryInfo::Clones const &
SymmetryInfo::bb_clones( Size const seqpos ) const
{
	std::map< Size, Clones >::const_iterator it( bb_clones_.find( seqpos ) );
  if ( it == bb_clones_.end() ) {
  	return empty_list;
  }
  return it->second;
}

SymmetryInfo::Clones const &
SymmetryInfo::chi_clones( Size const seqpos ) const
{
  std::map< Size, Clones >::const_iterator it( chi_clones_.find( seqpos ) );
  if ( it == chi_clones_.end() ) {
    return empty_list;
  }
  return it->second;
}

SymmetryInfo::Clones const &
SymmetryInfo::jump_clones( Size const seqpos ) const
{
  std::map< Size, Clones >::const_iterator it( jump_clones_.find( seqpos ) );
  if ( it == jump_clones_.end() ) {
    return empty_list;
  }
  return it->second;
}

void
SymmetryInfo::add_bb_clone( Size const base_pos, Size const clone_pos )
{
	if ( bb_follows_[ base_pos ] != 0 ) {
		std::cerr << "Error: add_bb_clone: base_pos is not independent: " <<
			base_pos << ' ' << bb_follows_[ base_pos ] << std::endl;
		utility_exit();
	}
	if ( bb_follows_[ clone_pos ] != 0 &&
			 bb_follows_[ clone_pos ] != base_pos ) {
		std::cerr << "Error: add_bb_clone: clone_pos already a follower: " <<
			clone_pos << ' ' << bb_follows_[ clone_pos ] << ' ' << base_pos <<
			std::endl;
		utility_exit();
	}

	bb_follows_[ clone_pos ] = base_pos;
	bb_clones_[ base_pos ].push_back( clone_pos );
}

void
SymmetryInfo::add_chi_clone( Size const base_pos, Size const clone_pos )
{
	if ( chi_follows_[ base_pos ] != 0 ) {
		std::cerr << "Error: add_chi_clone: base_pos is not independent: " <<
			base_pos << ' ' << chi_follows_[ base_pos ] << std::endl;
		utility_exit();
	}
	if ( chi_follows_[ clone_pos ] != 0 &&
			 chi_follows_[ clone_pos ] != base_pos ) {
		std::cerr << "Error: add_chi_clone: clone_pos already a follower: " <<
			clone_pos << ' ' << chi_follows_[ clone_pos ] << ' ' << base_pos <<
			std::endl;
		utility_exit();
	}

	chi_follows_[ clone_pos ] = base_pos;
	chi_clones_[ base_pos ].push_back( clone_pos );
}

void
SymmetryInfo::add_jump_clone( Size const base_pos, Size const clone_pos, Real const jump_wt )
{
	if ( jump_follows_[ base_pos ] != 0 ) {
		std::cerr << "Error: add_jump_clone: base_pos is not independent: " <<
			base_pos << ' ' << bb_follows_[ base_pos ] << std::endl;
		utility_exit();
	}
	if ( jump_follows_[ clone_pos ] != 0 &&
			 jump_follows_[ clone_pos ] != base_pos ) {
		std::cerr << "Error: add_jump_clone: clone_pos already a follower: " <<
			clone_pos << ' ' << jump_follows_[ clone_pos ] << ' ' << base_pos <<
			std::endl;
		utility_exit();
	}

	jump_follows_[ clone_pos ] = base_pos;
	jump_clones_[ base_pos ].push_back( clone_pos );
	jump_clone_wts_[ clone_pos ] = jump_wt;
}

bool
SymmetryInfo::get_use_symmetry() const
{
	return use_symmetry_;
}

bool
SymmetryInfo::cp_weighting_during_minimization() const
{
	return cp_weighting_during_minimization_;
}

void
SymmetryInfo::set_cp_weighting_during_minimization( bool setting )
{
  cp_weighting_during_minimization_ = setting;
}

void
SymmetryInfo::set_use_symmetry( bool setting )
{
	use_symmetry_ = setting;
}

/// @details  Returns set of DOF_IDs that follow a given one. Inefficient: it creates a list each time
SymmetryInfo::DOF_IDs
SymmetryInfo::dependent_dofs( DOF_ID const & id, Conformation const & conf ) const
{
	if ( !dof_is_independent( id, conf ) ) {
		utility_exit_with_message( "SymmetryInfo::dependent_dofs: dof is not independent!" );
	}

	Size const seqpos( id.rsd() );
	Size const atomno( id.atomno() );
	id::TorsionType const type( guess_torsion_type_of_dof_id( id, conf ) );
	Clones const & clones( type == id::JUMP ? jump_clones( conf.fold_tree().get_jump_that_builds_residue( id.rsd() ) ):
												 ( type == id::BB ? bb_clones( seqpos ) : chi_clones( seqpos ) ) );

	DOF_IDs dofs;
	for ( Clones::const_iterator pos= clones.begin(), epos=clones.end(); pos != epos; ++pos ) {
		if ( type == id::JUMP ) {
			dofs.push_back( DOF_ID( id::AtomID( atomno, conf.fold_tree().downstream_jump_residue( *pos )  ), id.type() ) );
		} else {
			dofs.push_back( DOF_ID( id::AtomID( atomno, *pos ), id.type() ) );
		}
	}
	return dofs;
}

///
bool
SymmetryInfo::dof_is_independent( DOF_ID const & id, Conformation const & conf ) const
{
	id::TorsionType const type( guess_torsion_type_of_dof_id( id, conf ) );

	switch ( type ) {
	case id::BB:
		return bb_is_independent( id.rsd() );
	case id::CHI:
		return chi_is_independent( id.rsd() );
	case id::JUMP:
		return jump_is_independent( conf.fold_tree().get_jump_that_builds_residue( id.rsd() ) );
	}

	utility_exit_with_message("dof_is_independent: unrecognized TorsionType!");
	return false;
}

// get a weight for derivative calculations
// weights are 1 for indep DOFs, 0 for dependent NON-JUMP DOFs
//    and may be any real for dependent jump dofs
core::Real
SymmetryInfo::get_dof_derivative_weight( DOF_ID const & id, Conformation const & conf ) const {
	id::TorsionType const type( guess_torsion_type_of_dof_id( id, conf ) );

	if ( type == id::BB ) {
		return bb_is_independent( id.rsd() ) ? 1. : 0.;
	} else if ( type == id::CHI ) {
		return chi_is_independent( id.rsd() ) ? 1. : 0.;
	} else if ( type == id::JUMP ) {
		int jumpnum = conf.fold_tree().get_jump_that_builds_residue( id.rsd() );
		std::map< Size, Real >::const_iterator it( jump_clone_wts_.find( jumpnum ) );
		return ( it == jump_clone_wts_.end() ? 1. : it->second );
	}

	utility_exit_with_message("get_dof_derivative_weight: unrecognized TorsionType!");
	return false;
}



///
bool
SymmetryInfo::torsion_is_independent( TorsionID const & id ) const
{
	return ( ( id.type() == id::BB   &&   bb_is_independent( id.rsd() ) ) ||
					 ( id.type() == id::CHI  &&  chi_is_independent( id.rsd() ) ) ||
					 ( id.type() == id::JUMP && jump_is_independent( id.rsd() ) ) );
}

bool
SymmetryInfo::atom_is_independent( AtomID const & id ) const
{
	return fa_is_independent( id.rsd() );
}

/// @details  Returns set of TorsionIDs that follow a given one. Inefficient: it creates a list each time
SymmetryInfo::TorsionIDs
SymmetryInfo::dependent_torsions( TorsionID const & id ) const
{
	if ( !torsion_is_independent( id ) ) {
		utility_exit_with_message( "SymmetryInfo::dependent_torsions: torsion is not independent!" );
	}

	Size const seqpos( id.rsd() );
	Clones const & seqpos_clones( id.type() == id::BB ? bb_clones( seqpos ) : chi_clones( seqpos ) );

//std::cerr << " dependent_torsions( TorsionID const & )  bb=" << (id.type() == id::BB) << "   chi=" << (id.type() == id::CHI) << "   jump=" << (id.type() == id::JUMP) << std::endl;

	TorsionIDs tors;
	for ( Clones::const_iterator pos= seqpos_clones.begin(), epos=seqpos_clones.end(); pos != epos; ++pos ) {
		tors.push_back( TorsionID( *pos, id.type(), id.torsion() ) );
	}
	return tors;
}

/// @details  Returns set of AtomIDs that follow a given one. Inefficient: it creates a list each time
SymmetryInfo::AtomIDs
SymmetryInfo::dependent_atoms( AtomID const & id ) const
{
  if ( !atom_is_independent( id ) ) {
    utility_exit_with_message( "SymmetryInfo::dependent_atoms: atom is not independent!" );
  }
	Size const seqpos( id.rsd() );
  Clones const & seqpos_clones( bb_clones( seqpos ) );

	AtomIDs atoms;
	for ( Clones::const_iterator pos= seqpos_clones.begin(), epos=seqpos_clones.end(); pos != epos; ++pos ) {
		atoms.push_back( AtomID( id.atomno(), *pos ) );
	}
	return atoms;
}

bool
SymmetryInfo::is_asymmetric_seqpos( Size const res ) const
{
	Size nres_monomer = num_independent_residues(), num_monomers = subunits() ;
	return ( !get_use_symmetry() || res > nres_monomer*num_monomers || res <= nres_monomer );
}

Size
SymmetryInfo::get_asymmetric_seqpos( Size const res ) const
{
	if( res > 0 && get_use_symmetry() ) {
		Size nres_monomer = num_independent_residues(), num_monomers = subunits() ;
		return ( res > nres_monomer*num_monomers ? res - nres_monomer*(num_monomers-1) : (res-1)%nres_monomer + 1 );
	} else {
		return res;
	}
}

Size
SymmetryInfo::subunit_index( Size const seqpos ) const {
	return ( (seqpos-1) / num_independent_residues() + 1 );
}


} // symmetry
} // conformation
} // core


