// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file  core/conformation/symmetry/util.hh
/// @brief utility functions for handling of symmetric conformations
/// @author Ingemar Andre

// Unit headers
#include <core/conformation/symmetry/util.hh>
#include <core/conformation/symmetry/SymDof.hh>
#include <core/conformation/symmetry/SymmetricConformation.hh>
#include <core/scoring/symmetry/SymmetricEnergies.hh>
#include <core/conformation/ResidueFactory.hh>
#include <core/pose/Pose.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/kinematics/MoveMap.hh>

// Utility functions
#include <core/options/option.hh>
#include <core/options/keys/symmetry.OptionKeys.gen.hh>
#include <core/options/keys/fold_and_dock.OptionKeys.gen.hh>
#include <core/id/AtomID.hh>
#include <numeric/random/random.hh>
#include <numeric/xyzTriple.hh>

// Package Headers
#include <core/kinematics/Edge.hh>

#include <core/util/Tracer.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <numeric/xyzVector.hh>

namespace core {
namespace conformation {
namespace symmetry {

static core::util::Tracer TR("core.conformation.symmetry.util");
static numeric::random::RandomGenerator RG(408529); // <- Magic number, do not change it!!!

/// @details  Attempt to detect whether a conformation is symmetric
bool
is_symmetric( Conformation const & conf )
{
	return ( dynamic_cast< SymmetricConformation const * >( &conf ) );
}

bool
is_symmetric( scoring::Energies const & energies )
{
	return ( dynamic_cast< scoring::symmetry::SymmetricEnergies const * >( &energies ) );
}

/// @details  Attempt to detect whether a pose is symmetric
bool
is_symmetric( pose::Pose const & pose )
{
	return is_symmetric( pose.conformation() );
}

bool
is_symmetric( SymmetryInfo const & symminfo )
{
	return symminfo.get_use_symmetry();
}

/// @details Attempts to grab the symmetry info if the pose is symmetric
SymmetryInfo const& symmetry_info( pose::Pose const & pose )
{
	runtime_assert( is_symmetric( pose ) );
  SymmetricConformation const & SymmConf (
   dynamic_cast<SymmetricConformation const &> ( pose.conformation()) );
  return SymmConf.Symmetry_Info() ;
}

/// @details  Attempt to detect whether a conformation is symmetric
bool
scorefxn_is_symmetric( Conformation const & conf )
{
  return ( dynamic_cast< SymmetricConformation const * >( &conf ) );
}

/// @details  Attempt to detect whether a pose is symmetric
bool
scorefxn_is_symmetric( pose::Pose const & pose )
{
  return is_symmetric( pose.conformation() );
}

/// @details Generate a symmetric conformation from a monomeric pose and symmetry information
/// stored in the SymmData object
SymmetricConformationOP
setup_symmetric_conformation(
	Conformation & src_conformation,
	SymmData & symmdata
)
{
	Size njump_orig = src_conformation.fold_tree().num_jump();

	// maybe a little inefficient: first build a standard conformation, then construct symmetric conformation at the end
  Conformation conf( src_conformation );
	// recenter the pose to the origin?
	if ( symmdata.get_recenter() ) {
		recenter( conf, symmdata );
	}

	// Setup temporary variables
  Size const nres_monomer( conf.size() );
  Size const njump_monomer( conf.fold_tree().num_jump() );

	runtime_assert( nres_monomer == symmdata.get_nres_subunit() );
	runtime_assert( njump_monomer == symmdata.get_njump_subunit() );
  // setup the pseudo residues
	std::map< std::string, VirtualCoordinate > coords ( symmdata.get_virtual_coordinates() );
	std::map< std::string, std::pair< std::string, std::string > > virtual_connects( symmdata.get_virtual_connects() );
	std::map< Size, std::string > virtual_num_to_id (symmdata.get_virtual_num_to_id() );

	// Setup virtual residues
  {
    // create the new residue
    chemical::ResidueTypeSet const & rsd_set( src_conformation.residue(1).residue_type_set() );
    ResidueOP rsd( ResidueFactory::create_residue( rsd_set.name_map( "VRT" ) ) );

    // root the fold_tree at this pseudo residue
//    {
//      kinematics::FoldTree f( conf.fold_tree() );
//      f.reorder( conf.size() );
//      conf.fold_tree( f );
//    }

		for ( Size i =1; i <= virtual_num_to_id.size(); ++i ) {
			if ( virtual_num_to_id.find( i ) == virtual_num_to_id.end() ) {
				utility_exit_with_message( "[ERROR] Cannot find jump number..." );
			}
			std::string tag ( virtual_num_to_id.find( i )->second );
			if ( coords.find( tag ) == coords.end() ) {
        utility_exit_with_message( "[ERROR] Cannot find tag " + tag );
      }
			VirtualCoordinate virt_coord( coords.find( tag )->second );
      rsd->set_xyz( "ORIG", virt_coord.get_origin() );
      rsd->set_xyz( "X", virt_coord.get_x().normalized() + virt_coord.get_origin() );
      rsd->set_xyz( "Y", virt_coord.get_y().normalized() + virt_coord.get_origin() );
      conf.append_residue_by_jump( *rsd, conf.size() ); //append it to the end of the monomeri
		}
	}

	// now insert the other conformations. This step will create a fold_tree that will be discarded at a later stage.
	// Optimally we want to set the fold tree directly but that turns out to be difficult. TODO change
  kinematics::Jump const monomer_jump( conf.jump( njump_monomer+1 ) );
  for ( Size i=1; i< symmdata.get_subunits(); ++i ) {
		 Size const insert_seqpos( nres_monomer * i + 1 ); // desired sequence position of residue 1 in inserted conformation
    Size const insert_jumppos( njump_monomer * i + 1 ); // desired jump number of 1st jump in inserted conformation
    Size const anchor_pseudo( nres_monomer * i + i + 1 ); // in current numbering
    Size const new_jump_number( njump_monomer*( i + 1 ) + i + 1 );
    conf.insert_conformation_by_jump( src_conformation, insert_seqpos, insert_jumppos, anchor_pseudo, new_jump_number);
//		conf.set_jump( new_jump_number, monomer_jump );
  }
	// Now generate the fold_tree from the symmdata and set it into the conformation. In the future we want to be able to use a
	// atom tree for this.
	kinematics::FoldTree f ( set_fold_tree_from_symm_data( src_conformation, symmdata ) );
	// set the root of the fold tree
	Size new_root ( conf.size() - coords.size() + symmdata.get_root() );
	f.reorder( new_root );
	TR.Debug << f << std::endl;
	conf.fold_tree( f );

  // now build the symmetry info object
	SymmetryInfo const symm_info( symmdata );

  // now create the symmetric conformation
	SymmetricConformationOP symm_conf = new SymmetricConformation( conf, symm_info );

	// apply independent jumps so the structure is created symmetrically
	for ( Size i=1; i<= f.num_jump(); ++i ) {
		if ( symm_info.jump_is_independent( i ) )
			symm_conf->set_jump( i, conf.jump( i ) );
	}

	// renumber the dof information to take the internal jumps into consideration
	shift_jump_numbers_in_dofs( *symm_conf, njump_orig * symmdata.get_subunits() );

	return symm_conf;

}

// @details Make a fold tree from data stored in the SymmData object. This would have to make sense.
// this function would probably enjoy more checks...
kinematics::FoldTree
set_fold_tree_from_symm_data(
	Conformation & src_conformation,
  SymmData & symmdata
)
{
	using namespace kinematics;
	FoldTree f, f_orig = src_conformation.fold_tree();
	f.clear();

	// Save number of monomer residues, subunits and virtuals
	Size num_res_subunit( src_conformation.size() );
	Size subunits( symmdata.get_subunits() );
	Size num_cuts_subunit( f_orig.num_cutpoint() );
	Size num_jumps_subunit( f_orig.num_jump() );
	Size num_virtuals( symmdata.get_virtual_coordinates().size() );
	Size num_virtual_connects ( symmdata.get_virtual_connects().size() );
	Size num_res_real( num_res_subunit*subunits );
	Size anchor ( symmdata.get_anchor_residue() );

	// Check that input data makes sense
	if ( anchor > num_res_subunit ) {
			utility_exit_with_message( "Anchor outside the subunit..." );
	}

	// Store number of jumps and cuts
	Size njumps( 0 );
	Size total_num_cuts( num_cuts_subunit*subunits + subunits + num_virtuals - 1 );
	Size total_num_jumps( num_jumps_subunit*subunits + num_virtual_connects );

	// store information from subunit foldtree
	utility::vector1< int > cuts_subunit( f_orig.cutpoints() );
	ObjexxFCL::FArray1D_int cuts( total_num_cuts );
	ObjexxFCL::FArray2D_int jump_points_subunit( 2, num_jumps_subunit ),
													jumps( 2, total_num_jumps );

	for ( Size i = 1; i<= f_orig.num_jump(); ++i ) {
		jump_points_subunit(1,i) = f_orig.upstream_jump_residue(i);
		jump_points_subunit(2,i) = f_orig.downstream_jump_residue(i);
	}

	// Now add jumps and cuts for the other subunits
	for ( Size i = 0; i < subunits; ++i ) {
    for ( Size j = 1; j <= num_jumps_subunit; ++j ) {
      ++njumps;
			int new_cut_pos( i*num_res_subunit + cuts_subunit.at( j ) );
			cuts( njumps ) = new_cut_pos;
      int new_jump_pos1( i*num_res_subunit + jump_points_subunit(1,j) );
			int new_jump_pos2( i*num_res_subunit + jump_points_subunit(2,j) );
      jumps(1, njumps ) = new_jump_pos1;
			jumps(2, njumps ) = new_jump_pos2;
    }
  }

	// why copy the maps?
	std::map< std::string, std::pair< std::string, std::string > > const virtual_connect( symmdata.get_virtual_connects() );
	std::map< std::string, Size > virtual_id_to_num (symmdata.get_virtual_id_to_num() );
	std::map< Size, std::string > virtual_num_to_id (symmdata.get_virtual_num_to_id() );
	std::map< std::string, Size > virtual_id_to_subunit (symmdata.get_virt_id_to_subunit_num() );

	std::map< std::string, std::pair< std::string, std::string > >::const_iterator it_start = virtual_connect.begin();
	std::map< std::string, std::pair< std::string, std::string > >::const_iterator it_end = virtual_connect.end();
	for ( std::map< std::string, std::pair< std::string, std::string > >::const_iterator it = it_start; it != it_end; ++it ) {
		std::pair< std::string, std::string > connect( it->second );
		std::string pos_id1( connect.first );
		std::string pos_id2( connect.second );
		if ( pos_id2 == "SUBUNIT" ) {
			int subunit = virtual_id_to_subunit.find( pos_id1 )->second;
			++njumps;
			cuts( njumps ) = num_res_subunit*subunit;
			int jump_pos1( ( subunit-1 )*num_res_subunit + anchor );
			int jump_pos2( num_res_real + virtual_id_to_num.find( pos_id1 )->second );
			jumps(1, njumps ) = jump_pos1;
			jumps(2, njumps ) = jump_pos2;
		}
	}

	// Read jump structure from symmdata

	// Read the virtual connect data and add the new connections. They are stored in virtual_connects. We also need to
	// know the mapping between mapping between name of virtual residues and the jump number
	Size pos( 0 );
	for ( std::map< std::string, std::pair< std::string, std::string > >::const_iterator it = it_start; it != it_end; ++it ) {
				std::pair< std::string, std::string > connect( it->second );
		std::string pos_id1( connect.first );
		std::string pos_id2( connect.second );
		// We have already added the jumps from virtual residues to their corresponding subunits
		if ( pos_id2 == "SUBUNIT" ) continue;
		++njumps;
		++pos;
		assert( virtual_id_to_num.find( pos_id1 ) != virtual_id_to_num.end() && virtual_id_to_num.find( pos_id2 ) != virtual_id_to_num.end() );
		Size pos1 ( virtual_id_to_num.find( pos_id1 )->second );
		Size pos2 ( virtual_id_to_num.find( pos_id2 )->second );
     cuts( njumps ) = num_res_real + pos;
		if ( pos2 > pos1 ) {
			jumps(1, njumps ) = num_res_real + pos1;
			jumps(2, njumps ) = num_res_real + pos2;
		} else {
			jumps(1, njumps ) = num_res_real + pos2;
			jumps(2, njumps ) = num_res_real + pos1;
		}
	}

	// Now create foldtree
	f.tree_from_jumps_and_cuts( num_res_real + num_virtuals, njumps, jumps, cuts );
	f.reorder( num_res_real + 1 );
	return f;
}

// this function is STILL under construction...
kinematics::FoldTree
replaced_symmetric_foldtree_with_new_monomer(
	kinematics::FoldTree symm_f,
	SymmetryInfo symmetry_info,
	kinematics::FoldTree monomer_f
)
{
	using namespace kinematics;
	FoldTree f=symm_f, f_new=monomer_f;

	// Save number of monomer residues, subunits and virtuals
	Size num_res_subunit( symmetry_info.num_independent_residues() );
	Size subunits( symmetry_info.subunits() );
	//Size anchor ( symmdata.get_anchor_residue() );


	// Now add jumps and cuts for the other subunits
	for ( Size i = 0; i < subunits; ++i ) {
		int begin ( i*num_res_subunit+1 );
		int end ( (i+1)*num_res_subunit );
		f.delete_segment( begin, end );
		//f.insert_fold_tree_by_jump(f, insert_seqpos, insert_jumppos, anchor_pos, anchor_jump_number, anchor_atom, root_atom);

  }

	return f;
}


/// @details constructs a symmetric pose with a symmetric conformation and energies object from a monomeric pose
/// and symmetryinfo object
void
make_symmetric_pose(
	pose::Pose & pose,
	SymmetryInfo symmetry_info
)
{
	SymmetricConformationOP symm_conf ( new core::conformation::symmetry::SymmetricConformation( pose.conformation(), symmetry_info ) ) ;

	scoring::symmetry::SymmetricEnergiesOP symm_energy( new scoring::symmetry::SymmetricEnergies( pose.energies()) );

	pose.set_new_conformation( symm_conf );
	pose.set_new_energies_object( symm_energy );
	assert( is_symmetric( pose ) );

}

/// @details constructs a symmetric pose with a symmetric conformation and energies object from a monomeric pose
/// and symmData object
void
make_symmetric_pose(
	pose::Pose & pose,
	SymmData & symmdata
)
{

	SymmetricConformationOP symm_conf
    ( setup_symmetric_conformation( pose.conformation(), symmdata ) );

	scoring::symmetry::SymmetricEnergiesOP symm_energy ( new scoring::symmetry::SymmetricEnergies( pose.energies()) );

  pose.set_new_conformation( symm_conf );
  pose.set_new_energies_object( symm_energy );
	assert( is_symmetric( pose ) );

}

/// @details constructs a symmetric pose with a symmetric conformation and energies object from a monomeric pose
/// and symmetry definition file on command line. Requires the presence of a symmetry_definition file
void
make_symmetric_pose(
  pose::Pose & pose
)
{
	using namespace options;

	SymmData symmdata( pose );
	std::string symm_def = option[ OptionKeys::symmetry::symmetry_definition ];
  symmdata.read_symmetry_data_from_file(symm_def);
	make_symmetric_pose( pose, symmdata );
}

// @details make a symmetric pose assymmetric by instantiating new conformation and energies objects
// nukes the objects!!!
void
make_asymmetric_pose(
	pose::Pose & pose
 )
{
	scoring::EnergiesOP energies = new scoring::Energies();
	conformation::ConformationOP conformation = new conformation::Conformation();
	pose.set_new_conformation( conformation );
	pose.set_new_energies_object( energies );
	assert( !is_symmetric( pose ) );
}

// @details center the pose at the origin. Use the CA of the anchor residue
// as the anchor point
void
recenter(
	Conformation & src_conformation,
  SymmData & symmdata
)
{
	Residue anchor ( src_conformation.residue( symmdata.get_anchor_residue() ) );
	Vector trans( anchor.xyz( "CA" ) );
	for ( Size i = 1; i <= src_conformation.size(); ++i ) {
    for ( Size j = 1; j <= src_conformation.residue_type(i).natoms(); ++j ) {
      id::AtomID id( j, i );
      src_conformation.set_xyz( id, src_conformation.xyz(id) - trans );
    }
  }
}

// @details shift jump numbers in dof
void
shift_jump_numbers_in_dofs(
	Conformation & conformation,
	Size shift
)
{
	using namespace core::conformation::symmetry;

	runtime_assert( is_symmetric( conformation ) );
	SymmetricConformation & symm_conf (
	      dynamic_cast<SymmetricConformation & > ( conformation ) );
	SymmetryInfo & symm_info( symm_conf.Symmetry_Info() );

	std::map< Size, SymDof > dofs ( symm_conf.Symmetry_Info().get_dofs() );
	std::map< Size, SymDof > dofs_shifted;
	std::map< Size, SymDof >::iterator it;
	std::map< Size, SymDof >::iterator it_begin = dofs.begin();
	std::map< Size, SymDof >::iterator it_end = dofs.end();

	for ( it = it_begin; it != it_end; ++it ) {
		int jump_nbr ( (*it).first + shift );
		dofs_shifted.insert( std::make_pair( jump_nbr, (*it).second ) );
	}
	symm_info.set_dofs( dofs_shifted );
}

// @details setting the movemap to only allow for symmetrical dofs.
// Jumps information is discarded and dof information in symmetry_info
// object is used instead
void
make_symmetric_movemap(
	pose::Pose const & pose,
	kinematics::MoveMap & movemap
)
{
	using namespace core::conformation::symmetry;

	runtime_assert( is_symmetric( pose ) );
  SymmetricConformation const & symm_conf (
        dynamic_cast<SymmetricConformation const & > ( pose.conformation()) );
	SymmetryInfo const & symm_info( symm_conf.Symmetry_Info() );

	// allow only one subunit to change chi and torsions
	for ( Size i=1; i<= pose.conformation().size(); ++i ) {
		if ( !symm_info.bb_is_independent( i ) ) {
			movemap.set_bb ( i,false );
			movemap.set_chi( i, false );
		}
	}

	// use the dof information in the symmetry_info object set allowed rb
	// dofs
  std::map< Size, SymDof > dofs ( symm_info.get_dofs() );

	std::map< Size, SymDof >::iterator it;
  std::map< Size, SymDof >::iterator it_begin = dofs.begin();
  std::map< Size, SymDof >::iterator it_end = dofs.end();

	// first find out whether we have any allowed jumps. we just need to
	// find one jump that is true to know that set_jump have been set to
	// true
	kinematics::MoveMapOP movemap_in ( new kinematics::MoveMap( movemap ) );

	// First set all jump to false
	movemap.set_jump(false);

	// Then allow only dofs according to the dof information
  for ( it = it_begin; it != it_end; ++it ) {
    int jump_nbr ( (*it).first );
		SymDof dof( (*it).second );

		//bool jump_move( false );
		bool jump_move( movemap_in->get_jump( jump_nbr ) );
		for ( Size i = X_DOF; i <= Z_ANGLE_DOF; ++i ) {
			id::DOF_ID const & id
				( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,i)));
			jump_move |= movemap_in->get( id );
		}
		if ( !jump_move ) {
			continue;
		}

		if ( dof.allow_dof( X_DOF ) ) {
			id::DOF_ID const & id
				( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,1)));
			movemap.set( id, true );
		}
		if ( dof.allow_dof( Y_DOF ) ) {
			id::DOF_ID const & id
				( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,2)));
			movemap.set( id, true );
		}
		if ( dof.allow_dof( Z_DOF ) ) {
			id::DOF_ID const & id
				( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,3)));
			movemap.set( id, true );
		}
		if ( dof.allow_dof( X_ANGLE_DOF ) ) {
			id::DOF_ID const & id
				( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,4)));
			movemap.set( id, true );
		}
		if ( dof.allow_dof( Y_ANGLE_DOF ) ) {
			id::DOF_ID const & id
				( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,5)));
			movemap.set( id, true );
		}
		if ( dof.allow_dof( Z_ANGLE_DOF ) ) {
			id::DOF_ID const & id
				( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,6)));
			movemap.set( id, true );
		}
	}
}

// @details find the anchor residue in the first subunit. This function assumes that the
// fold tree is rooted at the first jump from a virtual to its subunit
/*int
find_symmetric_basejump_anchor( pose::Pose & pose )
{
	kinematics::FoldTree const & f  = pose.conformation().fold_tree();
	kinematics::FoldTree::const_iterator it=f.begin();
	assert ( it->is_jump() );
	return it->stop();
}*/
// @details find the anchor residue in the first subunit. This function assumes that
// there is only one one jump between a subunit and a virtual and that the subunit is
// folded downstream to the virtual
int
find_symmetric_basejump_anchor( pose::Pose & pose )
{
	kinematics::FoldTree const & f  = pose.conformation().fold_tree();
	for ( int i = 1; f.num_jump(); ++i ) {
		if ( pose.residue( f.downstream_jump_residue(i) ).name() != "VRT" &&
				 pose.residue( f.upstream_jump_residue(i) ).name() == "VRT"	 )
			return f.downstream_jump_residue(i);
	}
	utility_exit_with_message( "No anchor residue is found..." );
	return 0;
}

// @ details move the anchor residue of a symmetric system. This function moves the anchors from
// virtual residues that define the coordinate system to the subunits. Useful during fragment insertion
// make sure to call rotate_to_x after this step so that the anchor moves correctly in the coordinate
// system defined by the virtual. The new anchor point is selected randomly or is selected as the point
// where the chains are closest together
void
find_new_symmetric_jump_residues( core::pose::Pose & pose )
{
  using namespace core::conformation::symmetry;

  runtime_assert( is_symmetric( pose ) );
  SymmetricConformation & symm_conf (
        dynamic_cast<SymmetricConformation & > ( pose.conformation()) );
  SymmetryInfo const & symm_info( symm_conf.Symmetry_Info() );

  Size nres_subunit ( symm_info.num_independent_residues() );
  Size anchor_start ( conformation::symmetry::find_symmetric_basejump_anchor( pose ) );

  //Looking in central half of each segment -- pick a random point.
  Size anchor = static_cast<Size>( RG.uniform() * (nres_subunit/2) ) +
    (nres_subunit/4) + 1;

  if ( core::options::option[ core::options::OptionKeys::fold_and_dock::set_anchor_at_closest_point ] )
  {
    //Find Closest point of contact.
    core::Real mindist = pose.residue(anchor).xyz("CEN").distance( pose.residue(anchor + nres_subunit).xyz("CEN") );
    for (Size i = 1; i <= nres_subunit; i++) {
      Size const j = i + nres_subunit;
      core::Real dist = pose.residue(i).xyz("CEN").distance( pose.residue(j).xyz("CEN") );
      if ( dist < mindist ){
        mindist = dist;
        anchor = i;
      }
    }
  }

  // Update the fold tree with the new jump points
  kinematics::FoldTree f ( pose.conformation().fold_tree() );

	// Setyp the lists of jumps and cuts
	Size num_jumps( f.num_jump() );
	Size num_cuts( f.num_cutpoint() );

	utility::vector1< int > cuts_vector( f.cutpoints() );
	ObjexxFCL::FArray1D_int cuts( num_cuts );
	ObjexxFCL::FArray2D_int jumps( 2, num_jumps );

	// Initialize jumps
	for ( Size i = 1; i<= num_jumps; ++i ) {
		int down ( f.downstream_jump_residue(i) );
		int up ( f.upstream_jump_residue(i) );
		if ( down < up ) {
			jumps(1,i) = down;
			jumps(2,i) = up;
		} else {
			jumps(1,i) = up;
			jumps(2,i) = down;
		}
	}

	for ( Size i = 1; i<= num_cuts; ++i ) {
		cuts(i) = cuts_vector[i];
	}

  // This is the basejump
  int root ( f.root() );
  int const jump_number ( f.get_jump_that_builds_residue( anchor_start ) );
	int residue_that_builds_anchor( f.upstream_jump_residue( jump_number ) );

	jumps(1, jump_number ) = anchor - symm_info.get_asymmetric_seqpos( anchor_start ) + anchor_start;
	jumps(2, jump_number ) = residue_that_builds_anchor;

  for ( std::vector< Size>::const_iterator
          clone     = symm_info.bb_clones( anchor_start ).begin(),
          clone_end = symm_info.bb_clones( anchor_start ).end();
          clone != clone_end; ++clone ) {
    int jump_clone ( f.get_jump_that_builds_residue( *clone ) );
    int takeoff_pos ( f.upstream_jump_residue( jump_clone ) );
    int new_anchor ( anchor - symm_info.get_asymmetric_seqpos( anchor_start ) + *clone );
		runtime_assert( jumps(1,jump_clone) == int( *clone ) && jumps(2,jump_clone) == takeoff_pos );
		jumps(1, jump_clone ) = new_anchor;
	  jumps(2, jump_clone ) = takeoff_pos;
	  //std::cout<<"new_anchor "<<new_anchor<< " " <<anchor<<" "<<anchor_start<<" "<<symm_info.get_asymmetric_seqpos( anchor_start )<<std::endl;
  }
  /* debug
  std::cout<<"cuts ";
	for ( Size i = 1; i<= num_cuts; ++i ) {
		std::cout<< cuts(i) << ' ';
	}
	std::cout<<std::endl;
  std::cout<<"jumps ";
	for ( Size i = 1; i<= num_jumps; ++i ) {
		std::cout<< " ( "<<jumps(1,i) << " , " << jumps(2,i) << " ) ";
	}
	std::cout<<std::endl; */
	f.tree_from_jumps_and_cuts( pose.conformation().size(), num_jumps, jumps, cuts );
	f.reorder( root );
   pose.conformation().fold_tree( f );
}

// @details this function rotates a anchor residue to the x-axis defined by the virtual residue
// at the root of the fold tree. It is necessary that the anchor is placed in the coordinate system
// of the virtual residue so that rigid body movers and minimization code moves in the correct direction
void
rotate_anchor_to_x_axis( core::pose::Pose & pose ){

  //first anchor point -- assume its the first jump.
  kinematics::FoldTree f( pose.fold_tree() );
	int anchor ( find_symmetric_basejump_anchor( pose ) );
	int const jump_number ( f.get_jump_that_builds_residue( anchor ) );
	int coordsys_residue( f.upstream_jump_residue( jump_number ) );
  //Where is this stupid anchor point now?
	Vector anchor_pos ( pose.residue( anchor ).xyz("CA") );
  float theta = std::atan2(  anchor_pos(2), anchor_pos(1) );

	// we should make sure that the residue type is VRT
	runtime_assert( pose.residue( coordsys_residue ).name() == "VRT" );
	Vector x_axis( pose.residue( coordsys_residue ).xyz("X") - pose.residue( coordsys_residue ).xyz("ORIG") );
	Vector y_axis( pose.residue( coordsys_residue ).xyz("Y") - pose.residue( coordsys_residue ).xyz("ORIG") );


	Vector z_axis( x_axis );
	z_axis.normalize();
	Vector y_axis_nrml( y_axis );
	y_axis_nrml.normalize();
	z_axis = z_axis.cross( y_axis_nrml );
	z_axis.normalize();


	numeric::xyzMatrix< Real > coordsys_rot ( numeric::xyzMatrix< Real >::rows( x_axis, y_axis, z_axis ) );
	// make sure it is a rotation matrix
	runtime_assert( std::abs( coordsys_rot.det() - 1.0 ) < 1e-6 );

	// initialize a rotation matrix that rotates the anchor to cartesian x
  numeric::xyzMatrix< Real > z_rot = numeric::z_rotation_matrix_degrees(
		( -1.0 * numeric::conversions::to_degrees( theta ) ) );

	kinematics::Jump base_jump( pose.jump( jump_number ) );
	// rotate around absolute x
	Vector center(0,0,0);
	// Find the stub
	core::kinematics::Stub upstream_stub = pose.conformation().upstream_jump_stub( jump_number );
	// rotate to cartesian x, then rotate to the coordinate system defined by the virtual residue
	base_jump.rotation_by_matrix( upstream_stub, center, coordsys_rot.transpose()*z_rot );
	// set the jump
	pose.set_jump( jump_number, base_jump );
	// The convention is to place the subunit so that it has positive x values for slide moves to
	// go in the right direction. Silly!!!
	Vector new_anchor_pos ( pose.residue( anchor ).xyz("CA") );
//	std::cout << "before: " << anchor_pos(1) << " " << anchor_pos(2) << " " << anchor_pos(3) << std::endl
//		<<  "after: " << new_anchor_pos(1) << " " << new_anchor_pos(2) << " " << new_anchor_pos(3) << std::endl;
	if ( new_anchor_pos(1) < 0 ) {
		upstream_stub = pose.conformation().upstream_jump_stub( jump_number );
		numeric::xyzMatrix< Real > twofold = numeric::z_rotation_matrix_degrees( 180.0 );
		base_jump.rotation_by_matrix( upstream_stub, center, twofold );
		pose.set_jump( jump_number, base_jump );
	}
}



void
symmetrize_fold_tree( core::pose::Pose const &p, kinematics::FoldTree &f ) {
  if( !is_symmetric( p ) ) {
		return;
	}

	// basic idea: grabs jumps and cuts from monomer, symmetrize them,
	//    retain VRT jumps and cuts, generate a new fold tree
 	kinematics::FoldTree f_orig = f;

  SymmetricConformation const & symm_conf (
        dynamic_cast<SymmetricConformation const & > ( p.conformation() ) );
  SymmetryInfo const & symm_info( symm_conf.Symmetry_Info() );
  Size nres_subunit ( symm_info.num_independent_residues() );
  Size nsubunits ( symm_info.subunits() );
	Size num_nonvrt = symm_info.num_total_residues_without_pseudo();

	// 1 - Get monomer jumps, cuts, and anchor
	Size new_anchor = 0;
	utility::vector1< Size > new_cuts;
	utility::vector1< std::pair<Size,Size> > new_jumps;

	for ( Size i = 1; i<= f_orig.num_jump(); ++i ) {
		Size down ( f_orig.downstream_jump_residue(i) );
		Size up ( f_orig.upstream_jump_residue(i) );
		if ( up > nres_subunit ) {
			new_anchor = down;
			TR << "symmetrize_fold_tree(): setting anchor to " << new_anchor << std::endl;
		} else {
			// add this jump in each subunit
			for ( Size j=0;  j<nsubunits; ++j) {
				// dont worry about order, they get sorted later
				new_jumps.push_back( std::pair<int,int>( j*nres_subunit+down, j*nres_subunit+up ) );
			}
		}
	}
	utility::vector1< int > cuts_vector( f_orig.cutpoints() );

	// 1A - symmetrize
	for ( Size i = 0;  i<nsubunits; ++i) {
		for ( Size j = 1; j<=cuts_vector.size(); ++j ) {
			if (i*nres_subunit + cuts_vector[j] != num_nonvrt)
				new_cuts.push_back( i*nres_subunit + cuts_vector[j] );
		}
	}

	// 2 - Get symmetic jumps cuts and anchor
	// inter-VRT jumps
	kinematics::FoldTree const &f_pose = p.fold_tree();
	for ( Size i = 1; i<= f_pose.num_jump(); ++i ) {
		Size down ( f_pose.downstream_jump_residue(i) );
		Size up ( f_pose.upstream_jump_residue(i) );
		// connections between VRTs are unchanged
		if ( up > num_nonvrt && down > num_nonvrt) {
			new_jumps.push_back( std::pair<Size,Size>(up,down) );
		}
		// jumps to new anchor
		if ( up > num_nonvrt && down < num_nonvrt) {
			int subunit_i = symm_info.subunit_index(down);
			new_jumps.push_back( std::pair<Size,Size>( up, (subunit_i-1)*nres_subunit + new_anchor ) );
		}
	}

	// cuts
	cuts_vector = f_pose.cutpoints();
	for ( Size i = 1; i<=cuts_vector.size(); ++i ) {
		if ( (int)cuts_vector[i] >= num_nonvrt)
			new_cuts.push_back( cuts_vector[i] );
	}

	// 3 - combine
	ObjexxFCL::FArray1D_int cuts( new_cuts.size() );
	ObjexxFCL::FArray2D_int jumps( 2, new_jumps.size() );
	// Initialize jumps
	for ( Size i = 1; i<= new_jumps.size(); ++i ) {
		jumps(1,i) = std::min( (int)new_jumps[i].first, (int)new_jumps[i].second);
		jumps(2,i) = std::max( (int)new_jumps[i].first, (int)new_jumps[i].second);
		// DEBUG -- PRINT JUMPS AND CUTS
		//std::cerr << " jump " << i << " : " << jumps(1,i) << " , " << jumps(2,i) << std::endl;
	}
	for ( Size i = 1; i<= new_cuts.size(); ++i ) {
		cuts(i) = (int)new_cuts[i];
		//std::cerr << " cut " << i << " : " << cuts(i) << std::endl;
	}

	// 4 make the foldtree
	f.clear();
	f.tree_from_jumps_and_cuts( p.total_residue(), new_jumps.size(), jumps, cuts );
	f.reorder( f_pose.root() );
}


void
set_asymm_unit_fold_tree( core::pose::Pose &p, kinematics::FoldTree const &f) {
  if( !is_symmetric( p ) ) {
		return p.fold_tree( f );
	}

	kinematics::FoldTree f_new = f;
	TR.Error << "set_asymm_unit_fold_tree() called with " << f << std::endl;
	symmetrize_fold_tree( p, f_new );
	TR.Error << "set_asymm_unit_fold_tree() setting fold tree to " << f_new << std::endl;
	p.fold_tree( f_new );
}


kinematics::FoldTree
get_asymm_unit_fold_tree( core::pose::Pose const &p ) {
  if( !is_symmetric( p ) ) {
		return p.fold_tree();
	}

	// basic idea: only take edges with at least one end in the 1st subunit
	kinematics::FoldTree const &f = p.fold_tree();
	kinematics::FoldTree f_new;

  SymmetricConformation const & symm_conf (
        dynamic_cast<SymmetricConformation const & > ( p.conformation()) );
  SymmetryInfo const & symm_info( symm_conf.Symmetry_Info() );
  Size nres_subunit ( symm_info.num_independent_residues() );

	for ( core::kinematics::FoldTree::const_iterator it=f.begin(), eit=f.end(); it != eit; ++it) {
		if ( it->start() <=	(int)nres_subunit && it->stop() <=(int)nres_subunit ) {
			f_new.add_edge( *it );
		} else if ( it->stop() <= nres_subunit ) { // this is the jump to the subunit
			core::kinematics::Edge e_new = *it;
			e_new.start() = nres_subunit + 1;
			f_new.add_edge( e_new );
		}
	}

	TR.Error << "get_asymm_unit_fold_tree() called with " << f << std::endl;
	TR.Error << "get_asymm_unit_fold_tree() returning " << f_new << std::endl;
	return f_new;
}

void
make_residue_mask_symmetric( core::pose::Pose const &p, utility::vector1< bool > & msk ) {
  if( !is_symmetric( p ) ) return;

  SymmetricConformation const & symm_conf (
        dynamic_cast<SymmetricConformation const & > ( p.conformation() ) );
  SymmetryInfo const & symm_info( symm_conf.Symmetry_Info() );
  Size nres_subunit ( symm_info.num_independent_residues() );
  Size nsubunits ( symm_info.subunits() );

	runtime_assert( symm_info.num_total_residues_with_pseudo() == msk.size() );

	for (int i=1; i<= msk.size(); ++i) {
		bool msk_i=msk[i];
		if (msk_i && !symm_info.fa_is_independent( i )) {
			msk[i] = false;
			msk[ symm_info.bb_follows( i ) ] = true;
		}
	}
}


} // symmetry
} // conformation
} // core
