// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragments/FragData.hh
/// @brief  A fragment as list of SingleResidue Data
/// @author Oliver Lange (olange@u.washington.edu)
/// @date   Wed Oct 20 12:08:31 2007
///
#ifndef core_fragments_FragData_HH
#define core_fragments_FragData_HH

// Unit Headers
#include <core/fragment/FragData.fwd.hh>

// Package Headers
#include <core/fragment/Frame.fwd.hh>
#include <core/fragment/SingleResidueFragData.hh>

// Project Headers
#include <core/pose/Pose.fwd.hh>
#include <core/kinematics/MoveMap.fwd.hh>
#include <core/types.hh>

#include <core/conformation/Residue.fwd.hh> // for ResidueSRFD

#include <core/kinematics/types.hh>
#include <core/id/TorsionID.hh>

// Utility headers
#include <utility/vector1.hh>
#include <utility/pointer/ReferenceCount.hh>

#include <iostream>

namespace core {
namespace fragment {

typedef utility::vector1 < Size > PositionList;

class FragData : public utility::pointer::ReferenceCount
{
public:
	typedef	utility::vector1 < SingleResidueFragDataOP > SRFD_List;

public:

	FragData () : valid_( false ) {};

	//@brief convience constructor to create FragData object that contains n SRFD of same type
	// argument SRFD will only be used for cloning
	FragData( SingleResidueFragDataOP, core::Size n);

	//@brief
	virtual ~FragData() {};

	virtual
	FragDataOP clone() const;

	//might be overwritten to shortcut for continous changes
	virtual Size apply( kinematics::MoveMap const&, pose::Pose&, Size start, Size end ) const; // continous application to length residues
	virtual Size apply( kinematics::MoveMap const&, pose::Pose&, Frame const& ) const; //application to any set of residues specified by the Fr

	//without movemap --- just apply the full fragment
	virtual Size apply( pose::Pose&, Frame const& ) const; //application to any set of residues specified by the Fr
	virtual Size apply( pose::Pose&, Size start, Size end ) const; // continous application to length residues

	//	virtual bool apply( pose::Pose&, PositionList const& ) const; //application to any set of residues specified by list

	// @brief apply framents sec-struct info to ss-string
	virtual Size apply_ss( kinematics::MoveMap const&, std::string& ss, Frame const& ) const;

	// @brief check weather the dofs changed by this fragment are allowed according to movemap
	// return the number of allowed residues before a disabled one is met
	virtual Size is_applicable( kinematics::MoveMap const&, Size start, Size end ) const;
	virtual Size is_applicable( kinematics::MoveMap const&, Frame const& ) const;

	//
	// set FragData from the given pose ( inverse of apply )
	virtual bool steal( pose::Pose const&, Size start, Size end ); // continous application to length residues
	virtual bool steal( pose::Pose const&, Frame const& ); //application to any set of residues specified by the Frame
	//	virtual bool apply( pose::Pose&, PositionList const& ) const; //application to any set of residues specified by list

	// check if both frag_data contain the same number and types of SRFD
	bool is_compatible( FragData const& frag_data ) const;

	bool is_valid() const {
		return valid_;
	}

	/*	// this shows which degrees of freedom would be changed
	virtual void generate_move_map( MoveMap&, Size start = 1 ) const {
		start = 1; //avoid warning
		std::cerr << "generate_move_map in FragData has stubbed out " << std::endl;
	};

	virtual void generate_move_map( MoveMap&, PositionList const& ) const {
		std::cerr << "generate_move_map in FragData has stubbed out " << std::endl;
	}

	virtual void generate_move_map( MoveMap&, Frame const& ) const {
		std::cerr << "generate_move_map in FragData has stubbed out " << std::endl;
	}
	*/

	//
	Size size() const
	{ return data_.size(); };

	/// set secstruct for this position
	//// 	void
// 	set_secstruct( Size const pos, char const setting )
// 	{
// 		data_[pos]->set_secstruct( setting );
// 	}

 	char
		secstruct( Size const pos) const {
 		return data_[pos]->secstruct();
 	}

	char sequence( Size const pos ) const{
		return data_[ pos ] -> sequence();
	};

	std::string sequence() const {
		std::string str;
		for (Size pos=1; pos<=size(); pos++ ) {
			str.push_back( sequence( pos) );
		};
		return str;
	};

	void set_sequence( Size const pos, char const setting ) {
		data_[ pos ] -> set_sequence( setting );
	}

	void set_residue( Size pos, SingleResidueFragDataOP res ) {
		if ( pos < data_.size() ) {
			data_.resize( pos );
		};
		data_[ pos ] = res;
	};

	void add_residue( SingleResidueFragDataOP res ) {
		data_.push_back ( res );
	};

	SingleResidueFragDataCOP get_residue( core::Size pos ) const {
		return data_[ pos ];
	}


	//returns new FragData object with residues start...stop
	FragDataOP generate_sub_fragment( Size start, Size stop );

	virtual
	void show( std::ostream &out ) const;

	virtual
	void show( std::ostream &out, Frame const& ) const;

	virtual
	void show_classic( std::ostream &out ) const;

	//@brief notify FragData that it now contains some proper numbers ( would like to have this protected )
	// but IO needs access to this.
	void set_valid( bool setting = true ) {
		valid_ = setting;
	}

	virtual
	std::string pdbid() const {
		return "no_pdb";
	}

	virtual
	Size pdbpos() const {
		return 0;
	}


protected: // make private
	FragData( Size nr_res ) : data_( nr_res ) {};


private:
	SRFD_List data_;

	// this will be true if dofs are set ( via steal, or IO )
	bool valid_;
};

class AnnotatedFragData : public FragData {
	typedef FragData Parent;
public:
	AnnotatedFragData( std::string pdbid, Size startpos ) :
		pdbid_( pdbid),
		startpos_( startpos )
	{ }

	AnnotatedFragData( std::string pdbid, Size startpos, FragData const& frag ) :
		FragData( frag ),
		pdbid_( pdbid ),
		startpos_( startpos )
	{	};

	virtual
	FragDataOP clone() const;

	virtual
	std::string pdbid() const  {
		return pdbid_;
	}

	virtual
	Size pdbpos() const {
		return startpos_;
	}

private:
	std::string pdbid_;
	Size startpos_; //or list of indices
};


inline std::ostream& operator<<( std::ostream& out, FragData const& fr ) {
	fr.show( out );
	return out;
}

} // fragment
} // core

#endif
