// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragments/Frame.cc
/// @brief  set of fragments for a certain alignment frame
/// @author Oliver Lange (olange@u.washington.edu)
/// @date   Wed Oct 20 12:08:31 2007
///

// Unit Headers
#include <core/fragment/Frame.hh>

// Package Headers

// Project Headers
#include <core/pose/Pose.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/conformation/ResidueFactory.hh>

#include <core/sequence/SequenceMapping.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>

// Utility headers
#include <utility/vector1.fwd.hh>
#include <core/util/Tracer.hh>

namespace core {
namespace fragment {

using namespace kinematics;


static util::Tracer tr("core.fragment");
//pose::PoseOP Frame::my_static_pose_for_testing_ = NULL;
void
make_pose_from_sequence_(
	std::string sequence,
	chemical::ResidueTypeSet const& residue_set,
	pose::Pose& pose
) {
	using namespace chemical;
	// clear all of the old data in the pose
	pose.clear();

	// setup the pose by appending the appropriate residues residues
	for ( Size seqpos = 1; seqpos <= sequence.length(); ++seqpos ) {
		char aa = sequence[seqpos-1]; // string indexing is zero-based!
		//		std::cerr<<  aa << " aminoacid requested" << std::endl;
		AA my_aa = aa_from_oneletter_code( aa );
		ResidueTypeCAPs const & rsd_type_list( residue_set.aa_map( my_aa ) );
		runtime_assert( rsd_type_list.size() );
		Size best_index = 1;
		ResidueType const & rsd_type( *(rsd_type_list[ best_index ]) );
		conformation::ResidueOP new_rsd( conformation::ResidueFactory::create_residue( rsd_type ) );
		if ( seqpos == 1 ) {
			pose.append_residue_by_jump( *new_rsd, 1 );
		} else {
			pose.append_residue_by_bond( *new_rsd, true );
		}
	} // for seqpos

	//	if ( pose.total_residue() > 1)  pose.conformation().insert_chain_ending( pose.total_residue() - 1 );		// probably not necessary

} // make_pose_match_sequence_

void Frame::fragment_as_pose(
	Size frag_num,
	pose::Pose & pose,
	chemical::ResidueTypeSetCAP restype_set ) const
{
	//	if (!my_static_pose_for_testing_ ) {
	//		std::string const pdbfile ( "protocols/abinitio/2GB3.pdb" );
	//my_static_pose_for_testing_=new pose::Pose;
	pose.clear();
	make_pose_from_sequence_( frag_list_[ frag_num ]->sequence(),
		*restype_set,
		pose );
		//		io::pdb::pose_from_pdb( *my_static_pose_for_testing_, pdbfile );
	fragment( frag_num ).apply( pose, 1, length() );
}

bool Frame::add_fragment( FragDataList new_frag ) {
	for ( FragDataList::const_iterator it=new_frag.begin(), eit=new_frag.end();
				it!=eit; ++it ) {
		bool success ( add_fragment( *it ) );
		if ( !success ) return false;
	}
	return true;
}

void Frame::show_classic( std::ostream &out ) const {
	using namespace ObjexxFCL::fmt;
	Size position = start();
	out << "position: " << RJ( 10, position) << " neighbors: " << RJ( 10, nr_frags() ) << std::endl << std::endl;
	for ( Size nr = 1; nr <= nr_frags(); nr ++ ) {
		fragment( nr ).show( out );
		out << std::endl;
	}
}

void Frame::show( std::ostream &out ) const {
// 	using namespace ObjexxFCL::fmt;
// 	out << "FRAME " << " " << RJ( 10, start() ) << RJ( 10, end() ) << std::endl;
	show_header( out );
	show_fragments( out );
}

void Frame::show_header( std::ostream &out ) const {
	using namespace ObjexxFCL::fmt;
	out << "FRAME " << " " << RJ( 3, start() ) << " " << RJ( 3, end() ) << std::endl;
}

void Frame::read( std::istream &in ) {
	//because of one FragData - multiple frames.. reading of fragments is outsourced to FragmentIO
	//	std::string tag;
	in >> start_ >> end_;
	nr_res_ = end_ - start_ + 1;
}

void Frame::show_fragments( std::ostream& out ) const {
	for ( Size nr = 1; nr <= nr_frags(); nr ++ ) {
		assert( fragment_ptr( nr ) );
		fragment( nr ).show( out, *this );
		out << std::endl << std::endl;
	}
}

bool Frame::merge( Frame const& other ) {
	if ( !is_mergeable( other) ) return false;
	//append fragdata
	//copy cached data
	Size insert_pos = frag_list_.size()+1;
	Size other_frag_num = 1;
	for ( FragDataList::const_iterator it = other.frag_list_.begin(),
					eit = other.frag_list_.end(); it!=eit; ++it ) {
		frag_list_.push_back( *it );
		clone_cache_data( other, other_frag_num++, insert_pos++ );
	}
	return true;
}

///@brief change frames residue numbers according to map
bool Frame::align( core::sequence::SequenceMapping const& map) {
	Size s( map[ start() ] );
  Size e( map[ end() ] );
	bool success = ( s>0 && e>0 );
	if ( is_continuous() && success ) {
		success = ( (e - s + 1 ) == length() );
		if ( !success ) tr.Trace << "failed to align frame " << start() << " " << end() << " different size after alignment" << std::endl;
		//check that all residues within frame can be mapped -- or is that an unwanted restriction ?
		for ( Size pos = start(); success && pos<=end(); pos ++ ) { // this should only happen if we hae insertion and deletion cancelling each other  ?
			success = map[ pos ] > 0;
			if ( !success ) tr.Trace << "failed to align frame " << start() << " " << end() << " due to pos " << pos << std::endl;
		}
	}
	if ( success ) {
		start_ = s;
		end_ = e;
	}
	return success;
}

FrameOP Frame::generate_sub_frame(  Size length, Size start /* = 1*/ ) {
	if ( !is_continuous() ) return NULL;
	FrameOP sub_frame( clone() ); //creates empty frame with same type
	sub_frame->start_ += start - 1;
	sub_frame->nr_res_ = length;
	sub_frame->end_ = sub_frame->start_ + length - 1;
	for ( FragDataList::const_iterator it = frag_list_.begin(), eit = frag_list_.end();
				it!=eit; ++it ) {
		sub_frame->add_fragment( (*it)->generate_sub_fragment( start, start + length - 1 ) );
	}
	runtime_assert( nr_frags() == sub_frame->nr_frags() );
	return sub_frame;
}

bool Frame::steal( pose::Pose const& pose) {
	runtime_assert( nr_frags() );
	FragDataOP new_frag = frag_list_.front()->clone();
	bool success ( new_frag->steal( pose, *this ) );
	if ( success ) {
		// check if first fragment was just an unitialized template...
		if ( frag_list_.front()->is_valid() ) {
			frag_list_.push_back( new_frag );
		} else {  // remove unitialized template and replace it with the new fragment
			tr.Trace << "remove unitialized template FragData and replace it with stolen one" << std::endl;
			runtime_assert( nr_frags() == 1 );
			frag_list_.clear();
			frag_list_.push_back( new_frag );
		}
	}
	return success;
}

void Frame::clear() {
	cache_.clear();
	if ( nr_frags() ) {
		FragDataOP frag_template = frag_list_.front()->clone();
		frag_template->set_valid( false ); //make it a template
		add_fragment( frag_template );
	}
}

}
}
