// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragment/picking/vall/util.cc
/// @brief  utility functions for interacting with VallLibrary
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <core/fragment/picking/vall/util.hh>

// package headers
#include <core/fragment/picking/FragmentLibraryManager.hh>
#include <core/fragment/picking/vall/VallLibrarian.hh>
#include <core/fragment/picking/vall/VallLibrary.hh>
#include <core/fragment/picking/vall/eval/IdentityEval.hh>
#include <core/fragment/picking/vall/gen/LengthGen.hh>


namespace core {
namespace fragment {
namespace picking {
namespace vall {


/// @brief pick fragments by default sec.struct IdentityScore
/// @param[in] ss secondary structure string of desired frag length
/// @param[in] top_n return the top 'n' fragments, default 200
/// @param[in] randomize add random noise within [0, 0.001) to
///  score to break up equivalent scores
/// @param[in] srfd_type The BBTorsionSRFD type to use.
core::fragment::FragDataList
pick_fragments_by_ss(
	std::string const & ss,
	core::Size const top_n,
	bool const randomize,
	BBTorsionSRFD const & srfd_type
)
{
	using eval::IdentityEval;
	using gen::LengthGen;

	VallLibrary const & library = FragmentLibraryManager::get_instance()->get_Vall();

	VallLibrarian librarian;
	librarian.add_fragment_gen( new LengthGen( ss.length() ) );
	librarian.add_fragment_eval( new IdentityEval( ss, 1.0, randomize ) );

	// catalog fragments
	librarian.catalog( library );

	// grab top fragments
	return librarian.top_fragments( top_n, srfd_type );
}


/// @brief pick fragments by default sec.struct IdentityScore
/// @param[in] ss secondary structure string of desired frag length
/// @param[in] aa amino acid string of same length as ss string
/// @param[in] top_n return the top 'n' fragments, default 200
/// @param[in] randomize add random noise within [0, 0.001) to
///  score to break up equivalent scores
/// @param[in] srfd_type The BBTorsionSRFD type to use.
core::fragment::FragDataList
pick_fragments_by_ss_plus_aa(
	std::string const & ss,
	std::string const & aa,
	core::Size const top_n,
	bool const randomize,
	BBTorsionSRFD const & srfd_type
)
{
	using eval::IdentityEval;
	using gen::LengthGen;

	VallLibrary const & library = FragmentLibraryManager::get_instance()->get_Vall();

	VallLibrarian librarian;
	librarian.add_fragment_gen( new LengthGen( ss.length() ) );
	librarian.add_fragment_eval( new IdentityEval( ss, aa, 1.0, 1.0, randomize ) );

	// catalog fragments
	librarian.catalog( library );

	// grab top fragments
	return librarian.top_fragments( top_n, srfd_type );
}


} // vall
} // picking
} // fragment
} // core
