// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/graph/Graph.hh
/// @brief  generic graph class header
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com)


// Unit Headers
#include <core/graph/DisjointSets.hh>

namespace core {
namespace graph {

DisjointSets::DisjointSets()
{
}

DisjointSets::DisjointSets( Size n_nodes )
:
	nodes_( n_nodes )
{
	for ( Size ii = 1; ii <= n_nodes; ++ii ) {
		nodes_[ ii ].parent = ii;
		nodes_[ ii ].rank = 0;
	}
}

Size DisjointSets::n_nodes() const
{
	return nodes_.size();
}

/// @details This implementation uses indices to objects in arrays instead of pointers
/// and so it relies on vector push-back methods (O(N)) instead of list push-back
/// methods (O(1)).  If enough people clamour, I'll go back and make this faster...
void
DisjointSets::ds_make_set()
{
	nodes_.resize( n_nodes() + 1 );
	nodes_[ n_nodes() ].parent = n_nodes();
	nodes_[ n_nodes() ].rank = 0;
}

/// @brief return the representative for a node
Size
DisjointSets::ds_find( Size node_id ) const
{
	if ( nodes_[ node_id ].parent != node_id ) {
		nodes_[ node_id ].parent = ds_find( nodes_[ node_id ].parent );
	}
	return nodes_[ node_id ].parent;
}

/// @brief make it so that two nodes end up in the same set
void
DisjointSets::ds_union( Size node1, Size node2 )
{
	Size parent_node1 = ds_find( node1 );
	Size parent_node2 = ds_find( node2 );
	if ( nodes_[ parent_node1 ].rank < nodes_[ parent_node2 ].rank ) {
		nodes_[ parent_node1 ].parent = parent_node2;
		++nodes_[ parent_node2 ].rank;
	} else if ( nodes_[ parent_node1 ].rank > nodes_[ parent_node2 ].rank ) {
		nodes_[ parent_node2 ].parent = parent_node1;
		++nodes_[ parent_node1 ].rank;
	} else if ( parent_node1 != parent_node2 ) {
		nodes_[ parent_node1 ].parent = parent_node2;
		++nodes_[ parent_node2 ].rank;
	}
}


/// @brief count the number of disjoint sets. O(N)
Size DisjointSets::n_disjoint_sets() const
{
	Size n_disjoint( 0 );
	for ( Size ii = 1; ii <= n_nodes(); ++ii ) {
		if ( nodes_[ ii ].parent == ii ) ++n_disjoint;
	}
	return n_disjoint;
}

/// @brief count the size of each disjoint set. O(N)
utility::vector1< Size >
DisjointSets::disjoint_set_sizes() const
{
	utility::vector1< Size > index_2_ds( n_nodes(), 0 );
	Size n_disjoint( 0 );
	for ( Size ii = 1; ii <= n_nodes(); ++ii ) {
		if ( nodes_[ ii ].parent == ii ) index_2_ds[ ii ] = ++n_disjoint;
	}

	utility::vector1< Size > ds_set_sizes( n_disjoint, 0 );
	for ( Size ii = 1; ii <= n_nodes(); ++ii ) {
		++ds_set_sizes[ index_2_ds[ ds_find( ii ) ] ];
	}
	return ds_set_sizes;
}

/// @brief return the nodes in the set containing the specified node.
utility::vector1< Size >
DisjointSets::nodes_in_set( Size node_id ) const
{
	utility::vector1< Size > nis;

	Size const root = ds_find( node_id );

	for ( Size i = 1, ie = n_nodes(); i <= ie; ++i ) {
		if ( root == ds_find( i ) ) {
			nis.push_back( i );
		}
	}

	return nis;
}

/// @brief return a map from the representative node of each set to
///  the list of nodes in their sets
std::map< Size, utility::vector1< Size > >
DisjointSets::sets() const
{
	std::map< Size, utility::vector1< Size > > r2s;

	for ( Size i = 1, ie = n_nodes(); i <= ie; ++i ) {
		// operator[] safe here
		r2s[ ds_find( i ) ].push_back( i );
	}

	return r2s;
}


}
}

