// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/grid/CartGrid.cc
///
/// @brief
/// @author Ian W. Davis


#include <core/grid/CartGrid.hh>

#include <numeric/xyzVector.hh>
#include <utility/exit.hh>
#include <utility/io/izstream.hh>
#include <ObjexxFCL/string.functions.hh>

#include <sstream>
#include <fstream>
#include <iostream>

namespace core {
namespace grid {


CartGrid::CartGrid():
	its_nX(0), its_nY(0), its_nZ(0),
	its_lX(0.0), its_lY(0.0), its_lZ(0.0),
	its_bX(0.0), its_bY(0.0), its_bZ(0.0),
	its_tX(0.0), its_tY(0.0), its_tZ(0.0),
	its_name("default"),
	its_npoints(0),
	bFullyOccupied(false),
	zones(NULL)
{
}


CartGrid::~CartGrid() {
	if (zones != NULL) {
		delete [] zones;
		zones = NULL;
	}
}


std::ostream & operator << (std::ostream & os, CartGrid const & mygrid) {
	os << mygrid.its_bX << " " << mygrid.its_bY << " " << mygrid.its_bZ << " | "
	<< mygrid.its_tX << " " << mygrid.its_tY << " " << mygrid.its_tZ;

	return os;
}


void CartGrid::setBase(core::Real x, core::Real y, core::Real z) {
	its_bX = x;
	its_bY = y;
	its_bZ = z;

	this->setTop();
}


void CartGrid::setDimensions(int nX, int nY, int nZ, core::Real lX, core::Real lY, core::Real lZ) {
	its_nX = nX;
	its_nY = nY;
	its_nZ = nZ;

	its_lX = lX;
	its_lY = lY;
	its_lZ = lZ;

	this->setTop();
}


void CartGrid::setTop() {
	its_tX = its_bX + its_nX*its_lX;
	its_tY = its_bY + its_nY*its_lY;
	its_tZ = its_bZ + its_nZ*its_lZ;
}


void CartGrid::set_name(std::string const & name) {
	its_name = name;
}


std::string CartGrid::get_name() const {
	return its_name;
}


int CartGrid::longestSide() const {
	int ls = this->its_nX;
	if (this->its_nY > ls) ls = this->its_nY;
	if (this->its_nZ > ls) ls = this->its_nZ;

	return ls;
}


bool CartGrid::equalDimensions(CartGrid const & rhs) const {
	if (its_nX != rhs.its_nX) return false;
	if (its_nY != rhs.its_nY) return false;
	if (its_nZ != rhs.its_nZ) return false;

	if (its_lX != rhs.its_lX) return false;
	if (its_lY != rhs.its_lY) return false;
	if (its_lZ != rhs.its_lZ) return false;

	return true;
}


bool CartGrid::equalBase(CartGrid const & rhs) const {
	if (std::abs(this->its_bX - rhs.its_bX) > 0.01) return false;
	if (std::abs(this->its_bY - rhs.its_bY) > 0.01) return false;
	if (std::abs(this->its_bZ - rhs.its_bZ) > 0.01) return false;
	return true;
}


bool CartGrid::is_in_grid(core::Real x, core::Real y, core::Real z) const {
	if (x < its_bX || x > its_tX) {
		return false;
	}
	if (y < its_bY || y > its_tY) {
		return false;
	}
	if (z < its_bZ || z > its_tZ) {
		return false;
	}

	return true;
}


bool CartGrid::setupZones() {
	if (zones != NULL) {
		delete zones;
		zones = NULL;
	}

	its_npoints = its_nX*its_nY*its_nZ;
	zones = new int[its_npoints];

	return true;
}


void CartGrid::translate(core::Real x, core::Real y, core::Real z) {
	its_bX += x;
	its_bY += y;
	its_bZ += z;
	this->setTop();
}


void CartGrid::setValue(int index, int value) {
	if (value == 0) {
		bFullyOccupied = false;
	}
	assert( index >= 0 && index < its_npoints );
	this->zones[index] = value;
}


bool CartGrid::setValue(int ix, int iy, int iz, int value) {
	if (value == 0) {
		bFullyOccupied = false;
	}

	int index = this->get_index(ix, iy, iz);
	assert( index >= 0 && index < its_npoints );
	this->zones[index] = value;
	return true;
}


bool CartGrid::setValue(core::Real fx, core::Real fy, core::Real fz, int value) {
	if (value == 0) {
		bFullyOccupied = false;
	}

	int ix = int((fx - its_bX)/its_lX);
	if (ix < 0 || ix >= its_nX) return 0;

	int iy = int((fy - its_bY)/its_lY);
	if (iy < 0 || iy >= its_nY) return 0;

	int iz = int((fz - its_bZ)/its_lZ);
	if (iz < 0 || iz >= its_nZ) return 0;

	int index = this->get_index(ix,iy,iz);
	assert( index >= 0 && index < its_npoints );
	this->zones[index] = value;
	return true;
}


int CartGrid::getValue(int index) const {
	if (index < 0 || index >= its_npoints) {
		return -1;
	} else {
		return this->zones[index];
	}
}


int CartGrid::getValue(int ix, int iy, int iz) const {
	int index = this->get_index(ix, iy, iz);
	return this->getValue(index);
}


int CartGrid::getValue(core::Real fx, core::Real fy, core::Real fz) const {
	// --- round down --- //
	int ix = int((fx - its_bX)/its_lX);
	if (ix < 0 || ix >= its_nX) return 0;

	int iy = int((fy - its_bY)/its_lY);
	if (iy < 0 || iy >= its_nY) return 0;

	int iz = int((fz - its_bZ)/its_lZ);
	if (iz < 0 || iz >= its_nZ) return 0;

	int index = ix*(its_nY*its_nZ) + iy*(its_nZ) + iz;
	return this->zones[index];
}


int CartGrid::get_index(int ix, int iy, int iz) const {
	int index;
	index = ix*(its_nY*its_nZ) + iy*(its_nZ) + iz;
	return index;
}


void CartGrid::zero() {
	for (int i=0; i < its_npoints; i++) {
		this->zones[i] = 0;
	}
}


void CartGrid::setFullOccupied(int value) {
	for (int i=0; i < its_npoints; i++) {
		this->zones[i] = value;
	}
	bFullyOccupied = true;
}


void CartGrid::clone(CartGrid & pCopy) const {
	// copy over gross information
	pCopy.its_nX = this->its_nX;
	pCopy.its_nY = this->its_nY;
	pCopy.its_nZ = this->its_nZ;

	pCopy.its_lX = this->its_lX;
	pCopy.its_lY = this->its_lY;
	pCopy.its_lZ = this->its_lZ;

	pCopy.its_bX = this->its_bX;
	pCopy.its_bY = this->its_bY;
	pCopy.its_bZ = this->its_bZ;

	pCopy.its_tX = this->its_tX;
	pCopy.its_tY = this->its_tY;
	pCopy.its_tZ = this->its_tZ;

	pCopy.its_name = this->its_name;

	pCopy.setupZones();

	for (int i=0; i < this->its_npoints; i++) {
		pCopy.zones[i] = this->zones[i];
	}
}


void CartGrid::reset_boundaries() {
	int minx = this->its_nX;
	int miny = this->its_nY;
	int minz = this->its_nZ;

	int maxx = 0;
	int maxy = 0;
	int maxz = 0;

	for (int ix=0; ix < this->its_nX; ix++) {
		for (int iy=0; iy < this->its_nY; iy++) {
			for (int iz=0; iz < this->its_nZ; iz++) {
				if (this->getValue(ix,iy,iz)) {
					minx = std::min(minx,ix);
					miny = std::min(miny,iy);
					minz = std::min(minz,iz);

					maxx = std::max(maxx,ix);
					maxy = std::max(maxy,iy);
					maxz = std::max(maxz,iz);
				}
			}
		}
	}

	if (maxx*maxy*maxz == 0) {
		return;   // no change
	}

	CartGrid tmpgrid;
	this->clone(tmpgrid);

	int nx = maxx - minx + 1;
	int ny = maxy - miny + 1;
	int nz = maxz - minz + 1;

	this->its_nX = nx;
	this->its_nY = ny;
	this->its_nZ = nz;

	this->its_bX += minx*this->its_lX;
	this->its_bY += miny*this->its_lY;
	this->its_bZ += minz*this->its_lZ;

	this->setupZones();

	int index2=0;
	int value = 0;
	for (int i=0; i < nx; i++) {
		for (int j=0; j < ny; j++) {
			for (int k=0; k < nz; k++) {
				index2 = tmpgrid.get_index(minx+i, miny+j, minz+k);
				value = tmpgrid.getValue(index2);
				this->setValue(i,j,k,value);
			}
		}
	}
}



void CartGrid::fluff(CartGridOP input, CartGridOP original, int amount=6) {
	int istart, iend, jstart, jend, kstart, kend;
	for (int i=0; i < input->its_nX; i++) {
		for (int j=0; j < input->its_nY; j++) {
			for (int k=0; k < input->its_nZ; k++) {
				if (input->getValue(i,j,k) != 0) {
					istart = std::max(0, (i-amount));
					iend   = std::min(input->its_nX, (i+amount));

					jstart = std::max(0, (j-amount));
					jend   = std::min(input->its_nY, (j+amount));

					kstart = std::max(0, (k-amount));
					kend   = std::min(input->its_nZ, (k+amount));

					for (int ii=istart; ii < iend; ii++) {
						for (int jj=jstart; jj < jend; jj++) {
							for (int kk=kstart; kk < kend; kk++) {
								if (original->getValue(ii,jj,kk) != 0) {
									this->setValue(ii,jj,kk,1);
								}
							}
						}
					}
				}
			}
		}
	}
}


void CartGrid::read(std::string const & filename) {
	//std::ifstream file;
	utility::io::izstream file;
	std::istringstream line_stream;

	file.open(filename.c_str());

	if (!file) {
		std::cout << "read_gridfile - unable to open gridfile:" << filename << std::endl;
		std::exit( EXIT_FAILURE );
	}

	std::string line;
	std::string keyword;
	std::string name;
	core::Real bx=0.0, by=0.0, bz=0.0;
	core::Real lx=0.0, ly=0.0, lz=0.0;
	int occupied=0;
	int nx=0, ny=0, nz=0;
	int ix=0, iy=0, iz=0;

	while(file) {
		getline(file, line);


		if (ObjexxFCL::is_blank(line)) {
			ix++;
			iy=0;
			iz=0;
			continue;
		}

		line_stream.clear();
		line_stream.str(line);
		line_stream.seekg( std::ios::beg );

		line_stream >> keyword;
		if (keyword == "NAME:") {
			line_stream >> name;
			this->set_name(name);
		}
		else if (keyword == "BASE:") {
			line_stream >> bx >> by >> bz;
			this->setBase(bx, by, bz);
		} else if (keyword == "SIZE:") {
			line_stream >> nx >> ny >> nz;
		} else if (keyword == "LENGTH:") {
			line_stream >> lx >> ly >> lz;
			this->setDimensions(nx, ny, nz, lx, ly, lz);
			this->setupZones();
		} else {
			line_stream.seekg( std::ios::beg );
			iz = 0;
			for (int iz=0; iz < nz; iz++) {
				line_stream >> occupied;
				this->setValue(ix,iy,iz,occupied);
			}
			iy++;
		}
	}

	file.close();
}


void CartGrid::write(std::string const & filename) const {
	std::ofstream file;
	file.open(filename.c_str());
	file << "NAME: " << this->get_name() << std::endl;
	file << "BASE: " << this->its_bX << " " << this->its_bY << " " << this->its_bZ << std::endl;
	file << "SIZE: " << this->its_nX << " " << this->its_nY << " " << this->its_nZ << std::endl;
	file << "LENGTH: " << this->its_lX << " " << this->its_lY << " " << this->its_lZ << std::endl;

	for (int i=0; i < this->its_nX; i++) {
		for (int j=0; j < this->its_nY; j++) {
			for (int k=0; k < this->its_nZ; k++) {
				file << this->getValue(i,j,k) << " ";
			}
			file << std::endl;
		}
		file << std::endl;
	}
	file.close();
}


bool CartGrid::isFullyOccupied() const {
	return bFullyOccupied;
}


bool CartGrid::isEmpty() const {
	for (int i=0; i < its_npoints; i++) {
		if (this->zones[i] != 0) {
			return false;
		}
	}
	return true;
}


void CartGrid::sum(utility::vector0<CartGridOP> const & list_grids) {
	int ngrids = static_cast<int>(list_grids.size());

	this->zero();
	int curr_value, new_value;
	for (int i=0; i < ngrids; i++) {
		if (!this->equalDimensions(*(list_grids[i]))) {
			return;
		}
		if (!this->equalBase(*(list_grids[i]))) {
			return;
		}

		for (int j=0; j < this->its_npoints; j++) {
			curr_value = this->getValue(j);
			new_value = list_grids[i]->getValue(j);
			curr_value += new_value;
			this->setValue(j,curr_value);
		}
	}
}


void CartGrid::expand(int expansion) {
	this->its_bX -= this->its_lX*core::Real(expansion);
	this->its_bY -= this->its_lY*core::Real(expansion);
	this->its_bZ -= this->its_lZ*core::Real(expansion);

	this->its_nX += 2*expansion;
	this->its_nY += 2*expansion;
	this->its_nZ += 2*expansion;

	this->setupZones();
}



void CartGrid::split(int nsplits, int igrid, core::Real pad, CartGridOP pGrid) {
	int tsplits = nsplits*nsplits*nsplits;
	if (igrid < 0 || igrid >= tsplits) {
		utility_exit_with_message("accessing split out of bounds");
	}

	int split_x = int(core::Real(this->its_nX)/core::Real(nsplits)) + 1;
	int split_y = int(core::Real(this->its_nY)/core::Real(nsplits)) + 1;
	int split_z = int(core::Real(this->its_nZ)/core::Real(nsplits)) + 1;

	int iz = int(igrid / (nsplits*nsplits));
	int rem = igrid - (iz*nsplits*nsplits);
	int iy = int(rem/nsplits);
	int ix = rem - (iy*nsplits);

	int padx = int((pad/its_lX)-0.1);
	int pady = int((pad/its_lY)-0.1);
	int padz = int((pad/its_lZ)-0.1);

	int leftx = padx;
	int rightx = padx;
	int lefty = pady;
	int righty = pady;
	int leftz = padz;
	int rightz = padz;

	if (ix == 0) { leftx = 0; }
	if (iy == 0) { lefty = 0; }
	if (iz == 0) { leftz = 0; }

	if (ix == nsplits-1) { rightx = 0; }
	if (iy == nsplits-1) { righty = 0; }
	if (iz == nsplits-1) { rightz = 0; }

	pGrid->its_nX = split_x+leftx+rightx;
	pGrid->its_nY = split_y+lefty+righty;
	pGrid->its_nZ = split_z+leftz+rightz;

	core::Real bx = this->its_bX + ix*split_x*(this->its_lX) - padx*(this->its_lX);
	core::Real by = this->its_bY + iy*split_y*(this->its_lY) - pady*(this->its_lY);
	core::Real bz = this->its_bZ + iz*split_z*(this->its_lZ) - padz*(this->its_lZ);

	pGrid->its_bX = std::max(this->its_bX, bx);
	pGrid->its_bY = std::max(this->its_bY, by);
	pGrid->its_bZ = std::max(this->its_bZ, bz);

	pGrid->its_lX = this->its_lX;
	pGrid->its_lY = this->its_lY;
	pGrid->its_lZ = this->its_lZ;

	// copy over grid data
	pGrid->setDimensions(pGrid->its_nX,pGrid->its_nY,pGrid->its_nZ,pGrid->its_lX,pGrid->its_lY,pGrid->its_lZ);
	pGrid->setupZones();
	pGrid->zero();

	int xstart = std::max(0,ix*split_x-padx);
	int ystart = std::max(0,iy*split_y-pady);
	int zstart = std::max(0,iz*split_z-padz);

	for (int i=0; i < pGrid->its_nX; i++) {
		for (int j=0; j < pGrid->its_nY; j++) {
	 	for (int k=0; k < pGrid->its_nZ; k++) {
			int value = this->getValue(xstart+i,ystart+j,zstart+k);
			if (value == -1) value = 0;
			pGrid->setValue(i,j,k,value);
		}
	 }
	}
}


core::Vector CartGrid::getBase() const
{
	return core::Vector(its_bX, its_bY, its_bZ);
}


core::Vector CartGrid::getTop() const
{
	return core::Vector(its_tX, its_tY, its_tZ);
}


void CartGrid::getNumberOfPoints(int & x, int & y, int & z) const
{
	x = its_nX;
	y = its_nY;
	z = its_nZ;
}


// All of these functions could be inlined for speed if it became necessary,
// but I think that means they can no longer be virtual.

CartGrid::GridPt CartGrid::gridpt(Vector const & coords) const
{
	// Round down.
	// Just casting to int will give wrong values for points outside the grid.
	return GridPt(
		(int) std::floor((coords.x() - its_bX)/its_lX),
		(int) std::floor((coords.y() - its_bY)/its_lY),
		(int) std::floor((coords.z() - its_bZ)/its_lZ)
	);
}


Vector CartGrid::coords(GridPt const & gridpt) const
{
	return Vector(
		(its_bX + (gridpt.x() + 0.5)*its_lX),
		(its_bY + (gridpt.y() + 0.5)*its_lY),
		(its_bZ + (gridpt.z() + 0.5)*its_lZ)
	);
}


int CartGrid::getValue(GridPt const & gridpt) const
{ return getValue(gridpt.x(), gridpt.y(), gridpt.z()); }

int CartGrid::getValue(Vector const & coords) const
{ return getValue(coords.x(), coords.y(), coords.z()); }

void CartGrid::setValue(GridPt const & gridpt, int value)
{ setValue(gridpt.x(), gridpt.y(), gridpt.z(), value); }

void CartGrid::setValue(Vector const & coords, int value)
{ setValue(coords.x(), coords.y(), coords.z(), value); }


/// @details This format was choosen because it's space-efficient for small
/// integer values (such as are typically stored in grids) and PyMOL can read it.
/// Typical extension is .brix or .omap
void CartGrid::write_to_BRIX(std::string const & filename)
{
	std::ofstream out( filename.c_str() );
	write_to_BRIX(out);
	out.close();
}

void CartGrid::write_to_BRIX(std::ostream & out)
{
	int const plus = 127; // to convert values to unsigned bytes
	// Crystallographic maps always expect one grid point to be at the origin.
	// This may shift our map slightly...
	GridPt origin = gridpt(Vector(0.,0.,0.));
	out << ":-)"; // magic number
	out << " Origin " << -origin.x() << ' ' << -origin.y() << ' ' << -origin.z() ; // starting indices for grid points actually present
	out << " Extent " << its_nX << ' ' << its_nY << ' ' << its_nZ ; // number of grid points in each dimension
	out << " Grid " << its_nX << ' ' << its_nY << ' ' << its_nZ ; // number of grid points in one unit cell
	out << " Cell " << its_nX*its_lX << ' ' << its_nY*its_lY << ' ' << its_nZ*its_lZ << " 90.0 90.0 90.0"; // dimensions of unit cell
	out << " Prod 1 Plus " << plus << " Sigma 1\f";
	// Now pad with spaces until we've writen 512 characters:
	for(long i = out.tellp(); i < 512; ++i) out << ' ';
	// Data stored in (padded) 8x8x8 blocks with X fast, Y medium, Z slow
	typedef unsigned char ubyte;
	for(int zz = 0; zz < its_nZ; zz += 8) {
		for(int yy = 0; yy < its_nY; yy += 8) {
			for(int xx = 0; xx < its_nX; xx += 8) {
				for(int z = zz, z_end = zz+8; z < z_end; ++z) {
					for(int y = yy, y_end = yy+8; y < y_end; ++y) {
						for(int x = xx, x_end = xx+8; x < x_end; ++x) {
							if( x < its_nX && y < its_nY && z < its_nZ ) out << ubyte( getValue(x, y, z) + plus );
							else out << ubyte( 0 + plus );
						}
					}
				}
			}
		}
	}
}


} // namespace grid
} // namespace core
