// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/grid/CartGrid.hh
///
/// @brief  port of Andrew Wollacott's cart_grid from Rosetta++
/// @author Ian W. Davis
/// @author Andrew Wollacott


#ifndef INCLUDED_core_grid_CartGrid_HH
#define INCLUDED_core_grid_CartGrid_HH

#include <core/grid/CartGrid.fwd.hh>
#include <utility/pointer/ReferenceCount.hh>

#include <core/types.hh>
#include <numeric/xyzVector.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <utility/vector0.hh>

#include <ostream>
#include <string>

namespace core {
namespace grid {


///@brief abstract grid object that can be associated with a given protein structure
///
///@details
/// a grid object is centered at a given center in the current coordinate frame
/// (i.e. relative to the one implicitly used in fullcoord arrays)
/// Several grids may be attached to a given protein structure. In addition to its
/// center, a grid is also defined by its boundaries (4 coordinates, so the grid
/// is currently limited to be rectangular), and an increment in each of the 3
/// direction.
///
/// The intention is to eventually move some of this functionality to a superclass,
/// which would probably be templated to handle other kinds of indices than three reals.
///
/// To fully initialize a grid, you must call setBase(), then setDimensions(), then setupZones().
///
class CartGrid : public utility::pointer::ReferenceCount
{
public:
	typedef numeric::xyzVector< int > GridPt; //< for i,j,k grid indices
	// use core::Vector for xyz coordinates

	CartGrid();
	virtual ~CartGrid();

	// functions from the original "grid" base class:
	friend std::ostream & operator << (std::ostream &, CartGrid const &);
	/// @brief Set the low-end boundaries of the grid.
	virtual void          setBase(core::Real bx, core::Real by, core::Real bz);
	/// @brief Set the total extent of the grid, and implicitly, the high-end boundaries.
	virtual void          setDimensions(int, int, int, core::Real, core::Real, core::Real);
	/// @brief Maximum number of grid points on any side (not actually "longest" side)
	virtual int           longestSide() const;
	/// @brief Set identifying label for this grid
	virtual void          set_name(std::string const & name);
	/// @brief Get identifying label for this grid
	virtual std::string   get_name() const;

	/// @brief (Re-)Allocate enough memory to store all grid points.
	virtual bool   setupZones();
	/// @brief Are the given coordinates within the bounds of the grid?
	virtual bool   is_in_grid( core::Real, core::Real, core::Real ) const;
	/// @brief Do these two grids have the same number of grid points and the same spacing between them?
	virtual bool   equalDimensions(CartGrid const &) const;
	/// @brief Do these two grids have the same base coordinates, to within a small tolerance?
	virtual bool   equalBase(CartGrid const &) const;
	/// @brief Shift the location of the grid in space by adding the specified amounts.
	virtual void   translate( core::Real, core::Real, core::Real);
	/// @brief Set the grid value for the cell containg these coordinates.
	virtual bool   setValue(core::Real, core::Real, core::Real, int);
	/// @brief Set the grid value for the cell with these indices.
	virtual bool   setValue(int, int, int, int);
	/// @brief Get the grid value for the cell with these indices.
	virtual int    getValue(int, int, int) const;
	/// @brief Get the grid value for the cell containg these coordinates.
	virtual int    getValue(core::Real, core::Real, core::Real) const;
	/// @brief Store 0 in all grid positions.
	virtual void   zero();
	/// @brief Store the given value in all grid positions, and flag it as fully occupied.
	virtual void   setFullOccupied(int value=1);
	/// @brief Copy information from this grid into the supplied one.
	virtual void   clone(CartGrid &) const;
	/// @brief For successive trimming of grid during enzyme design "matching" step.
	virtual void   fluff(CartGridOP, CartGridOP, int);
	/// @brief Shrink the grid to encompass only non-zero points.
	virtual void   reset_boundaries();
	/// @brief Set this grid's value to the sum of those in the other grids.
	/// Silently ignore grids with different sizes / placements.
	virtual void   sum(utility::vector0<CartGridOP> const &);
	/// @brief Extend this grid by the specified number of cells, in all directions, wiping out all existing values.
	virtual void   expand(int);
	/// @brief Read a grid from the named file.
	virtual void   read(std::string const &);
	/// @brief Write a grid to the named file.
	virtual void   write(std::string const &) const;
	/// @brief False if any grid positions are zero.
	/// True if all grid positions are non-zero because they were set by setFullOccupied().
	/// Very fast because it just checks a boolean flag.
	virtual bool   isFullyOccupied() const;
	/// @brief True iff all grid positions are zero.  Slow because it loops over all grid positions.
	virtual bool   isEmpty() const;
	/// @brief Copy a sub-region of this grid into the specified grid.
	virtual void   split(int nsplits, int igrid, core::Real pad, CartGridOP);

	/// @brief Return the minimum X,Y,Z boundary coordinates.
	virtual core::Vector getBase() const;
	/// @brief Return the maximum X,Y,Z boundary coordinates.
	virtual core::Vector getTop() const;
	/// @brief Return the number of grid points in each dimension
	virtual void getNumberOfPoints(int & x, int & y, int & z) const;

	/// @brief Convert XYZ to grid indices.  May return out-of-bounds indices.
	virtual GridPt gridpt(Vector const & coords) const;
	/// @brief Convert grid indices to box center XYZ.  Works OK for out-of-bounds indices.
	virtual Vector coords(GridPt const & gridpt) const;
	/// @brief Convenience function
	virtual int getValue(GridPt const & gridpt) const;
	/// @brief Convenience function
	virtual int getValue(Vector const & coords) const;
	/// @brief Convenience function
	virtual void setValue(GridPt const & gridpt, int value);
	/// @brief Convenience function
	virtual void setValue(Vector const & coords, int value);

	/// @brief Save the grid as an electron density map, in new-style "O" format (BRIX).
	virtual void write_to_BRIX(std::string const & filename);
	virtual void write_to_BRIX(std::ostream & out);

private:   // functions

	/// @brief Set the upper boundaries of the grid, based on its lower boundaries and extent.
	virtual void   setTop();
	/// @brief Return the linear lookup index from X, Y, and Z indices.
	virtual int    get_index(int, int, int) const;
	/// @brief Set the grid value for the cell with this absolute index in the storage array (why was this public?)
	virtual void   setValue(int, int);
	/// @brief Get the grid value for the cell with this absolute index in the storage array (why was this public?)
	virtual int    getValue(int) const;


private:
	int         its_nX, its_nY, its_nZ;   // number of grid points
	core::Real  its_lX, its_lY, its_lZ;   // length of grid points
	core::Real  its_bX, its_bY, its_bZ;   // coordinates of base (bottom left limit)
	core::Real  its_tX, its_tY, its_tZ;   // coordinates of top (top right limit)
	std::string its_name;
	int         its_npoints;              // length of zones (its_nX*its_nY*its_nZ)
	bool        bFullyOccupied;           // true if all values are non-zero (but not reliably so)
	int*        zones;

}; // CartGrid


} // namespace grid
} // namespace core

#endif // INCLUDED_core_grid_CartGrid_HH
