// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/init.cc
/// @brief  options system initialization routines
/// @author Sergey Lyskov
///

#ifdef USEMPI
#include <mpi.h> // Must go first
#include <core/util/TracerToFile.hh>
#endif

// Unit headers
#include <core/init.hh>

// Project Headers
#include <core/svn_version.hh>
#include <core/types.hh>
#include <core/options/option.hh>
#include <utility/basic_sys_util.hh>
#include <stdlib.h>
#include <core/util/Tracer.hh>

// Classes in core that must register with factorys
#include <core/scoring/carbon_hbonds/CarbonHBondEnergyCreator.hh>
#include <core/scoring/constraints/ConstraintsEnergyCreator.hh>
#include <core/scoring/custom_pair_distance/FullatomCustomPairDistanceEnergyCreator.hh>
#include <core/scoring/disulfides/CentroidDisulfideEnergyCreator.hh>
#include <core/scoring/disulfides/FullatomDisulfideEnergyCreator.hh>
#include <core/scoring/electron_density/ElecDensCenEnergyCreator.hh>
#include <core/scoring/electron_density/ElecDensAllAtomCenEnergyCreator.hh>
#include <core/scoring/electron_density/ElecDensEnergyCreator.hh>
#include <core/scoring/etable/CoarseEtableEnergyCreator.hh>
#include <core/scoring/etable/EtableEnergyCreator.hh>
#include <core/scoring/geometric_solvation/ExactOccludedHbondSolEnergyCreator.hh>
#include <core/scoring/geometric_solvation/GeometricSolEnergyCreator.hh>
#include <core/scoring/geometric_solvation/OccludedHbondSolEnergyCreator.hh>
#include <core/scoring/hackelec/HackElecEnergyCreator.hh>
#include <core/scoring/hackelec/RNAHackElecEnergyCreator.hh>
#include <core/scoring/hbonds/HBondEnergyCreator.hh>
#include <core/scoring/methods/EnergyMethodRegisterer.hh>
#include <core/scoring/methods/CenPairEnergyCreator.hh>
#include <core/scoring/methods/ChainbreakEnergyCreator.hh>
#include <core/scoring/methods/ContactOrderEnergyCreator.hh>
#include <core/scoring/methods/DNA_BaseEnergyCreator.hh>
#include <core/scoring/methods/DirectReadoutEnergyCreator.hh>
#include <core/scoring/methods/DistanceChainbreakEnergyCreator.hh>
#include <core/scoring/methods/DunbrackEnergyCreator.hh>
#include <core/scoring/methods/EnvEnergyCreator.hh>
#include <core/scoring/methods/EnvSmoothEnergyCreator.hh>
#include <core/scoring/methods/GenBornEnergyCreator.hh>
#include <core/scoring/methods/HybridVDW_EnergyCreator.hh>
#include <core/scoring/methods/InterchainEnvEnergyCreator.hh>
#include <core/scoring/methods/InterchainPairEnergyCreator.hh>
#include <core/scoring/methods/LK_CosThetaEnergyCreator.hh>
#include <core/scoring/methods/LK_hackCreator.hh>
#include <core/scoring/methods/LinearChainbreakEnergyCreator.hh>
#include <core/scoring/methods/MMBondAngleEnergyCreator.hh>
#include <core/scoring/methods/MMTorsionEnergyCreator.hh>
#include <core/scoring/methods/MembraneCbetaEnergyCreator.hh>
#include <core/scoring/methods/MembraneCenPairEnergyCreator.hh>
#include <core/scoring/methods/MembraneEnvEnergyCreator.hh>
#include <core/scoring/methods/MembraneEnvPenaltiesCreator.hh>
#include <core/scoring/methods/MembraneLipoCreator.hh>
#include <core/scoring/methods/OmegaTetherEnergyCreator.hh>
#include <core/scoring/methods/PCS/PseudocontactShiftEnergyCreator.hh>
#include <core/scoring/methods/P_AA_EnergyCreator.hh>
#include <core/scoring/methods/P_AA_pp_EnergyCreator.hh>
#include <core/scoring/methods/PackStatEnergyCreator.hh>
#include <core/scoring/methods/PairEnergyCreator.hh>
#include <core/scoring/methods/PeptideBondEnergyCreator.hh>
#include <core/scoring/methods/ProClosureEnergyCreator.hh>
#include <core/scoring/methods/RG_Energy_FastCreator.hh>
#include <core/scoring/methods/RMS_EnergyCreator.hh>
#include <core/scoring/methods/RamachandranEnergy2BCreator.hh>
#include <core/scoring/methods/RamachandranEnergyCreator.hh>
#include <core/scoring/methods/ReferenceEnergyCreator.hh>
#include <core/scoring/methods/ResidualDipolarCouplingEnergyCreator.hh>
#include <core/scoring/methods/ResidualDipolarCouplingEnergy_RohlCreator.hh>
#include <core/scoring/methods/SecondaryStructureEnergyCreator.hh>
#include <core/scoring/methods/SuckerEnergyCreator.hh>
#include <core/scoring/methods/SurfaceEnergyCreator.hh>
#include <core/scoring/methods/UnfoldedStateEnergyCreator.hh>
#include <core/scoring/methods/VDW_EnergyCreator.hh>
#include <core/scoring/methods/WaterAdductHBondEnergyCreator.hh>
#include <core/scoring/methods/WaterAdductIntraEnergyCreator.hh>
#include <core/scoring/NV/NVscoreCreator.hh>
#include <core/scoring/packing/HolesEnergyCreator.hh>
#include <core/scoring/packing/SurfVolEnergyCreator.hh>
#include <core/scoring/rna/RG_Energy_RNACreator.hh>
#include <core/scoring/rna/RNA_FullAtomStackingEnergyCreator.hh>
#include <core/scoring/rna/RNA_LJ_BaseEnergyCreator.hh>
#include <core/scoring/rna/RNA_PairwiseLowResolutionEnergyCreator.hh>
#include <core/scoring/rna/RNA_TorsionEnergyCreator.hh>
#include <core/scoring/rna/RNA_VDW_EnergyCreator.hh>
#include <core/scoring/symE/symECreator.hh>

#include <core/options/keys/in.OptionKeys.gen.hh>
#include <cstring>


using core::util::T;
using core::util::Error;
using core::util::Warning;

// STL headers
#include <ctime>
#include <fstream>
#include <string>


// option key includes

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>


namespace core {

/// The following global varialbles force the linker to always include
/// the EnergyMethodCreator files to be included in staticly linked
/// executables.  These variables will be initialized before main()
/// begins during the "dynamic initialization" phase of loading.
/// During this time, the Registrotor classes will register their templated
/// EnergyMethods with the ScoringManager before main() begins.

using namespace scoring::methods;

static EnergyMethodRegistrator< scoring::NV::NVscoreCreator > NVscoreCreator_registrator;
static EnergyMethodRegistrator< scoring::carbon_hbonds::CarbonHBondEnergyCreator > CarbonHBondEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::constraints::ConstraintsEnergyCreator > ConstraintsEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::custom_pair_distance::FullatomCustomPairDistanceEnergyCreator > FullatomCustomPairDistanceEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::disulfides::CentroidDisulfideEnergyCreator > CentroidDisulfideEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::disulfides::FullatomDisulfideEnergyCreator > FullatomDisulfideEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::electron_density::ElecDensCenEnergyCreator > ElecDensCenEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::electron_density::ElecDensAllAtomCenEnergyCreator > ElecDensAllAtomCenEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::electron_density::ElecDensEnergyCreator > ElecDensEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::etable::CoarseEtableEnergyCreator > CoarseEtableEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::etable::EtableEnergyCreator > EtableEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::geometric_solvation::ExactOccludedHbondSolEnergyCreator > ExactOccludedHbondSolEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::geometric_solvation::GeometricSolEnergyCreator > GeometricSolEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::geometric_solvation::OccludedHbondSolEnergyCreator > OccludedHbondSolEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::hackelec::HackElecEnergyCreator > HackElecEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::hackelec::RNAHackElecEnergyCreator > RNAHackElecEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::hbonds::HBondEnergyCreator > HBondEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::ChainbreakEnergyCreator > ChainbreakEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::CenPairEnergyCreator > CenPairEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::ContactOrderEnergyCreator > ContactOrderEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::DNA_BaseEnergyCreator > DNA_BaseEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::DirectReadoutEnergyCreator > DirectReadoutEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::DistanceChainbreakEnergyCreator > DistanceChainbreakEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::DunbrackEnergyCreator > DunbrackEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::EnvEnergyCreator > EnvEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::EnvSmoothEnergyCreator > EnvSmoothEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::GenBornEnergyCreator > GenBornEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::HybridVDW_EnergyCreator > HybridVDW_EnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::InterchainEnvEnergyCreator > InterchainEnvEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::InterchainPairEnergyCreator > InterchainPairEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::LK_CosThetaEnergyCreator > LK_CosThetaEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::LK_hackCreator > LK_hackCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::LinearChainbreakEnergyCreator > LinearChainbreakEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::MMBondAngleEnergyCreator > MMBondAngleEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::MMTorsionEnergyCreator > MMTorsionEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::MembraneCbetaEnergyCreator > MembraneCbetaEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::MembraneCenPairEnergyCreator > MembraneCenPairEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::MembraneEnvEnergyCreator > MembraneEnvEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::MembraneEnvPenaltiesCreator > MembraneEnvPenaltiesCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::MembraneLipoCreator > MembraneLipoCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::OmegaTetherEnergyCreator > OmegaTetherEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::PCS::PseudocontactShiftEnergyCreator > PseudoconstactShiftEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::P_AA_EnergyCreator > P_AA_EnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::P_AA_pp_EnergyCreator > P_AA_pp_EnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::PackStatEnergyCreator > PackStatEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::PairEnergyCreator > PairEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::PeptideBondEnergyCreator > PeptideBondEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::ProClosureEnergyCreator > ProClosureEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::RG_Energy_FastCreator > RG_Energy_FastCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::RMS_EnergyCreator > RMS_EnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::RamachandranEnergy2BCreator > RamachandranEnergy2BCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::RamachandranEnergyCreator > RamachandranEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::ReferenceEnergyCreator > ReferenceEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::ResidualDipolarCouplingEnergyCreator > ResidualDipolarCouplingEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::ResidualDipolarCouplingEnergy_RohlCreator > ResidualDipolarCouplingEnergy_RohlCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::SecondaryStructureEnergyCreator > SecondaryStructureEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::SuckerEnergyCreator > SuckerEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::SurfaceEnergyCreator > SurfaceEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::UnfoldedStateEnergyCreator > UnfoldedStateEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::VDW_EnergyCreator > VDW_EnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::WaterAdductHBondEnergyCreator > WaterAdductHBondEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::methods::WaterAdductIntraEnergyCreator > WaterAdductIntraEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::packing::HolesEnergyCreator > HolesEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::packing::SurfVolEnergyCreator > SurfVolEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::rna::RG_Energy_RNACreator > RG_Energy_RNACreator_registrator;
static EnergyMethodRegistrator< scoring::rna::RNA_FullAtomStackingEnergyCreator > RNA_FullAtomStackingEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::rna::RNA_LJ_BaseEnergyCreator > RNA_LJ_BaseEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::rna::RNA_PairwiseLowResolutionEnergyCreator > RNA_PairwiseLowResolutionEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::rna::RNA_TorsionEnergyCreator > RNA_TorsionEnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::rna::RNA_VDW_EnergyCreator > RNA_VDW_EnergyCreator_registrator;
static EnergyMethodRegistrator< scoring::symE::symECreator > symECreator_registrator;


static core::util::Tracer TR("core.init");

using namespace numeric::random;

/// @brief Init basic core systems: options system, random system.
void init(int argc, char * argv [])
{
	using namespace options::OptionKeys;

#ifdef USEMPI
	{ // scope
	int already_initialized( 0 );
	MPI_Initialized( & already_initialized );
	if ( already_initialized == 0 ) MPI_Init(&argc, &argv);
	}
#endif

#ifdef BOINC
	std::cerr << "Initializing options.... ok " << std::endl;std::cerr.flush();
#endif

	// initialize options
	options::initialize().load( argc, argv, false /* no "free" cmd line args (just discarded anyway) */ );
#ifdef BOINC
	std::cerr << "Loaded options.... ok " << std::endl;std::cerr.flush();
#endif
	options::process();
#ifdef BOINC
	std::cerr << "Processed options.... ok " << std::endl; std::cerr.flush();
#endif

	// Set option system global
	options::OptionCollection::set_show_accessed_options_flag( core::options::option[ out::show_accessed_options ].value() );

	// set Tracer options
	core::util::TracerOptions & TO( core::util::Tracer::tracer_options() );
	if( core::options::option[ out::mute ].active() )   TO.muted = core::options::option[ out::mute ]();
	if( core::options::option[ out::mute_warning ].active() )   TO.muted_warning = core::options::option[ out::mute_warning ]();
	if( core::options::option[ out::mute_info ].active() )   TO.muted_info = core::options::option[ out::mute_info ]();
	if( core::options::option[ out::mute_debug ].active() )   TO.muted_debug = core::options::option[ out::mute_debug ]();

	if( core::options::option[ out::unmute_error ].active() )   TO.unmuted_error = core::options::option[ out::unmute_error ]();
	if( core::options::option[ out::unmute_warning ].active() )   TO.unmuted_warning = core::options::option[ out::unmute_warning ]();
	if( core::options::option[ out::unmute_info ].active() )   TO.unmuted_info = core::options::option[ out::unmute_info ]();
	if( core::options::option[ out::unmute_debug ].active() )   TO.unmuted_debug = core::options::option[ out::unmute_debug ]();

	if( core::options::option[ out::unmute ].active() ) TO.unmuted = core::options::option[ out::unmute ]();
	if( core::options::option[ out::level ].active() )  TO.level = core::options::option[ out::level ]();
	if( core::options::option[ out::chname ].active() ) TO.print_channel_name = core::options::option[ out::chname ]();

	if( core::options::option[ run::version ]() ) {
		TR << "Mini-Rosetta version " << core::minirosetta_svn_version() << " from " << core::minirosetta_svn_url() << std::endl;
	}


	TR << "command:";
	for ( int i=0; i< argc; ++i ) {
		TR << ' ' <<  argv[i];
	}
	TR << std::endl;

	int seed = 1111111;
	int seed_offset = 0;
	bool const_seed = false;
	bool use_time_as_seed = false;
	if( core::options::option[ run::constant_seed ].active() )  const_seed = core::options::option[ run::constant_seed ]();
	if( core::options::option[ run::jran ].active() )  seed = core::options::option[ run::jran ]();
	if( core::options::option[ run::seed_offset ].active() )  seed_offset = core::options::option[ run::seed_offset ]();
	if( core::options::option[ run::use_time_as_seed ].active() )  use_time_as_seed = core::options::option[ run::use_time_as_seed ]();

#ifdef USEMPI
	if( core::options::option[ out::mpi_tracer_to_file ].user() ){
		int mpi_rank( 0 );
		MPI_Comm_rank( MPI_COMM_WORLD, &mpi_rank );

		std::stringstream outfilename;
		outfilename << core::options::option[ out::mpi_tracer_to_file ]() << "_" << mpi_rank;
		core::util::otstreamOP redirect_tracer = new core::util::TracerToFile( outfilename.str() );
		core::util::Tracer::set_ios_hook( redirect_tracer, core::util::Tracer::AllChannels, false );
		core::util::Tracer::super_mute( true );
	}

#endif


	std::string random_device_name( core::options::option[ run::rng_seed_device ]() ); // typically /dev/urandom or /dev/random

	int real_seed;

	if( const_seed ) {
		real_seed = seed + seed_offset;
#ifdef USEMPI
		{ // scope
			/// Give a different RNG seed to each processor
			int mpi_rank( 0 );
			MPI_Comm_rank( MPI_COMM_WORLD, &mpi_rank );
			real_seed += mpi_rank;
		}
#endif
		T("core.init") << "Constant seed mode, seed=" << seed << " seed_offset=" << seed_offset
			<< " real_seed=" << real_seed << std::endl;
	}
	else {
		// attempt to open rng device, if failure then fallback to time
		std::ifstream random_device( random_device_name.c_str(), std::ios::in | std::ios::binary );
		if ( random_device.fail() || use_time_as_seed ) {
			if ( !use_time_as_seed ) {
				// notify user that opening rng device has failed
				T("core.init") << "NOTICE: rng device failure, using time as seed" << std::endl;
			}
			random_device.close();

			//iwd  When using time-based seed on a cluster, seed_offset is usually from 0 to num_processes.
			//iwd  If jobs start a few seconds apart, a simple sum of seed and seed_offset can lead to collisions.
			//iwd  Thus we multiply the time by some number larger than the largest expected value of seed_offset.
			//iwd  If anyone's using this on more than 1000 machines at a time, we need to increase this value.
			//iwd  (Rosetta++ used a multiplier of 20, which helps some, but is nonetheless too small.)
			seed = time(0);
			//seed = seed%10000; // PB-- USE THIS ON OUR CLUSTER TO GET UNIQUE RUNS
			//real_seed = seed + seed_offset;
			real_seed = 1000*seed + seed_offset;

#ifdef USEMPI
			// When we use MPI and time-based seeds on a cluster, adjust the RNG seed so that it is the seed of the head node
			// (the node with an mpi rank of zero) plus the rank of the processer. This or is garentees that each node will
			// have a unique seed.

			/// get the processor rank
			int mpi_rank( 0 );
			MPI_Comm_rank( MPI_COMM_WORLD, &mpi_rank );
			// set the real_seed of each processor to the real seed of the head node
			MPI_Bcast( &real_seed, 1, MPI_INT, 0, MPI_COMM_WORLD );
			// adjust the real seed based on the rank
			real_seed += mpi_rank;
#endif

			T("core.init") << "'Time' seed mode, seed=" << seed << " seed_offset=" << seed_offset
				<< " real_seed=" << real_seed << std::endl;
		} else {
			// grab seeds from device
			uint32_t unsigned_32bit_seed;
			random_device.read( reinterpret_cast< char * >( &unsigned_32bit_seed ), sizeof( uint32_t ));
			random_device.close();

			// calculate actual seeds
			seed = static_cast< int >( unsigned_32bit_seed );
			real_seed = seed + seed_offset;

#ifdef USEMPI
			// Although not as critical with device-based seeding as with time-based clusters, when we use MPI and
			// device-based seeds on a cluster, adjust the RNG seed so that it is the seed of the head node (the node with an
			// mpi rank of zero) plus the rank of the processer. This or is garentees that each node will have a unique
			// seed. Different OSs impliment their RNG differently and as we use increassingly large numbers of processors
			// this may become an issue (but probaly not).

			/// get the processor rank
			int mpi_rank( 0 );
			MPI_Comm_rank( MPI_COMM_WORLD, &mpi_rank );
			// set the real_seed of each processor to the real seed of the head node
			MPI_Bcast( &real_seed, 1, MPI_INT, 0, MPI_COMM_WORLD );
			// adjust the real seed based on the rank
			real_seed += mpi_rank;
#endif

			// log seeds
			T("core.init") << "'RNG device' seed mode, using '" << random_device_name << "', seed=" << seed << " seed_offset=" << seed_offset
			    << " real_seed=" << real_seed << std::endl;
		}

	}





	/*numeric::random::RandomGenerator::initializeRandomGenerators(
		 real_seed, numeric::random::_RND_ConstantSeed_,
		 core::options::option[ run::rng ]  );
	 */
#ifdef BOINC
	std::cerr << "Initializing random generators... ok " << std::endl;std::cerr.flush();
#endif

	init_random_generators(real_seed, _RND_NormalRun_, core::options::option[ run::rng ]);

	// no silly waiting in DEBUG or BOINC builds

#ifdef NDEBUG
#ifndef BOINC
	if( !core::options::option[ run::nodelay ]() ){
		if( core::options::option[ run::delay ]() > 0 ) {
			int waittime = core::options::option[ run::delay ]();
			TR << "Delaying start of mini for " << waittime << " seconds due to -delay option" << std::endl;
			utility::sys_sleep( waittime );
		} else
		if( core::options::option[ run::random_delay ]() > 0 ) {
			int waittime = (int) ( (Real)core::options::option[ run::random_delay ]() * numeric::random::uniform() );
			TR << "Delaying of mini for " << waittime << " seconds (maximum = "
			   <<  core::options::option[ run::random_delay ]()
				 << " )" << std::endl
				 << "This prevents extreme IO levels when multiple jobs start simultaneously on" << std::endl
				 << "large computer clusters  and is default now. To prevent this add the option -nodelay" << std::endl
				 << "To change the random wait time use -run::random_delay <int> " << std::endl;
			utility::sys_sleep( waittime );
		}
	}
#endif
#endif

#ifdef BOINC
	std::cerr << "Initialization complete. " << std::endl;
#endif

	if ( !core::options::option[ options::OptionKeys::in::path::database ].user() ) {
		char * descr = getenv("ROSETTA3_DB");
		if (descr) {
			TR << "found database environment variable ROSETTA3_DB: "<< descr << std::endl;
			core::options::option[ options::OptionKeys::in::path::database ].def( descr );
		} else {
			TR << "ROSETTA3_DB not defined" << std::endl;
		}
	}


}


/// @brief wrapper for core system Init
void init( utility::vector1<std::string> const args )
{
	// create arguments in argc/argv format
	int argc = args.size();
	char **argv = new char*[ argc ];
	for( int ii = 0; ii < argc; ++ii ) {
		argv[ ii ] = new char[ args[ii+1].size()+1 ];
		strncpy( argv[ii], args[ii+1].c_str(), args[ii+1].size() );
		argv[ ii ][ args[ii+1].size() ] = 0; // ensure null termination
	}

	// call init
	init( argc, argv );

	// delete freestore
	for ( int ii = 0; ii < argc; ++ii ) {
		delete[] argv[ ii ];
	}
	delete[] argv;
}

/// @brief Initialize random generator systems (and send debug io to tracer with seed/mode info).
void init_random_generators(int const start_seed, RND_RunType run_type, std::string const & RGtype)
{
	if( run_type == _RND_TestRun_ ) {
		T("core.init.random") << "RandomGenerator:init: _RND_TestRun_ mode, seed=" << start_seed <<
			" RG_type=" << RGtype << std::endl;
	}
	else {
		T("core.init.random") << "RandomGenerator:init: Normal mode, seed=" << start_seed <<
			" RG_type=" << RGtype << std::endl;
	}

	RandomGenerator::initializeRandomGenerators(start_seed, run_type, RGtype);
}


} // namespace core

