// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/io/silent/atom_tree_diff.hh
///
/// @brief  Silent-file format based on "diffs" of AtomTree DOFs
/// @author Ian W. Davis

#ifndef INCLUDED_core_io_silent_atom_tree_diff_HH
#define INCLUDED_core_io_silent_atom_tree_diff_HH

#include <core/types.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <utility/vector1.hh>
#include <utility/pointer/ReferenceCount.hh>

#include <fstream>
#include <iostream>
#include <map>
#include <string>

namespace core {
namespace io {
namespace atom_tree_diffs {


class AtomTreeDiff; // fwd declaration
typedef utility::pointer::owning_ptr< AtomTreeDiff > AtomTreeDiffOP;
typedef utility::pointer::owning_ptr< AtomTreeDiff const > AtomTreeDiffCOP;

///@brief An object wrapper for reading atom_tree_diff files,
/// complete with embedded reference structures.
///
///@details Only works with uncompressed files, because we have to be able to
/// to random access (seekg) to pull out single structures in random order.
///
class AtomTreeDiff : public utility::pointer::ReferenceCount
{
public:

	typedef std::map< std::string, core::Real > Scores;
	//typedef std::map< std::string, Scores > ScoresMap;
	/// Just like ScoresMap, but can be sorted into some particular order *by scores*.
	/// Maps can only be sorted by key, which here is just a pose tag.
	typedef utility::vector1< std::pair< std::string, Scores > > ScoresPairList;

	AtomTreeDiff(std::string filename);
	virtual ~AtomTreeDiff();

	///@brief returns true if a reference struct with the given tag is present
	bool has_ref_pose(std::string const & tag) const;

	/// @brief True if a (non-reference) structure with the given tag is present in the file
	bool has_tag(std::string const & tag) const;

	/// @brief Return list of (pose tag, score sets) pairs for all poses, in file order.
	ScoresPairList const & scores() const { return scores_; }

	/// @brief Utility function for selecting subsets of structures by score.
	static void sort_by(std::string const & score_name, ScoresPairList & scores, bool descending=false);

	/// @brief Reads the pose data from file and reconstructs the complete pose.
	void read_pose(std::string const & tag, core::pose::Pose & pose_out);

	/// @brief Reads the pose data from file and reconstructs the complete pose, using the supplied reference pose.
	void read_pose(std::string const & tag, core::pose::Pose & pose_out, core::pose::Pose const & ref_pose);

	/// @brief Returns the default reference pose for the given tag.  Fails if none is available.
	core::pose::PoseCOP ref_pose_for(std::string const & tag);

	/// @brief Allows access to and mutation of (!) the references poses stored in this file.  Use with caution.
	/// @details This exists to allow setup on stored reference poses for properties that don't get saved/restored
	/// in PDB format, like covalent constraints for enzyme design.
	utility::vector1< core::pose::PoseOP > const & all_ref_poses() const
	{ return unique_ref_poses_; }

private:
	/// The list of (tag,scores) pairs for the diffed structures, for efficient selection of subsets.
	/// (E.g. the top 5% by total score.) Memory cost is probably ~1 kb / structure.
	ScoresPairList
		scores_;

	/// Maps tags to indices in scores_
	std::map< std::string, core::Size >
		tag_idx_;

	/// The map of (tag, file position) for getting random access to the structural data.
	std::map< std::string, long >
		offsets_;

	/// The map of (tag, reference pose) for decoding the diffs.  Some could be null.
	std::map< std::string, core::pose::PoseOP > ref_poses_;

	/// All references poses from the file, one copy each for convenience.
	utility::vector1< core::pose::PoseOP > unique_ref_poses_;

	// The file being read.
	std::ifstream in_;

}; // class AtomTreeDiff


///@brief Helper function for writing entries -- not usually called by clients.
void dump_score_line(
	std::ostream & out,
	std::string const & pose_tag,
	std::map< std::string, core::Real > const & scores
);

///@brief Helper function for writing entries -- not usually called by clients.
void dump_score_line(
	std::ostream & out,
	std::string const & pose_tag
);

///@brief Embeds a reference pose as PDB coords + foldtree; will be used for reconstructing subsequent diffs.
void dump_reference_pose(
	std::ostream & out,
	std::string const & pose_tag,
	std::map< std::string, core::Real > const & scores,
	core::pose::Pose const & pose
);

///@brief Encodes pose relative to ref_pose by noting which atom_tree DOFs are different.
void dump_atom_tree_diff(
	std::ostream & out,
	std::string const & pose_tag,
	std::map< std::string, core::Real > const & scores,
	core::pose::Pose const & ref_pose_in,
	core::pose::Pose const & pose,
	int bb_precision = 6,
	int sc_precision = 4,
	int bondlen_precision = 2
);

///@brief Gets next tag and scores from the stream, or returns false if none.
/// Call this to find desired structure, then call pose_from_atom_tree_diff().
bool header_from_atom_tree_diff(
	std::istream & in,
	std::string & pose_tag_out,
	std::map< std::string, core::Real > & scores_out
);


///@brief Sets pose = ref_pose and then starts modifying DOFs in pose to recreate a saved structure.
/// Call after header_from_atom_tree_diff().  Returns false on error.
bool pose_from_atom_tree_diff(
	std::istream & in,
	core::pose::Pose const & ref_pose,
	core::pose::Pose & pose
);


///@brief Helper for dump_atom_tree_diff(), fills map with weighted score terms.
void map_of_weighted_scores(
	core::pose::Pose & pose, //< pose is not modified but scoring is a non-const op
	core::scoring::ScoreFunction const & sfxn,
	std::map< std::string, core::Real > & scores_out
);


///@brief For use in deciding how many digits of precision you need when diffing an atom tree.
void rms_error_with_noise(
	core::pose::Pose const & ref_pose,
	int bb_precision = 6,
	int sc_precision = 4
);


} // silent
} // io
} // core

#endif // INCLUDED_core_io_silent_atom_tree_diff_HH
