// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
///
/// @brief
/// @author

#ifndef INCLUDED_core_io_pdb_pose_io_HH
#define INCLUDED_core_io_pdb_pose_io_HH


#include <core/types.hh>
#include <core/pose/Pose.fwd.hh>

#include <core/conformation/Residue.fwd.hh>
#include <core/chemical/ResidueTypeSet.fwd.hh>
#include <core/id/AtomID_Mask.fwd.hh>

#include <core/io/pdb/file_data.hh>
#include <core/util/Tracer.fwd.hh>

// C++ headers
#include <string>
#include <iosfwd>

namespace core {
namespace io {
namespace pdb {


/// @brief special Tracer instance acting as special param for all traced_dump_pdb functions
extern core::util::Tracer TR_dump_pdb_dummy;


void
write_additional_pdb_data(
	std::ostream & out,
	pose::Pose const & pose,
	FileData const & fd,
	bool write_fold_tree = false
);


void
read_additional_pdb_data(
	std::string const & file,
	pose::Pose & pose,
	FileData const & fd,
	bool read_fold_tree = false
);


/// @brief a function that reads a PDB file into a Pose.  This constructor allows for a non-default ResidueTypeSet
void
pose_from_pdb(
	pose::Pose & pose,
	chemical::ResidueTypeSet const & residue_set,
	std::string const & filename,
	bool read_fold_tree = false
);


/// @brief a function that reads a PDB file into a Pose.
/// uses the FA_STANDARD residue_set (fullatom) by default, but will use centroid if in::file::centroid_input is true
void
pose_from_pdb(
	pose::Pose & pose,
	std::string const & filename,
	bool read_fold_tree = false
);

/// @brief a function that reads a PDB file containing multiple models into a vector of Poses
void
pose_from_pdb(
	utility::vector1< pose::Pose > & poses,
	chemical::ResidueTypeSet const & residue_set,
	std::string const & filename,
	bool read_fold_tree = false
);

utility::vector1< core::pose::PoseOP > poseOPs_from_pdbs(
	utility::vector1< std::string > const & filenames,
	bool read_fold_tree = false
);

utility::vector1< core::pose::Pose > poses_from_pdbs(
	utility::vector1< std::string > const & filenames,
	bool read_fold_tree = false
);

utility::vector1< core::pose::Pose > poses_from_pdbs(
	chemical::ResidueTypeSet const & residue_set,
	utility::vector1< std::string > const & filenames,
	bool read_fold_tree = false
);

utility::vector1< core::pose::PoseOP > poseOPs_from_pdbs(
	chemical::ResidueTypeSet const & residue_set,
	utility::vector1< std::string > const & filenames,
	bool read_fold_tree = false
);

// FA_STANDARD residue set
void
pose_from_pdb(
	utility::vector1< pose::Pose > & poses,
	std::string const & filename,
	bool read_fold_tree = false
);

void
pose_from_pose(
		pose::Pose & new_pose,
		pose::Pose const & old_pose,
		utility::vector1< core::Size > const & residue_indices
);

void
pose_from_pose(
		pose::Pose & new_pose,
		pose::Pose const & old_pose,
		chemical::ResidueTypeSet const & residue_set,
		utility::vector1< core::Size > const & residue_indices
);

/// uses the CENTROID residue_set
void
centroid_pose_from_pdb(
	pose::Pose & pose,
	std::string const & filename,
	bool read_fold_tree = false
);


void
dump_pdb_residue(
	conformation::Residue const & rsd,
	Size & atom_number,
	std::ostream & out
);


void
dump_bfactor_pdb(
	pose::Pose const & pose,
	id::AtomID_Map< Real > const & bfactor,
	std::ostream & out,
	std::string const & tag="1"
);


void
dump_pdb(
	pose::Pose const & pose,
	std::ostream & out,
	id::AtomID_Mask const & mask,
	std::string const & tag="1"
);


void
dump_pdb(
	pose::Pose const & pose,
	std::ostream & out,
	std::string const & tag="1"
);


void
dump_pdb(
	pose::Pose const & pose,
	std::string const & filename,
	std::string const & tag="1"
);


/// @brief dump_pdb depending on visibility of tracer
/// @param[in] tr   output performed if tracer is visible or if passed dummy
///  tracer core::io::pdb::TR_dump_pdb_dummy
void
traced_dump_pdb(
	core::util::Tracer const & tr,
	pose::Pose const & pose,
	std::ostream & out,
	std::string const & tag="1"
);


/// @brief dump_pdb depending on visibility of tracer
/// @param[in] tr   output performed if tracer is visible or if passed dummy
///  tracer core::io::pdb::TR_dump_pdb_dummy
void
traced_dump_pdb(
	core::util::Tracer const & tr,
	pose::Pose const & pose,
	std::string const & filename,
	std::string const & tag="1"
);


} // namespace pdb
} // namespace io
} // namespace core


#endif // INCLUDED_core_io_pdb_pdb_file_data_HH
