// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/core/io/pose_stream/PDBPoseInputStream.cc
/// @brief
/// @author James Thompson

// libRosetta headers

#include <core/types.hh>
#include <core/chemical/ResidueTypeSet.fwd.hh>
#include <core/pose/Pose.hh>
#include <core/pose/datacache/CacheableDataType.hh>

#include <core/io/pdb/pose_io.hh>
#include <core/io/pose_stream/PoseInputStream.hh>
#include <core/io/pose_stream/PDBPoseInputStream.hh>

#include <core/util/datacache/BasicDataCache.hh>

#include <utility/vector1.hh>
#include <utility/file/FileName.hh>
#include <utility/io/izstream.hh>

#include <core/util/basic.hh>
#include <core/util/Tracer.hh>
#include <core/util/datacache/BasicDataCache.hh>
#include <core/util/datacache/CacheableString.hh>

// C++ headers
#include <string>

namespace core {
namespace io {
namespace pose_stream {


void PDBPoseInputStream::set_filenames(
	utility::vector1< utility::file::FileName > filenames
) {
	filenames_ = filenames;
	current_position_ = filenames_.begin();
}

utility::vector1< utility::file::FileName > PDBPoseInputStream::get_filenames() {
	return filenames_;
}

bool PDBPoseInputStream::has_another_pose() {
	return ( current_position_ != filenames_.end() );
}

void PDBPoseInputStream::fill_pose(
	core::pose::Pose & pose,
	core::chemical::ResidueTypeSet const & residue_set
) {
	// check to make sure that we have more poses!
	if ( !has_another_pose() ) {
		utility_exit_with_message(
			"PDBPoseInputStream: called fill_pose, but I have no more Poses!"
		);
	}

	core::io::pdb::pose_from_pdb( pose, residue_set, *current_position_ );
	// set up a tag using input filename.
	pose.data().set(
		core::pose::datacache::CacheableDataType::JOBDIST_OUTPUT_TAG,
		new core::util::datacache::CacheableString( *current_position_ )
	);
	++current_position_;
}

utility::vector1< core::pose::Pose > PDBPoseInputStream::get_all_poses(
	core::chemical::ResidueTypeSet const & residue_set
) {
	utility::vector1< core::pose::Pose > pose_list;
	pose_list.resize( filenames_.size() );
	while( has_another_pose() ) {
		core::pose::Pose pose;
		fill_pose( pose, residue_set );
		pose_list.push_back( pose );
	}

	return pose_list;
}

/// @brief adds a list of files each containing lists of PDBs
void PDBPoseInputStream::add_list_filenames(
	utility::vector1< utility::file::FileName > list_fns
) {
	using utility::vector1;
	bool init_current_position( filenames_.size() == 0 );

	for( vector1< utility::file::FileName >::iterator it = list_fns.begin(),
				it_end = list_fns.end();
				it != it_end; ++it
	) {
		std::string filename( it->name() );
		utility::io::izstream data( filename.c_str() );
		if ( !data.good() )
			utility_exit_with_message( "Unable to open file: " + filename + '\n' );

		std::string line;
		while( getline(data, line) ) {
			filenames_.push_back( utility::file::FileName(line) );
		}
		data.close();
	}

	if ( init_current_position ) {
		current_position_ = filenames_.begin();
	}
} // add_list_files

} // pose_stream
} // io
} // core
