// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file core/io/silent/BinaryProteinSilentStruct.hh
///
/// @brief
/// @author Frank DiMaio
/// @author Mike Tyka

#ifndef INCLUDED_core_io_silent_BinaryProteinSilentStruct_HH
#define INCLUDED_core_io_silent_BinaryProteinSilentStruct_HH

// mini headers
#include <core/types.hh>
#include <core/pose/Pose.fwd.hh>

#include <core/conformation/Residue.fwd.hh>

#include <core/io/silent/ProteinSilentStruct.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/kinematics/Jump.hh>

#include <core/util/OStream.fwd.hh>

#include <utility/vector1.hh>
#include <utility/pointer/ReferenceCount.hh>


// C++ Headers
#include <cstdlib>
#include <iostream>
#include <cassert>
#include <vector>
#include <string>
#include <map>

namespace core {
namespace io {
namespace silent {

class BinaryProteinSilentStruct : public ProteinSilentStruct {

public:

	/// @brief Constructors.
	BinaryProteinSilentStruct( Size const nres_in );

	BinaryProteinSilentStruct();

	BinaryProteinSilentStruct(
		core::pose::Pose const & pose,
		std::string tag = "empty_tag"
	);

	/// @brief Re-dimension the storage capacity of this BinaryProteinSilentStruct to the given number of residues.
	void resize(
		Size const nres_in
	);

	// destructor
	~BinaryProteinSilentStruct() {}

	/// @brief Test if this BinaryProteinSilentStruct is equal to the given
	/// BinaryProteinSilentStruct in terms of conformation.  Doesn't check
	/// energies.
	BinaryProteinSilentStruct & operator= (
		BinaryProteinSilentStruct const & src
	);

	/// @brief Initialize object from a set of lines
	virtual bool init_from_lines(
		utility::vector1< std::string > const & lines,
		SilentFileData & container
	);

	/// @brief Configure the conformation of the given Pose with the
	/// conformational data within this BinaryProteinSilentStruct. Calls
	/// pose.clear() and rebuilds Pose from scratch using FA_STANDARD residue
	/// types.
	virtual void fill_pose(
		core::pose::Pose & pose
	) const ;

	/// @brief Configure the conformation of the given Pose with the
	/// conformational data within this BinaryProteinSilentStruct. Calls
	/// pose.clear() and rebuilds Pose from scratch using the user-specified
	/// residue types.
	virtual void fill_pose(
		core::pose::Pose & pose,
		core::chemical::ResidueTypeSet const & residue_set
	) const;

	/// @brief opposite of fill_pose
	virtual void fill_struct( core::pose::Pose const & pose, std::string tag );

	/// @brief print header information
	virtual void print_header( std::ostream & out ) const;

	/// @brief Prints the conformation information within this
	/// BinaryProteinSilentStruct to the given std::ostream.
	virtual void print_conformation( std::ostream & output ) const;

	/// @brief returns the positions of the CA atoms
	virtual ObjexxFCL::FArray2D< Real > get_CA_xyz();

	// model quality-related methods.
	virtual Real CA_rmsd( BinaryProteinSilentStruct other_pss );

	/// @brief calculates the RMSD between the C-alpha atoms of a Pose built from
	/// the torsions in this ProteinSilentStruct and the C-alpha atoms from this
	/// ProteinSilentStruct.
	virtual Real get_debug_rmsd();

protected:
	utility::vector1< utility::vector1< numeric::xyzVector<float> > > atm_coords_;


	// the rest shouldn't have been here. it overwrites functionality of the Parent class which can have funny effects...
	// e.g. fullatom_ = true --> changes the value of BinarySilentStruct
	// but fullatom() retrieved the value of ProteinSilentStruct... hmpf

// 	void add_jump( kinematics::Jump jump ) {
// 		jumps_.push_back( jump.rt() );
// 	}

// 	kinematics::RT const & jump( Size jump_num ) const {
// 		return jumps_[ jump_num ];
// 	}

// 	void fold_tree( kinematics::FoldTree f ) {
// 		fold_tree_ = f;
// 	}

// 	kinematics::FoldTree const& fold_tree( ) const {
// 		return fold_tree_;
// 	}

// 	char secstruct( unsigned int seqpos ) const {
// 		return secstruct_[seqpos];
// 	}

// 	void secstruct( unsigned int seqpos, char ss ) {
// 		secstruct_[seqpos] = ss;
// 	}

	//NO this overwrites the parent class value
	//bool fullatom_;

	//	utility::vector1< char > secstruct_;

	//	utility::vector1< kinematics::RT > jumps_;
	//	bool bJumps_use_IntraResStub_;
	//	kinematics::FoldTree fold_tree_;

}; // class BinaryProteinSilentStruct

} // namespace silent
} // namespace io
} // namespace core

#endif
