// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file core/io/silent/SilentStruct.cc
///
/// @brief silent input file reader for mini.
/// @author James Thompson

// C++ Headers
#include <iostream>
#include <sstream>
#include <string>
#include <map>

#ifdef WIN32
	#include <ctime>
#endif

#ifdef __CYGWIN__
	#include <ctime>
#endif

// mini headers
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/char.functions.hh>
#include <ObjexxFCL/string.functions.hh>

#include <core/io/silent/SilentStruct.hh>
#include <core/io/silent/EnergyNames.hh>

#include <core/util/Tracer.hh>
#include <core/util/datacache/BasicDataCache.hh>
#include <core/util/datacache/CacheableString.hh>
#include <core/util/datacache/CacheableStringFloatMap.hh>

#include <core/options/option.hh>

#include <core/id/AtomID.hh>
#include <core/id/AtomID_Map.hh>

#include <core/id/AtomID_Map.Pose.hh>
#include <core/pose/Pose.hh>
#include <core/pose/util.hh>
#include <core/pose/datacache/CacheableDataType.hh>
#include <core/pack/optimizeH.hh>

#include <core/chemical/util.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/conformation/Residue.fwd.hh>
#include <core/conformation/ResidueFactory.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/rms_util.hh>

#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreTypeManager.hh>

#include <core/util/datacache/BasicDataCache.hh>

#ifdef BOINC
#include <protocols/boinc/boinc.hh>
#endif

// option key includes

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>


namespace core {
namespace io {
namespace silent {

static core::util::Tracer tr("core.io.silent");

SilentStruct::SilentStruct()
	: precision_(3), scoreline_prefix_("SCORE: ")
{}

SilentStruct::~SilentStruct() {}

void SilentStruct::fill_pose(
	core::pose::Pose & pose
) const {
	using namespace core::chemical;
	ResidueTypeSetCAP residue_set
		= ChemicalManager::get_instance()->residue_type_set( FA_STANDARD );
	fill_pose( pose, *residue_set );
}

/// @brief Fill a Pose with the conformation information in this SilentStruct
/// and the ResidueTypeSet provided by the caller. This is a virtual method
/// which must be implemented by classes derived from SilentStruct.
void SilentStruct::fill_pose(
	core::pose::Pose &,
	core::chemical::ResidueTypeSet const &
) const {
	tr.Error << "SilentStruct::fill_pose method stubbed out!" << std::endl;
}

void SilentStruct::finish_pose(
	core::pose::Pose & pose
) const
{
	using namespace options;
	using namespace options::OptionKeys;
	if ( option[ in::file::silent_optH ].user() ) {
		id::AtomID_Mask missing( false );
		id::initialize( missing, pose );
		pack::optimize_H_and_notify( pose, missing );
	}
	energies_into_pose( pose );
}

void
SilentStruct::print_header( std::ostream & out ) const {
	out << "SEQUENCE: " << one_letter_sequence() << std::endl;
	print_score_header( out );
}

void SilentStruct::print_score_header( std::ostream & out ) const {
	out << scoreline_prefix();
	using utility::vector1;

	typedef vector1< SilentEnergy >::const_iterator iter;
	for ( iter it = silent_energies_.begin(), end = silent_energies_.end();
			it != end; ++it
	) {
		out << " " << A( it->width(), it->name() );
	}

	using namespace core::options;
	using namespace core::options::OptionKeys;
	/// print out a column for the user tag if specified.
	if ( option[ out::user_tag ].user() ) {
		std::string const tag = option[ out::user_tag ]();
		int width = std::max( 11, static_cast< int > (tag.size()) );
		out << ' ' << A( width, "user_tag" );
	}

	int width = decoy_tag().size();
	if ( width < 11 ) width = 11; //size of "description"
	out << ' ' << A( width, "description" ) << std::endl;
}

void
SilentStruct::print_scores( std::ostream & out ) const {
	using namespace core::options;
	using namespace core::options::OptionKeys;
	//int precision = 3; // number of digits after decimal place

	out << scoreline_prefix();

	typedef utility::vector1< SilentEnergy >::const_iterator iter;
	for ( iter it = silent_energies_.begin(), end = silent_energies_.end();
			it != end; ++it
	) {

		if ( it->string_value() == "" ){
			core::Real weight = 1.0;
			if ( option[ out::file::weight_silent_scores ]() ) {
				weight = it->weight();
			}
			out << " " << F( it->width(), precision(), it->value() * weight );
		} else {
			out << " " << it->string_value() << " ";
		}
	}

	/// print out a column for the user tag if specified.
	if ( option[ out::user_tag ].user() ) {
		std::string const tag = option[ out::user_tag ]();
		int width = std::min( 11, static_cast< int > (tag.size()) );
		out << ' ' << A( width, tag );
	}

	int width = decoy_tag().size();
	if ( width < 11 ) width = 11; // size of "description"
	out << ' ' << A( width, decoy_tag() )  << "\n";
	print_comments( out );
}

void
SilentStruct::print_comments( std::ostream & out ) const {
	using std::map;
	using std::string;
	string const remark( "REMARK" );

	typedef map< string, string >::const_iterator c_iter;
	for ( c_iter it = silent_comments_.begin(), end = silent_comments_.end();
			it != end; ++it
	) {
		out << remark << ' ' << it->first << ' ' << it->second << std::endl;
	}
} // print_comments

bool SilentStruct::has_energy( std::string const scorename ) const {
	for ( utility::vector1< SilentEnergy >::const_iterator
				it = silent_energies_.begin(), end = silent_energies_.end();
				it != end; ++it
	) {
		if ( it->name() == scorename )
			return true;
	}
	return false;
}

void
SilentStruct::add_energy( std::string scorename, Real value, Real weight ) {
	// check if we already have a SilentEnergy with this scorename
	bool replace( false );
	typedef utility::vector1< SilentEnergy >::iterator iter;
	for ( iter it = silent_energies_.begin(), end = silent_energies_.end();
			it != end; ++it
	) {
		if ( it->name() == scorename ) {
			it->value( value );
			replace = true;
		}
	} // for (silent_energies_)

	// add this energy if we haven't added it already
	if ( !replace ) {
		int width = std::max( 10, (int) scorename.length() + 3 );
		SilentEnergy new_se( scorename, value, weight, width );
		silent_energies_.push_back( new_se );
	} // if (!replace)
} // add_energy

void
SilentStruct::add_string_value(
	std::string scorename, std::string const & value
) {
	// check if we already have a SilentEnergy with this scorename
	bool replace( false );
	typedef utility::vector1< SilentEnergy >::iterator iter;
	for ( iter it = silent_energies_.begin(), end = silent_energies_.end();
			it != end; ++it
	) {
		if ( it->name() == scorename ) {
			it->value( value );
			replace = true;
		}
	} // for (silent_energies_)

	// add this energy if we haven't added it already
	if ( !replace ) {
		int width = std::max( 10, (int) scorename.length() + 3 );
		SilentEnergy new_se( scorename, value, width );
		silent_energies_.push_back( new_se );
	}
} // add_string_value

core::Real SilentStruct::get_energy( std::string const & scorename ) const {
	if ( has_energy( scorename ) )
		return get_silent_energy( scorename ).value();
	else
		return 0.0;
}

std::string const & SilentStruct::get_string_value(
	std::string const & scorename
) const {
	static const std::string empty_string("");
	if ( has_energy( scorename ) )
		return get_silent_energy( scorename ).string_value();
	else
		return empty_string;
}


SilentEnergy const & SilentStruct::get_silent_energy(
	std::string const & scorename
) const {
	assert( has_energy( scorename ) );
	using utility::vector1;
	for ( vector1< SilentEnergy >::const_iterator it = silent_energies_.begin(),
				end = silent_energies_.end();
				it != end; ++it )
	{
		if ( it->name() == scorename ) return *it;
	}
	return (*silent_energies_.end()); // keep compiler happy
}


void SilentStruct::set_valid_energies( utility::vector1< std::string > valid ) {
	using utility::vector1;

	vector1< SilentEnergy > new_energies;
	for ( vector1< std::string >::const_iterator it = valid.begin(),
			end = valid.end();
			it != end; ++it
	) {
		SilentEnergy ener;
		if ( *it == "description" ) continue; // hack!
		if ( has_energy( *it ) ){
			ener = get_silent_energy( *it );
		}
		ener.name( *it );
		new_energies.push_back( ener );
	}
	silent_energies( new_energies );
} // set_valid_energies

///////////////////////////////////////////////////////////////////////
void
SilentStruct::copy_scores( SilentStruct & src_ss ) {
	typedef utility::vector1< SilentEnergy >::iterator iter;

	for ( iter it = src_ss.silent_energies_.begin(),
			end = src_ss.silent_energies_.end();
			it != end; ++it
	) {
		bool replace( false );

		//Check if score column already present.
		for ( iter it2 = silent_energies_.begin(), end = silent_energies_.end();
				it2 != end; ++it2
		) {
			if ( it2->name() == it->name() ) {
				it2->value( it->value() );
				replace = true;
			}
		} // for it2

		if (!replace) silent_energies_.push_back( *it );
	} // for it
} // copy_scores

///////////////////////////////////////////////////////////////////////
void SilentStruct::add_comment( std::string name, std::string value ) {
	silent_comments_.insert( std::make_pair( name, value ) );
}

void SilentStruct::comment_from_line( std::string const & line ) {
	std::istringstream line_stream( line );
	std::string dummy, key, val, remark_tag;
	line_stream >> remark_tag;
	line_stream >> key;
	line_stream >> val;
	if ( line_stream.fail() ) {
		tr.Error << "[ERROR] reading comment from line: " << line << std::endl;
		return;
	}

	line_stream >> dummy;
	while( line_stream.good() ) {
		val += " ";
		val += dummy;
		line_stream >> dummy;
	}
	add_comment( key, val );
}

std::map< std::string, std::string > SilentStruct::get_all_comments() const {
	return silent_comments_;
}

bool SilentStruct::has_comment( std::string const & name ) const {
	return ( silent_comments_.find( name ) != silent_comments_.end() );
}

std::string SilentStruct::get_comment( std::string const & name ) const {
	std::map< std::string, std::string >::const_iterator entry
		= silent_comments_.find( name );
	std::string comment( "" );
	if ( entry != silent_comments_.end() ) {
		comment = entry->second;
	}

	return comment;
}

void SilentStruct::parse_energies(
	std::istream & input,
	utility::vector1< std::string > const & energy_names
) {
	std::string tag;
	utility::vector1< std::string >::const_iterator energy_iter;
	Size input_count = 0;
	Size const energy_names_count( energy_names.size() );
	for ( energy_iter  = energy_names.begin();
			energy_iter != energy_names.end();
			++energy_iter
	) {
		input >> tag;
		if ( is_float( tag ) ) {
			Real score_val = static_cast< Real > ( float_of( tag ) );
			add_energy( *energy_iter, score_val );
		} else if ( *energy_iter == "description" ) {
			decoy_tag( tag );
		} else {
			add_string_value( *energy_iter, tag );
		}
		++input_count;
	} // for ( energy_iter ... )

	if ( energy_names_count != input_count ) {
		tr.Warning << "Warning: I have " << energy_names_count
			<< " energy names but I have " << input_count
			<< " energy values." << std::endl;
	}
} // parse_energies

void SilentStruct::update_score() {
	runtime_assert( silent_energies_.begin()->name() == "score" );

	// set the "score" term to its appropriate value
	typedef utility::vector1< SilentEnergy >::iterator energy_it;

	Real score( 0.0 );
	for ( energy_it it = silent_energies_.begin(), end = silent_energies_.end();
			it != end; ++it
	) {
			score += it->weight() * it->value();
	}

	silent_energies_.begin()->value( score );
} // update_score()

void SilentStruct::energies_from_pose( core::pose::Pose const & pose ) {
	using namespace core::pose::datacache;

	core::scoring::EnergyMap const emap = pose.energies().total_energies();
	core::scoring::EnergyMap const wts  = pose.energies().weights();

	clear_energies();

	// "score" is the total of weighted scores. always the first element in
	// silent_energies_
	SilentEnergy score_energy( "score", 0.0, 1.0, 8 );
	silent_energies_.push_back( score_energy );
	core::scoring::EnergyMap::const_iterator emap_iter, wts_iter;
	for ( emap_iter = emap.begin(), wts_iter = wts.begin();
			emap_iter != emap.end() && wts_iter!= wts.end();
			++emap_iter && ++wts_iter
	) {

		// only grab scores that have non-zero weights.
		if ( *wts_iter != 0.0 ) {
			core::scoring::ScoreType sc_type
				= core::scoring::ScoreType( emap_iter - emap.begin() + 1 );
			std::string name = core::scoring::name_from_score_type( sc_type );

			int width = std::max( 10, (int) name.length() + 3 );
			SilentEnergy new_se( name, *emap_iter, *wts_iter, width );
			silent_energies_.push_back( new_se );
		} // if ( *wts_iter != 0.0 )
	} // for ( emap_iter ...)

	// set the "score" term to its appropriate value
	update_score();

	// get arbitrary floating point scores from the map stored in the Pose data cache
	if ( pose.data().has( CacheableDataType::ARBITRARY_FLOAT_DATA ) ) {
		const core::util::datacache::CacheableStringFloatMap *data
			= dynamic_cast< const core::util::datacache::CacheableStringFloatMap* >
			( pose.data().get_raw_const_ptr(CacheableDataType::ARBITRARY_FLOAT_DATA) );

		using std::map;
		using std::string;
		for ( map< string, float >::const_iterator iter = data->map().begin(),
				end = data->map().end(); iter != end; iter++
		)	{
			// skip score entry, as it gets confusing
			if ( iter->first == "score" ) continue;
			SilentEnergy new_se(
				iter->first, iter->second, 1.0,
				static_cast< int > (iter->first.size() + 3)
			);
			silent_energies_.push_back( new_se );
		}
	} // 	if ( pose.data().has( CacheableDataType::ARBITRARY_FLOAT_DATA ) ) )

	// add comments from the Pose
	using std::map;
	using std::string;
	map< string, string > comments = core::pose::get_all_comments( pose );
	for ( map< string, string >::const_iterator it = comments.begin(),
				end = comments.end(); it != end; ++it
	) {
		add_comment( it->first, it->second );
	}

	// add score_line_strings from Pose
	map< string, string > score_line_strings(
		core::pose::get_all_score_line_strings( pose )
	);
	for ( map< string, string >::const_iterator it = score_line_strings.begin(),
				end = score_line_strings.end();
				it != end; ++it
	) {
		add_string_value( it->first, it->second );
	}

} // void energies_from_pose( core::pose::Pose const & pose )

void SilentStruct::energies_into_pose( core::pose::Pose & pose ) const {
	using namespace core::options;
	using namespace core::options::OptionKeys;
	using namespace core::pose::datacache;
	using namespace core::scoring;
	using std::string;
	// make sure that the pose has ARBITRARY_FLOAT_DATA in the DataCache
	if ( !pose.data().has( ( CacheableDataType::ARBITRARY_FLOAT_DATA ) ) ){
		pose.data().set(
			CacheableDataType::ARBITRARY_FLOAT_DATA,
			new core::util::datacache::CacheableStringFloatMap()
		);
	}

	core::util::datacache::CacheableStringFloatMap *data
		= dynamic_cast< core::util::datacache::CacheableStringFloatMap* >
		( pose.data().get_raw_ptr(CacheableDataType::ARBITRARY_FLOAT_DATA) );

	runtime_assert( data != NULL );

	EnergyMap weights;
	EnergyMap & emap( pose.energies().total_energies() );

	string const input_score_prefix( option[ in::file::silent_score_prefix ]() );
	typedef utility::vector1< SilentEnergy > elist;

	elist es = energies();
	for ( elist::const_iterator it = es.begin(), end = es.end();
			it != end; ++it
	) {
		// logic is:
		//	- if it->name() points to a ScoreType, put the value into the EnergyMap
		// - otherwise, add the value to the Pose DataCache ARBITRARY_FLOAT_DATA
		string const proper_name( input_score_prefix + it->name() );
		if ( ScoreTypeManager::is_score_type( proper_name ) )  {
			ScoreType sc_type( ScoreTypeManager::score_type_from_name( proper_name ) );
			emap   [ sc_type ] = it->value();
			weights[ sc_type ] = it->weight();
		} else if ( it->string_value() != "" ) {
			core::pose::add_score_line_string(
				pose, proper_name, it->string_value()
			);
		} else {
			data->map()[ proper_name ] = it->value();
		}
	} // for silent_energies

	pose.energies().weights( weights );
	pose.data().set( CacheableDataType::ARBITRARY_FLOAT_DATA, data );

	using std::map;
	using std::string;
	map< string, string > comments = get_all_comments();
	for ( map< string, string >::const_iterator it = comments.begin(),
				end = comments.end(); it != end; ++it
	) {
		string const proper_name( input_score_prefix + it->first );
		core::pose::add_comment( pose, proper_name, it->second );
	}
} // energies_into_pose

EnergyNames SilentStruct::energy_names() const {
	utility::vector1< SilentEnergy > se = energies();

	utility::vector1< std::string > names;
	for ( utility::vector1< SilentEnergy >::const_iterator it = se.begin(),
				end = se.end(); it != end; ++it
	) {
		names.push_back( it->name() );
	}
	EnergyNames enames;
	enames.energy_names( names );
	return enames;
}


void SilentStruct::rename_energies() {
	using namespace std;

	static map< string, string > scorename_conversions;
	static bool init( false );
	if ( !init ) {
		scorename_conversions["cb"] = "cbeta";
		scorename_conversions["hs"] = "hs_pair";
		scorename_conversions["ss"] = "ss_pair";
	}

	// iterate over the silent-file energies, convert them when appropriate.
	map< string, string >::const_iterator conv_it,
		conv_end( scorename_conversions.end() );
	for ( utility::vector1< SilentEnergy >::iterator it = silent_energies_.begin(),
			end = silent_energies_.end(); it != end; ++it
	) {
		conv_it = scorename_conversions.find( it->name() );
		if ( conv_it != conv_end ) {
			it->name( conv_it->second );
		}
	}
} // rename_energies

void SilentStruct::set_tag_from_pose(
	const core::pose::Pose & pose
) {
	using namespace core::pose::datacache;
	std::string tag( "empty_tag" );
	if ( pose.data().has( CacheableDataType::JOBDIST_OUTPUT_TAG ) ) {
		tag =
			static_cast< core::util::datacache::CacheableString const & >
			( pose.data().get( CacheableDataType::JOBDIST_OUTPUT_TAG ) ).str();
	}
	decoy_tag(tag);
}

//void
//SilentStruct::print_sequence( bool print_seq ) {
//	print_sequence_ = print_seq;
//}
//
//bool SilentStruct::print_sequence() const {
//	return print_sequence_;
//}

void
SilentStruct::precision( core::Size precision ) {
	precision_ = precision;
}

core::Size SilentStruct::precision() const {
	return precision_;
}

void
SilentStruct::scoreline_prefix( std::string const & prefix )
{
	scoreline_prefix_ = prefix;
}

std::string SilentStruct::scoreline_prefix() const {
	return scoreline_prefix_;
}

std::string
SilentStruct::one_letter_sequence() const {
	//using core::chemical::annotated_to_oneletter_sequence;
	//return annotated_to_oneletter_sequence( sequence () );
	std::string annotated_seq = sequence();
	std::string sequence("");
	bool in_bracket = false;
	for ( Size i = 0, ie = annotated_seq.length(); i < ie; ++i ) {
		char c = annotated_seq[i];
		if ( c == '[' ) {
			in_bracket = true;
			continue;
		} else if ( c == ']' ) {
			in_bracket = false;
			continue;
		} else {
			if ( in_bracket ) {
				continue;
			} else {
				sequence = sequence + c;
			}
		}
	}
	runtime_assert( ! in_bracket );
	return sequence;
}

} // namespace silent
} // namespace io
} // namespace core
