// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/kinematics/FoldTree.hh
/// @brief  Fold tree class
/// @author Phil Bradley


#ifndef INCLUDED_core_kinematics_FoldTree_HH
#define INCLUDED_core_kinematics_FoldTree_HH


// Unit headers
#include <core/kinematics/FoldTree.fwd.hh>

// Package Headers
#include <core/kinematics/Edge.hh>

// utility headers
#include <core/types.hh>
#include <core/sequence/SequenceMapping.fwd.hh>
#include <utility/vector1.fwd.hh>
#include <utility/pointer/ReferenceCount.hh>
#include <core/util/OStream.fwd.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>

// // C++ Headers
#include <cassert>
#include <vector>


namespace core {
namespace kinematics {


///@brief the fold_tree -- a residue-based treelike representation of molecules
///
///@note all the derived data is "mutable", so that we can
/// update them as needed on the fly inside "const" member functions
/// this emphasizes that the core data is just the edge_list_
///
///@note see @ref foldtree_overview "FoldTree overview and concepts" for details
class FoldTree : public utility::pointer::ReferenceCount
{

public:
	// types
	typedef std::vector< Edge > EdgeList;
	typedef EdgeList::iterator iterator;
	typedef EdgeList::const_iterator const_iterator;

	// MOVED TO CLASS Edge
	//static int const PEPTIDE = -1; // must be negative, see Edge::is_jump()
	//static int const CHEMICAL = -2; // for fold-tree edges that connect two chemically-bound residues

	/// @brief constructor
	FoldTree():
		new_topology (true), // new_topology == true ==> new_order == true
		nres_(0),
		num_jump_(0)
	{}

	/// @brief constructor, makes a simple foldtree
	FoldTree( int const nres_in ):
		new_topology (true), // new_topology == true ==> new_order == true
		nres_(0),
		num_jump_(0)
	{
		simple_tree( nres_in );
	}

	/// @brief operator=
	/// @note this version doesnt copy any of the derived data!
	// will this be too slow? it will trigger re-calculating of everything
	// every time we reject a move....
	FoldTree & operator =( FoldTree const & src )
	{
		edge_list_ = src.edge_list_;
		new_topology = true;
		return *this;
	}

	// non-modifying access /////////////////////////////////////////////////////

	/// @brief number of edges in the tree
	inline
	int
	size() const
	{
		return edge_list_.size();
	}

	/// @brief begin iterator of the edge_list
	inline
	const_iterator
	begin() const
	{
		return edge_list_.begin();
	}

	/// @brief end iterator of the edge_list
	inline
	const_iterator
	end() const
	{
		return edge_list_.end();
	}

	// modifiers //////////////////////////////////////////////////////////
	// add,delete edges:

	/// @brief add an edge into the fold tree
	void
	add_edge(
		int const start,
		int const stop,
		int const label
	);
	/// @brief especially useful for chemical edge construction
	void
	add_edge(
		int const start,
		int const stop,
		std::string const & start_atom,
		std::string const & stop_atom
	);
	/// @brief add an edge
	void
	add_edge( Edge const & new_edge );
	/// @brief delete an edge in the fold tree by iterator
 	void delete_edge( iterator edge );
	/// @brief delete an edge in the fold tree by example edge
	void delete_edge( Edge const & edge );
	/// @brief find an edge in fold tree and delete it
	void delete_unordered_edge( int const start, int const stop,int const label);
	/// @brief change label index of an edge in fold tree
	void update_edge_label( int const start, int const stop,
													int const old_label, int const new_label );
	/// @brief return edge label of edge<start,stop>.
	int edge_label( int const start, int const stop );
	/// @brief clear fold tree data
	void clear() {edge_list_.clear(); new_topology=true;}
	/// @brief renumber the "jump" edges in fold tree
	void renumber_jumps();
	/// @brief delete edges with start==stop, allowing 1->1 edge for single residue foldtree
	void delete_self_edges();
	/// @brief delete vertices that are no longer necessary any more
	void delete_extra_vertices();
	/// @brief delete a continuous segment from the fold tree
	void delete_segment( int const seg_begin, int const seg_end );

	/// @brief delete an arbitrary residue. Will rearrange topology if necessary.
	void
	delete_seqpos( int const seqpos );

	/// @brief brief  Insert a polymer residue at position seqpos
	void
	insert_polymer_residue(
		int const seqpos,
		bool const join_lower,
		bool const join_upper
	);

	/// @brief  inserts a residue attached only by a jump. precondition is that seqpos-1 is a cutpoint
	/// @note that anchor_pos is wrt the current numbering system (ie before insertion)
	void
	insert_residue_by_jump(
		int const seqpos,
		int anchor_pos,
		std::string const& anchor_atom = "",
		std::string const& root_atomno = ""
	);

  /// @brief  Insert a fold_tree as a subtree
  void
  insert_fold_tree_by_jump(
    FoldTree const & subtree,
    int const insert_seqpos,               // rsd 1 in subtree goes here
    int const insert_jumppos,              // jump 1 in subtree goes here
    int const anchor_pos,                  // in the old numbering system
    int anchor_jump_number = 0,            // in the new jump numbering system, default=0
    std::string const & anchor_atom = "",  // could be ""
    std::string const & root_atom   = ""   // ditto
  );

	/// @brief  Renumber all vertices according to an input sequence mapping
	void
	apply_sequence_mapping( sequence::SequenceMapping const & old2new );

	/// @brief add a new jump to an existing fold tree
	int new_jump( int const jump_pos1, int const jump_pos2,
								int const cutpoint );

	void
	new_chemical_bond(
										int const anchor_pos,
										int const root_pos,
										std::string const & anchor_atom,
										std::string const & root_atom,
										int const new_cutpoint
										);

	// ways to build an entire tree:

	/// @brief simple_tree makes a 1-edge tree: (1,total_residue,PEPTIDE)
	void simple_tree( int const nres_in ); // makes a 1-->total_residue tree

	/// @brief returns true if this is a simple FoldTree, false otherwise.
	bool is_simple_tree() const;

	///  @brief builds a tree from a list of jump_points and radom cut points based on some biased probability
	bool random_tree_from_jump_points(
		int const nres_in, int const num_jump_in,
		ObjexxFCL::FArray2D_int const & jump_point,
		ObjexxFCL::FArray1D_float const & cut_bias,
		int const root_in=1,
		bool const allow_jump_at_1_or_NRES = false
	);

	///  @brief builds a tree from a list of jump_points and radom cut points based on some biased probability,
	///             and any user-defined obligate cutpoints.
	bool random_tree_from_jump_points(
		int const nres_in, int const num_jump_in,
		ObjexxFCL::FArray2D_int const & jump_point,
		std::vector< int > const & obligate_cut_points,
		ObjexxFCL::FArray1D_float const & cut_bias,
		int const root_in = 1,
		bool const allow_jump_at_1_or_NRES = false );

	/// @brief construct a fold tree from listed jump points and cutpoints
	bool tree_from_jumps_and_cuts(
		int const nres_in, int const num_jump_in,
		ObjexxFCL::FArray2D_int const & jump_point,
		ObjexxFCL::FArray1D_int const & cuts,
		int const root_in = 1,
		bool const verbose = false
	);

	/// @brief append a new residue to the tree, either by a jump or as a continuation of a peptide segment
	void
	append_residue(
		bool const attach_by_jump = false,
		int const jump_anchor_residue = 0,
		std::string const& jump_upstream_atom = "",
		std::string const&  jump_downstream_atom = ""
	);

	/// @brief append a new residue to the tree using a chemical (APL-style) connection
	void
	append_residue_by_chemical_bond(
		int const anchor_residue,
		std::string const& anchor_atom,
		std::string const& root_atom
	);


	/// @brief reorder the tree to start at residue start_residue.
	bool reorder( int const start_residue );
//	void refold_reorder( int const begin_res, int const end_res,
//										 ObjexxFCL::FArray1D_bool const & jump_moved );


	// check status ////////////////////////////////////////////////////////
	/// @brief check the validity of fold tree
	bool check_fold_tree() const; // foldable?
	/// @brief if the fold tree is connected
	bool connected() const; // connected
	/// @brief chemical edges should have atom info
	bool check_edges_for_atom_info() const;
	/// @brief the staring residue for this jump
	int upstream_jump_residue( int const jump_number ) const;
	/// @brief the stopping residue for this jump
	int downstream_jump_residue( int const jump_number ) const;
	/// partition into two foldtrees by cutting at jump= jump_number

	void
	partition_by_jump(
		int const jump_number,
		FoldTree & f1, //contains N-terminal vertex in jump, like partner1 in partition_by_jump below
		FoldTree & f2
	) const;

	/// @brief partition the fold tree in two parts if the jump is disconnected.
	void
	partition_by_jump(
										int const jump_number,
										ObjexxFCL::FArray1D_bool & partner1
										) const ;

	/// @brief find the corresponding cutpoint for this jump
	/// WARNING: if you look for all cutpoints by cycling thru jump_numbers you may be dissapointed
	/// you will get most likely the same cutpoint for several different jump_numbers
	/// however: the method cutpoint( nr ) will give you the number you are looking for
	int
	cutpoint_by_jump(
									 int const jump_number
									 ) const;

	// these routines are for storing extra information about the jumps
	// you can specify which atoms should be the up/downstream atoms for
	// each jump. Note that this assumes that the topology and order of the
	// fold_tree are fixed, at least until this data is needed.
	//
	// the data are cleared upon re-ordering/wiring of the tree.

	/// @brief the jump atom on the staring side
	std::string
	upstream_atom( int const jump_number ) const;
	/// @brief the jump atom on the stopping side
	std::string
	downstream_atom( int const jump_number ) const;

	/// @brief Returns the direction (n2c, c2n) in which the given (peptide) residue is built during folding.
	int
	get_polymer_residue_direction( int const seqpos ) const;


	/// @brief Returns the edge that builds the residue seqpos. Does not work for root atom (will fail)
	Edge const &
	get_residue_edge( int const seqpos ) const;

	/// @brief  Returns all edges that build a residue directly off of seqpos
	utility::vector1< Edge >
	get_outgoing_edges( int const seqpos ) const;

  /// @brief  Get the number of the jump that builds (connects to) a given residue
  int
  get_jump_that_builds_residue( int const seqpos ) const;

	/// @brief define the specific atoms that should be connected by this jump
	/**
		 This information can then be used in setting up the AtomTree from the
		 FoldTree. Data is stored in the Edge corresponding to this Jump.
		 If not specified, residue-specific defaults will be used.
	*/


	void
	set_jump_atoms(
		int const jump_number,
		std::string const& upstream_atom,
		std::string const& downstream_atom,
		bool bKeepStubInResidue = false
	);

	//version of above but makes it permutation safe!
	//
	void
	set_jump_atoms(
		 int const jump_number,
		 core::Size res1,
		 std::string const&  atom1,
		 core::Size res2,
		 std::string const&  atom2,
		 bool bKeepStubInResidue = false
	);

	/// @brief this reorganizes upstream/downstream atoms of all jumps such that
	/// stubs are N-CA-C
	void put_jump_stubs_intra_residue();

	/// @brief this reorganizes upstream/downstream atoms of jumps that have flag
	/// keep_stub_in_resiue = true such that
	/// stubs are N-CA-C
	void reassign_atoms_for_intra_residue_stubs();


	// stream I/O //////////////////////////////////////////////////////////
	// these two should be inverses!
	// depend on I/O for the class: Edge

	/// @brief input operator
	friend std::istream & operator >>(std::istream & is, FoldTree & t);
	/// @brief output operator
	friend std::ostream & operator <<(std::ostream & os, FoldTree const & t);

	///////////////////////////// derived data //////////////////////
	// routines for retrieving derived data, typically fast

	/// @brief number of residues in the fold tree
	inline Size nres() const;
	/// @brief number of jumps in the fold tree
	inline Size num_jump() const;
	/// @brief starting or stopping residue of a jump edge
	inline int jump_point( int const lower_higher, int const jump_number ) const;
	/// @brief whether this residue is a startting or stopping residue of a jump edge
	inline bool is_jump_point( int const seqpos ) const;
	/// @brief get a cutpoint indexed by jump number
	inline int cutpoint( int const cut ) const;
	/// @brief number of cutpoints in the fold tree
	inline int num_cutpoint() const;
	/// @brief whether is a cutpoint
	inline bool is_cutpoint( int const seqpos ) const;
	/// @brief cutpoint number for this residue
	inline int cutpoint_map( int const seqpos ) const;
	/// @brief get a jump edge by jump number
	Edge const & jump_edge( int const jump_number ) const;
	/// @brief get a jump edge by jump number
	Edge & jump_edge( int const jump_number );
	/// @brief get the jump_nr connected to jump upstream->downstream, returns 0 if not found
	inline core::Size jump_nr( core::Size upstream_res, core::Size downstream_res  ) const;



	/// @brief  Is the tree empty?
	inline
	bool
	empty() const
	{
		return edge_list_.empty();
	}


	/// @brief  Is this seqpos the root?
	inline
	bool
	is_root( int const seqpos ) const
	{
		return ( !edge_list_.empty() && seqpos == begin()->start() );
	}


	/// @brief  Return the root vertex of the tree
	inline
	int
	root() const
	{
		assert( !empty() );
		return edge_list_.begin()->start();
	}


	/// @brief whether a jump exists between these residues
	inline
	bool
	jump_exists( int const pos1, int const pos2 ) const
	{
		for ( Size i=1; i<= num_jump(); ++i ) {
			if ( ( ( pos1 == jump_point(1,i) && pos2 == jump_point(2,i) ) ||
						 ( pos1 == jump_point(2,i) && pos2 == jump_point(1,i) ) ) ) return true;
		}
		return false;
	}



	// 	// old stuff -- still used? yeah abinitio mode- endbias check
	//inline ObjexxFCL::FArray1D_int const & get_jump_edge_count() const;
 	Size count_fixed_residues(
    Size const begin_res,
		Size const size,
		Size & min_edge_count_out
	) const;

	// more routines for getting derived data
	// slow: creates vector each time, just for convenience
	/// @brief get all cutpoints
	utility::vector1< int >
	cutpoints() const;

	/// @brief equal to operator
	friend
	bool
	operator==(
						 FoldTree const & a,
						 FoldTree const & b
						 );


	/// @brief  Slide a polymer cutpoint from one location to another
	void
	slide_cutpoint( Size const current_cut, Size const target_cut );

	/// @brief change an existing jump to start and end in new positions
	void
	slide_jump( Size const jump_number, Size const new_res1, Size const new_res2 );

	/// @brief  Useful for removing a loop modeling jump+cut
	void
	delete_jump_and_intervening_cutpoint(
																			 int jump_begin,
																			 int jump_end
																			 );

	/// @brief  Useful for removing a loop modeling jump+cut
	void
	delete_jump_and_intervening_cutpoint(
																			 int const jump_number
																			 );

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// private methods
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////

private:

	/// @brief delete a polymer residue (non-jump,non-root)
	void
	delete_seqpos_simple( int const seqpos );

	/// @brief delete a root/jump_point residue
	void
	delete_jump_seqpos( int const seqpos );


	/// @brief non-const begin iterator of edge_list
	inline
	iterator
	begin()
	{
		return edge_list_.begin();
	}

	/// @brief non-const end iterator of edge_list
	inline
	iterator
	end()
	{
		return edge_list_.end();
	}

	/// @brief  Helper function
	void
	add_vertex( int const v );

	///////////////////////////// internal book-keeping /////////////
	// maintain things that depend on both topology and/or order
	/// @brief update fold tree topology (when edges are changed) if necessary
	inline bool check_topology() const;
	/// @brief update fold tree order (when edges are same but the order in the edge_list is changed) if necessary
	inline bool check_order() const;
	// private methods for updating derived data
	/// @brief update total number residues in the fold tree
	void update_nres() const;
	/// @brief update number of jumps in the fold tree
	void update_num_jump() const;
	/// @brief update jump residues list
	void update_jump_points() const;
	/// @brief update the index of jump edges in the edge list
	void update_jump_edge() const;
	/// @brief update cutpoints info in the fold tree
	void update_cutpoints() const;
	/// @brief update edge counts info
	void setup_edge_counts() const;


	// private edge_list_ modifiers
	/// @brief update edge labels based on whether edges are separating or not.
	void update_edge_labels();

	/// @brief helper function to try cutting an edge in a tree.
	bool cut_edge( int const cut_point );

	/// @brief cut an edge randomly based on probability without disconnecting fold tree
	bool
	cut_random_edge(
		ObjexxFCL::FArray1D_float const & cut_bias_sum,
		int const nres_in
	);

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// data
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/// the list of edges.
	///@note vector for fast traversal, but re-ordering, deleting are slow.
	EdgeList edge_list_;

	// book-keeping, so we know when to update derived data
	/// @brief edges in the edge_list_ have been changed.
	mutable bool new_topology;
	/// @brief edges in the edge_list_ have been reorderd.
	mutable bool new_order;

	// some derived data ////////////////////////////////////////
	// you dont need to worry about setting any derived data, they are all
	// calculated as needed from the fold_tree.
	// accessed by get_XXX where XXX is the data name
	// note that these are MUTABLE so they can be synced with the
	// edge_list_ on the fly inside "const" access methods
	//

	/// @brief just the largest vertex in edge_list_
	mutable int nres_;
	/// @brief number of jump edges (edges in edge_list_ with label>0)
	mutable int num_jump_;
	/// @brief number of cutpoints in the fold tree.
	/// @note number_cutpoint_ == num_jump_ (if a connected tree)
	mutable int num_cutpoint_;
	/// @brief jump number to jump residue number. dimensioned as (2,num_jump_)
	mutable ObjexxFCL::FArray2D_int jump_point_; // num_jump
	/// @brief whehter a residue is a jump_point, dimensioned as nres_
	mutable ObjexxFCL::FArray1D_bool is_jump_point_; // nres
	/// @brief cutpoint number to cutpoint residue number, dimesioned as num_cutpoint_.
	mutable ObjexxFCL::FArray1D_int cutpoint_; // num_fold_tree_cutpoint
	/// @brief residue number of cutpoint number, 0 if it is not a cutpoint. dimensioned as nres_.
	mutable ObjexxFCL::FArray1D_int cutpoint_map_; // nres
	/// @brief whether a residue is a cutpoint, dimensioned as nres_
	mutable ObjexxFCL::FArray1D_bool is_cutpoint_; // nres
	/// @brief jump number to edge index number in the edge_list_, dimensioned as num_jump_.
	mutable ObjexxFCL::FArray1D_int jump_edge_;
	/// @brief dimensioned as nres_, see setup_edge_counts for more info
	mutable ObjexxFCL::FArray1D_int edge_count; // nres
	/// @brief the minimum number in edge_count and jump_edge_count.
	mutable int min_edge_count;
	/// @brief dimensioned as num_jump, see setup_edge_counts for more info
	mutable ObjexxFCL::FArray1D_int jump_edge_count; // num_jump
	//mutable std::map< int, std::pair< int, int > > jump_atoms;


}; // FoldTree



///////////////////////////////////////////////////////////////////////////////
/// @brief check_topology and check_order handle the updating of data that depends on
/// tree topology and/or tree order.
/// @details any routine that depends on the stored derived data (eg any of the access
/// methods ) should call check_topology() or check_order() at the beginning.
/// @note also, any method function that changes the tree topology or order should set
/// the private data members new_topology and/or new_topology to true.
///

inline
bool
FoldTree::check_topology() const
{
	if ( !new_topology ) return false;
	new_topology = false;
	new_order = true;

	// insert updating calls here: ///////////////////////////
	update_nres();
	update_num_jump();
	update_jump_points();
	update_cutpoints();
	setup_edge_counts();
	//////////////////////////////////////////////////////////
	return true;
}

// returns true if order has changed
/// see details for check_topology
inline
bool
FoldTree::check_order() const
{
	check_topology();
	if ( !new_order ) return false;
	new_order = false;

	// insert updating calls here: ///////////////////////////
	//////////////////////////////////////////////////////////
	update_jump_edge();

	//////////////////////////////////////////////////////////
	return true;
}


///////////////////////////////////////////////////////////////////////////////
///@brief  routines for retrieving the derived data
/// will call check_topology and/or check_order first

inline
Size
FoldTree::nres() const
{
	check_topology();
	return nres_;
}

///////////////////////////////////////////////////////////////////////////////
inline
Size
FoldTree::num_jump() const
{
	check_topology();
	return num_jump_;
}

///////////////////////////////////////////////////////////////////////////////
inline
Edge const &
FoldTree::jump_edge( int const jump_number ) const
{
	check_order();
	return edge_list_[ jump_edge_( jump_number ) ];
}

///////////////////////////////////////////////////////////////////////////////
inline
Edge &
FoldTree::jump_edge( int const jump_number )
{
	check_order();
	return edge_list_[ jump_edge_( jump_number ) ];
}


///////////////////////////////////////////////////////////////////////////////
// is seqpos a jump-point?
inline
bool
FoldTree::is_jump_point( int const seqpos ) const
{
	check_topology();
	return is_jump_point_(seqpos);
}


///////////////////////////////////////////////////////////////////////////////
inline
int
FoldTree::jump_point(
										 int const lower_higher, // = 1 or 2
										 int const jump_number
										 ) const
{
	check_topology();
	return jump_point_( lower_higher, jump_number );
}


///////////////////////////////////////////////////////////////////////////////
inline
int
FoldTree::cutpoint( int const cut ) const
{
	check_topology();
	return cutpoint_(cut);
}


///////////////////////////////////////////////////////////////////////////////
// number of cutpoints
inline
int
FoldTree::num_cutpoint() const
{
	check_topology();
	return num_cutpoint_;
}

/// @brief whether a jump exists between these residues
inline
core::Size
FoldTree::jump_nr( core::Size const pos1, core::Size const pos2 ) const
{
	for ( Size i=1; i<= (Size)num_jump(); ++i ) {
		if ( ( ( pos1 == (Size)jump_point(1,i) && pos2 == (Size)jump_point(2,i) ) ||
				( pos1 == (Size)jump_point(2,i) && pos2 == (Size)jump_point(1,i) ) ) ) return i;
	}
	return 0;
}

///////////////////////////////////////////////////////////////////////////////
/// @brief cutpoint_map is the inverse of cutpoint_, ie it assigns each
/// sequence position that is a cutpoint to the cutpoint number
/// associated with that cutpoint (cutpoints are numbered in increasing
/// residue number from the beginning of the chain)

inline
int
FoldTree::cutpoint_map( int const seqpos ) const
{
	check_topology();
	// sanity
	assert( (  is_cutpoint_(seqpos) && cutpoint_map_(seqpos)  > 0 &&
						 cutpoint_(cutpoint_map_(seqpos)) == seqpos ) ||
					( !is_cutpoint_(seqpos) && cutpoint_map_(seqpos) == 0 ) );
	return cutpoint_map_(seqpos);
}


///////////////////////////////////////////////////////////////////////////////
inline
bool
FoldTree::is_cutpoint( int const seqpos ) const
{
	check_topology();

	if ( seqpos > 0 && seqpos < nres_ ) {
		// 0 and nres count as cutpoints for is_cutpoint but they aren't internal cutpoints
		// in the foldtree so they dont appear in cutpoint_map_
		assert( (  is_cutpoint_(seqpos) && cutpoint_map_(seqpos)  > 0 &&
							 cutpoint_(cutpoint_map_(seqpos)) == seqpos ) ||
						( !is_cutpoint_(seqpos) && cutpoint_map_(seqpos) == 0 ) );
	}
	return is_cutpoint_(seqpos);
}


// ///////////////////////////////////////////////////////////////////////////////
// // this is old code -- not sure whether it's still used...
// //

// inline
// ObjexxFCL::FArray1D_int const &
// FoldTree::get_jump_edge_count() const
// {
// 	check_topology();
// 	return jump_edge_count;
// }

} // namespace kinematics
} // namespace core


#endif // INCLUDED_core_kinematics_FoldTree_HH
