// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/kinematics/util.cc
/// @brief  Kinematics utility functions
/// @author Phil Bradley

// Unit headers
#include <core/kinematics/util.hh>
//#include <core/kinematics/tree/IntraResidueStubJumpAtom.hh>
// Package headers
#include <core/id/DOF_ID_Mask.hh>
#include <core/id/DOF_ID_Map.Pose.hh>
#include <core/kinematics/constants.hh>
#include <core/kinematics/AtomTree.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/kinematics/tree/JumpAtom.hh>
#include <core/kinematics/tree/BondedAtom.hh>
#include <core/id/TorsionID.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/kinematics/types.hh>

// Project headers
#include <core/chemical/VariantType.hh>
#include <core/chemical/ResidueConnection.hh>
#include <core/chemical/ResConnID.hh>

#include <core/conformation/Conformation.hh>
#include <core/conformation/Residue.hh>

#include <core/pack/task/PackerTask.hh>

#include <core/util/Tracer.hh>

// ObjexxFCL headers
// #include <ObjexxFCL/FArray1Da.hh>
// #include <ObjexxFCL/FArray2D.hh>
// #include <ObjexxFCL/FArray3D.hh>
// #include <ObjexxFCL/FArray4D.hh>
// #include <ObjexxFCL/string.functions.hh>
// #include <ObjexxFCL/formatted.o.hh>

// Numeric headers
#include <numeric/conversions.hh>
#include <numeric/random/random.hh>

// Utility headers
//#include <utility/io/orstream.hh>
//#include <utility/io/ozstream.hh>
//#include <utility/io/izstream.hh>

// C++ headers
#include <cassert>
// #include <cstdlib>
// #include <iostream>
// #include <fstream>


namespace core {
namespace kinematics {

typedef utility::vector1< utility::vector1< Size > > Links;
using namespace id;
static numeric::random::RandomGenerator RG(62457); // <- Magic number, do not change it!!!
static util::Tracer TR( "core.kinematics.util");

///////////////////////////////////////////////////////////////////////////////
/// @details  Build an atom-tree from a fold-tree and a set of residues
/// atoms in the tree are allocated with new (on the heap) and held in owning pointers in atom_pointer
/// @note  atom_pointer is cleared at the beginning and then filled with AtomOPs to all the atoms
///

void
build_tree(
	FoldTree const & fold_tree,
	conformation::ResidueCAPs const & residues,
	AtomPointer2D & atom_pointer
)
{
	using conformation::Residue;

	// note -- we clear atom_pointer
	atom_pointer.clear();
	atom_pointer.resize( residues.size() );

	// build first atom. make it a "JumpAtom"
	{
		int const start_pos( fold_tree.root() );
		Residue const & rsd( *(residues[start_pos]) );

		build_residue_tree( residues, rsd, fold_tree, atom_pointer[ start_pos ] );

	}

	// traverse tree, build edges
	for ( FoldTree::const_iterator it = fold_tree.begin(),
					it_end = fold_tree.end(); it != it_end; ++it ) {
		if ( it->is_jump() ) {
			build_jump_edge( *it, residues, atom_pointer );

		} else if ( it->label() == Edge::PEPTIDE ) {
			// build a peptide edge
			build_polymer_edge( *it, residues, atom_pointer );

		} else if ( it->label() == Edge::CHEMICAL ) {
			build_chemical_edge( *it, residues, atom_pointer );

		} else {
			std::cerr << "Failed to identify kinematics::Edge label in core/kinematics/util.cc::build_tree()" << std::endl;
			std::cerr << "Label = " << it->label() << std::endl;
			utility_exit();
		}
	}

	// now guarantee that jump stubs are residue-internal if desired
	for ( FoldTree::const_iterator it = fold_tree.begin(),
					it_end = fold_tree.end(); it != it_end; ++it ) {
		if ( it->is_jump() && it->keep_stub_in_residue() ) {
			promote_sameresidue_child_of_jump_atom( *it, residues, atom_pointer );
		}
	}
}


///////////////////////////////////////////////////////////////////////////////
/// @details root_atomno is the root for the sub atom-tree of this edge.
/// anchor_atomno is the entry point of this sub atom-tree into the main atom-tree.

void
build_jump_edge(
	Edge const & edge,
	conformation::ResidueCAPs const & residues,
	AtomPointer2D & atom_pointer
)
{
	assert( edge.is_jump() );

	int const estart     ( edge.start() );
	int const estop      ( edge.stop () );

	// these may have been set in the edge
	Size anchor_atomno, root_atomno;
	get_anchor_and_root_atoms( *residues[ estart ], *residues[ estop ], edge, anchor_atomno, root_atomno );

	// get the anchor atom
	tree::AtomOP anchor_atom( atom_pointer( AtomID( anchor_atomno, estart ) ) );
	assert( anchor_atom );

	// build the new residue
	build_residue_tree( root_atomno, *residues[ estop ], atom_pointer[ estop ], true /*Jump*/ );

	// now wire in the new residue connection
	anchor_atom->insert_atom( atom_pointer[ AtomID( root_atomno, estop ) ]() );


// 	std::cout << "build_jump_edge: " << edge << ' ' <<  estop << ' ' << root_atomno << ' ' <<
// 		estart << ' ' << anchor_atomno << std::endl;

// 	std::cout << "TREE AFTER BEGIN: " << edge << std::endl;
// 	anchor_atom->show();
// 	assert( anchor_atom->id() == AtomID( anchor_atomno, estart ) );
// 	std::cout << "TREE AFTER END: " << edge << std::endl;

}


///////////////////////////////////////////////////////////////////////////////
/// @details assumes that the start residue of edge has already been built. Traverse
///the polymer edge residue by residue and after building sub atom-tree for this residue,
///attaches the edge's subtree to the anchor_atom in the previous residue.
///
void
build_polymer_edge(
	kinematics::Edge const & edge,
	conformation::ResidueCAPs const & residues,
	AtomPointer2D & atom_pointer
)
{
	int const start( edge.start() );
	int const stop ( edge.stop() );
	int const dir  ( edge.polymer_direction() );

	assert( dir == 1 || dir == -1 );

	AtomID first_anchor;
	for ( int pos=start+dir; pos != stop + dir; pos += dir ) {
		conformation::Residue const & rsd( *residues[pos] );
		int const anchor_pos( pos-dir );
		Size anchor_atomno, root_atomno;
		get_anchor_and_root_atoms( *residues[ anchor_pos ], rsd, edge, anchor_atomno, root_atomno );

		// build the new residue tree, fill in the atom_pointer array
		build_residue_tree( root_atomno, rsd, atom_pointer[pos], false /*Jump*/ );

		tree::AtomOP anchor_atom( atom_pointer( AtomID( anchor_atomno, anchor_pos ) ) );
		tree::AtomOP   root_atom( atom_pointer( AtomID(   root_atomno,        pos ) ) );
		assert( anchor_atom && root_atom );
		anchor_atom->insert_atom( root_atom() );
		//std::cout << "build_polymer_edge: " << edge << ' ' <<  pos << ' ' << root_atomno << ' ' <<
		//	anchor_pos << ' ' << anchor_atomno << std::endl;
		//if ( pos == start+dir ) first_anchor = AtomID( anchor_atomno, anchor_pos );
	}
// 	if ( start != stop ) {
// 		std::cout << "TREE AFTER BEGIN: " << edge << std::endl;
// 		atom_pointer[ first_anchor ]->show();
// 		std::cout << "TREE AFTER END: " << edge << std::endl;
// 	}
}

///////////////////////////////////////////////////////////////////////////////
/// @details assumes that the start residue of edge has already been built. Traverse
///the chemical edge residue by residue and after building sub atom-tree for this residue,
///attaches the edge's subtree to the anchor_atom in the previous residue.
///
void
build_chemical_edge(
	kinematics::Edge const & edge,
	conformation::ResidueCAPs const & residues,
	AtomPointer2D & atom_pointer
)
{

	int const estart     ( edge.start() );
	int const estop      ( edge.stop () );

	// these may have been set in the edge
	Size anchor_atomno, root_atomno;
	get_anchor_and_root_atoms( *residues[ estart ], *residues[ estop ], edge, anchor_atomno, root_atomno );

	build_residue_tree( root_atomno, *residues[ estop ], atom_pointer[ estop ], false /*Jump*/ );

	// get the anchor atom
	tree::AtomOP anchor_atom( atom_pointer( AtomID( anchor_atomno, estart ) ) );
	tree::AtomOP   root_atom( atom_pointer( AtomID(   root_atomno, estop  ) ) );
	assert( anchor_atom && root_atom );

	anchor_atom->insert_atom( root_atom() );
}



///////////////////////////////////////////////////////////////////////////////
/// @brief is atom2 the last atom of our chi angle ?
inline
bool
chi_continuation(
	Size const atom1, // current atom
	Size const atom2, // potential nbr
	utility::vector1< utility::vector1< Size > > const & chi_atoms
)
{
	int const nchi( chi_atoms.size() );

	for ( int i=1; i<= nchi; ++i ) {
		utility::vector1< Size > const & atoms( chi_atoms[i] );
		if ( atom1 == atoms[2] &&
			atom2 == atoms[1] ) return true;

		if ( atom1 == atoms[3] &&
			atom2 == atoms[4] ) return true;
	}
	return false;

}

///////////////////////////////////////////////////////////////////////////////
/// @brief would we be breaking into a chi angle by adding atom2 to the tree now?
inline
bool
chi_interruption(
	Size const atom1, // current atom
	Size const atom2, // potential nbr
	utility::vector1< utility::vector1< Size > > const & chi_atoms,
	utility::vector1< bool > const & is_done
)
{
	int const nchi( chi_atoms.size() );

	//not an interruption if atom1 and atom2 delineate a chi angle.
 	for ( int i=1; i<= nchi; ++i ) {
 		utility::vector1< Size > const & atoms( chi_atoms[i] );

 		if ( atom2 == atoms[2] &&
 				 ( atom1 == atoms[1] || atom1 == atoms[3] ) ) return false;

 		if ( atom2 == atoms[3] &&
 				 ( atom1 == atoms[2] || atom1 == atoms[4] ) ) return false;
 	}


	for ( int i=1; i<= nchi; ++i ) {
		utility::vector1< Size > const & atoms( chi_atoms[i] );

		if ( atom2 == atoms[2] &&
			atom1 != atoms[1] &&
			atom1 != atoms[3] ) return true;

		if ( atom2 == atoms[3] &&
			atom1 != atoms[2] &&
			atom1 != atoms[4] ) return true;

		if ( atom2 == atoms[1] && is_done[ atoms[4] ] ||
			atom2 == atoms[4] && is_done[ atoms[1] ] ) return true;

	}
	return false;

}

///////////////////////////////////////////////////////////////////////////////
// called recursively

// Rule I -- dont enter a chi angle in the middle (atoms 2 or 3 ),
//           and don't enter a chi angle from one side if the other side's atoms have already been added to the tree
//
//
// Rule II -- if you're atom 2 and atom 1 hasn't been done, put that first
//            likewise for atoms 3 and 4
//
// Rule III -- mainchain atoms 1st, tiebreaking by rule II
//
// Rule IV -- heavyatoms before hydrogens, subject to other three rules
//
/// @brief set correct order for how atoms are linked to each other.
///
/// this function is called recursively.
/// @li atom1 is the root of links at the current level.
/// @li full_links store information about UNORDERED bonded neighbors for each atom in this residue.
/// @li is_done indicates which atoms have already been linked.
/// @li is_mainchain, is_chi, is_hydrogen and chi atoms are self explanatory
/// @li new_links store information about ORDERED bonded neighbors (links) for each atom in this residue.
void
setup_atom_links(
	int const atom1,
	Links const & full_links,
	utility::vector1< bool > & is_done,
	utility::vector1< bool > const & is_mainchain,
	utility::vector1< bool > const & is_chi,
	utility::vector1< bool > const & is_hydrogen,
	utility::vector1< utility::vector1< Size > > const & chi_atoms,
	Links & new_links
)
{
	is_done[atom1] = true;

	utility::vector1< Size > const & full_l( full_links[atom1] );
	utility::vector1< Size >       &  new_l(  new_links[atom1] );

	assert( new_l.empty() );

	Size const n_nbrs( full_l.size() );

	if ( n_nbrs == 1 ) {
		Size const nbr( full_l[1] );
		if ( !is_done[nbr] ) {
			// special case -- we have no choice but to add this guy otherwise setup will fail
			new_l.push_back( nbr );
			setup_atom_links( nbr, full_links, is_done, is_mainchain, is_chi, is_hydrogen, chi_atoms, new_links );
		}
		return;
	}

	// top priority -- within a chi angle, and mainchain
	for ( Size i=1; i<= n_nbrs; ++i ) {
		int const atom2( full_l[i] );
		if ( is_done[ atom2 ] ) continue;

		if ( is_mainchain[ atom2 ] && is_chi[ atom1 ] && is_chi[ atom2 ] && chi_continuation( atom1, atom2, chi_atoms ) ) {
			new_l.push_back( atom2 );
			setup_atom_links( atom2, full_links, is_done, is_mainchain, is_chi,
				is_hydrogen, chi_atoms, new_links );
		}
	}

	// next priority -- mainchain
	for ( Size i=1; i<= n_nbrs; ++i ) {
		int const atom2( full_l[i] );
		if ( is_done[ atom2 ] ) continue;

		if ( is_mainchain[ atom2 ] ) {
			new_l.push_back( atom2 );
			setup_atom_links( atom2, full_links, is_done, is_mainchain, is_chi,
				is_hydrogen, chi_atoms, new_links );
		}
	}

	// next priority -- within a chi angle and heavy.
	for ( Size i=1; i<= n_nbrs; ++i ) {
		int const atom2( full_l[i] );
		if ( is_done[ atom2 ] ) continue;

		if ( is_chi[ atom1 ] && is_chi[ atom2 ] && chi_continuation( atom1, atom2, chi_atoms ) && !is_hydrogen[ atom2 ] ) {
			new_l.push_back( atom2 );
			setup_atom_links( atom2, full_links, is_done, is_mainchain, is_chi,
				is_hydrogen, chi_atoms, new_links );
		}
	}

	// next priority -- any heavy atom chi?
	for ( Size i=1; i<= n_nbrs; ++i ) {
		int const atom2( full_l[i] );
		if ( is_done[ atom2 ] ) continue;

		if ( is_chi[ atom2 ] && !is_hydrogen[ atom2 ]) {
			new_l.push_back( atom2 );
			setup_atom_links( atom2, full_links, is_done, is_mainchain, is_chi,
				is_hydrogen, chi_atoms, new_links );
		}
	}


	// next priority -- any chi -- could be hydrogen
	for ( Size i=1; i<= n_nbrs; ++i ) {
		int const atom2( full_l[i] );
		if ( is_done[ atom2 ] ) continue;

		if ( is_chi[ atom2 ] ) {
			new_l.push_back( atom2 );
			setup_atom_links( atom2, full_links, is_done, is_mainchain, is_chi,
				is_hydrogen, chi_atoms, new_links );
		}
	}

	// next priority -- heavyatoms
	for ( Size i=1; i<= n_nbrs; ++i ) {
		int const atom2( full_l[i] );
		if ( is_done[ atom2 ] ) continue;
		if ( is_chi[ atom2 ] && chi_interruption( atom1, atom2, chi_atoms, is_done ) ) continue;
		if ( !is_hydrogen[ atom2 ] ) {
			new_l.push_back( atom2 );
			setup_atom_links( atom2, full_links, is_done, is_mainchain, is_chi,
				is_hydrogen, chi_atoms, new_links );
		}
	}


	// lowest priority -- hydrogens
	for ( Size i=1; i<= n_nbrs; ++i ) {
		int const atom2( full_l[i] );
		if ( is_done[ atom2 ] ) continue;
		if ( is_chi[ atom2 ] && chi_interruption( atom1, atom2, chi_atoms, is_done ) ) continue;
		new_l.push_back( atom2 );
		setup_atom_links( atom2, full_links, is_done, is_mainchain, is_chi,
			is_hydrogen, chi_atoms, new_links );
	}
}

///////////////////////////////////////////////////////////////////////////////
/// @brief Check if this atom neighbor has been black-listed ("CUT_BOND" in params file).
bool
check_good_neighbor( Size const & atom_index, utility::vector1< Size > const & cut_nbrs ) {
	for ( Size kk=1; kk<=cut_nbrs.size(); ++kk )  {
		if ( cut_nbrs[kk] == atom_index ) return false;
	}
	return true;
}

///////////////////////////////////////////////////////////////////////////////
/// @brief simply fill the "links" by adding, for each atom, its bonded neighbors
void
setup_links_simple(
	conformation::Residue const & rsd,
	Links & links
)
{
	int const natoms( rsd.natoms() );

	links.clear();
	links.resize( natoms );

	for ( int atomno1=1; atomno1<= natoms; ++atomno1 ) {
		utility::vector1< Size > const & nbrs( rsd.nbrs( atomno1 ) );
		utility::vector1< Size > const & cut_nbrs( rsd.cut_bond_neighbor( atomno1 ) );
		for ( Size jj=1; jj<= nbrs.size(); ++jj ) {
			if (!check_good_neighbor( nbrs[jj], cut_nbrs ) ) continue;
			links[ atomno1 ].push_back( nbrs[jj] );
		}
	}
}


///////////////////////////////////////////////////////////////////////////////
/// @brief given the root_atomno, set up rules for how other atoms are linked for this residue
///a wrapper function calling setup_atom_links recursively .
void
setup_links(
	conformation::Residue const & rsd,
	int const root_atomno,
	Links & links
)
{
	typedef utility::vector1< bool > BVec;

	//////////////////////////////////
	// get the full list of atom nbrs:
	Links full_links;

	setup_links_simple( rsd, full_links );

	// natoms
	Size const natoms( rsd.natoms() );

	////////////////
	// chi atoms
	BVec is_chi( natoms, false ); // vector bool
	utility::vector1< utility::vector1< Size > > const & chi_atoms
		( rsd.chi_atoms() );
	for ( Size i=1; i<= chi_atoms.size(); ++i ) {
		for ( int j=1; j<= 4; ++j ) {
			is_chi[ chi_atoms[i][j] ] = true;
		}
	}


	//////////////////
	// mainchain atoms
	BVec is_mainchain( natoms, false ); // vector bool
	utility::vector1< Size > const & mainchain( rsd.mainchain_atoms() );
	for ( Size i=1; i<= mainchain.size(); ++i ) {
		is_mainchain[ mainchain[i] ] = true;
	}
	if ( rsd.has_variant_type( chemical::CUTPOINT_LOWER ) ) {
		is_mainchain[ rsd.atom_index("OVL1") ] = true;
		is_mainchain[ rsd.atom_index("OVL2") ] = true;
	}
	if ( rsd.has_variant_type( chemical::CUTPOINT_UPPER ) ) {
		is_mainchain[ rsd.atom_index("OVU1") ] = true;
	}

	////////////
	// hydrogens
	BVec is_hydrogen( natoms, false );
	for ( Size i=1; i<= natoms; ++i ) {
		is_hydrogen[i] = rsd.atom_type(i).is_hydrogen();
	}


	links.clear();
	links.resize( natoms );

	BVec is_done( natoms, false ); // keep track of what's done
	setup_atom_links( root_atomno, full_links, is_done, is_mainchain,  is_chi,
		is_hydrogen, chi_atoms, links );



	// check for an atom that wasn't added
	for ( Size i=1; i<= natoms; ++i ){
		if ( !is_done[ i ] ) {
			std::cout << "Unable to setup links for residue " << std::endl;
			for ( Size j=1; j<= natoms; ++j ) {
				std::cout << rsd.atom_name(j) << ' ' << is_done[j] << ' ' << is_mainchain[j] << ' ' << is_chi[j] << ' ' <<
					is_hydrogen[j] << std::endl;
			}
			utility_exit();
		}
	}
}


/// @details  Determine which atom to use as the root of the root residue in the atomtree.
/// It is sometimes useful to be able to control the atom chosen as the root of the atomtree, eg in graphics.
/// The logic below uses atom_info stored in the foldtree for jump edges emanating from the root residue
/// to override the default atom (if the root residue is a jump_point and said atom_info exists)
///

Size
get_root_residue_root_atomno(
														 conformation::Residue const & rsd,
														 FoldTree const & fold_tree
														 )
{
	Size const seqpos( rsd.seqpos() );
	assert( seqpos == Size( fold_tree.root() ) ); // need to refactor foldtree to use Size instead of int

	Size root_atomno = get_root_atomno( rsd, dir_jump ); // the default setting

	if ( fold_tree.is_jump_point( seqpos ) ) {
		for ( Size i=1; i<= fold_tree.num_jump(); ++i ) {
			Edge const & edge( fold_tree.jump_edge( i ) );
			if ( seqpos == Size(edge.start()) && edge.has_atom_info() ) {
				root_atomno = rsd.atom_index( edge.upstream_atom() );
				TR.Debug <<	"Using jump " << i <<
					" anchor atom as atomtree root " << edge.upstream_atom() << std::endl;
				break;
			}
		}
	}
	return root_atomno;
}


/// @details  A wrapper function to build an atom-tree for a residue. Uses information from the foldtree to
/// find the proper root atom if it is not defined, and determine the direction in which to build the residue.
/// The goal of this function is to allow the Conformation to rebuild parts of the AtomTree in a way that is
/// compatible with what would be built if we erased the entire atomtree and rebuilt it using the foldtree.
/// Also used to build the atomtree for the root residue when we are building the atomtree from scratch.

void
build_residue_tree(
	conformation::ResidueCAPs const & residues,
	conformation::Residue const & rsd,
	FoldTree const & fold_tree,
	AtomPointer1D & atom_ptr
)
{
	// determine root_atomno and whether root_atom is a jump_atom

	bool root_atom_is_jump_atom( false );
	int root_atomno(0);

	int const seqpos( rsd.seqpos() );

	if ( seqpos == fold_tree.root() ) {
		root_atom_is_jump_atom = true;

		root_atomno = get_root_residue_root_atomno( rsd, fold_tree );

	} else {
		// seqpos is not the root of the fold_tree
		Edge const & edge( fold_tree.get_residue_edge( seqpos ) ); // the edge that builds seqpos

		if ( edge.is_peptide() ) {
			// peptide edge
			root_atomno = get_root_atomno( rsd, edge.polymer_direction() ); // the default setting
		} else if ( edge.is_jump() ) {
			// jump edge
			root_atom_is_jump_atom = true;
			if ( edge.has_atom_info() ) {
				root_atomno = rsd.atom_index( edge.downstream_atom() );
			} else {
				root_atomno = get_root_atomno( rsd, dir_jump ); // the default setting
			}
		} else {
			//// chemical edge -- atomnumbers of connections should be programmed in
			//assert( edge.has_atom_info() );
			//root_atomno = rsd.atom_index( edge.downstream_atom() );
			// Connection atoms are not always present in the edge, but might be;  else use defaults.
			int const estart     ( edge.start() );
			int const estop      ( edge.stop () );
			Size anchor_atno, root_atno;
			get_anchor_and_root_atoms( *residues[ estart ], *residues[ estop ], edge, anchor_atno, root_atno );
			root_atomno = (int) root_atno; // stupid Size/int confusion...
		}
	}
	assert( root_atomno );

	// now call the main build_residue_tree function
	build_residue_tree( root_atomno, rsd, atom_ptr, root_atom_is_jump_atom );
}

///////////////////////////////////////////////////////////////////////////////
///\brief set up a local atom-tree for a residue from the defined root atom.
void
build_residue_tree(
	int const root_atomno,
	conformation::Residue const & rsd,
	AtomPointer1D & atom_ptr,
	bool const root_is_jump_atom
)
{
	Size const natoms( rsd.natoms() );

	atom_ptr.clear();
	atom_ptr.resize( natoms ); // default C-TOR for AtomOP is 0 initialized

	// setup the atom nbrs so that the tree will match the desired torsion
	// angles
	Links links;
	setup_links( rsd, root_atomno, links );

	// now build the residue atom-tree using the recursive function add_atom
	add_atom( root_atomno, rsd.seqpos(), links, atom_ptr, root_is_jump_atom );

	// fill in the coordinates from the residue into the atomtree
	for ( Size i=1; i<= natoms; ++i ) {
		atom_ptr[i]->xyz( rsd.atom(i).xyz() );
	}
}
///////////////////////////////////////////////////////////////////////////////
// wrapper for the add_atom recursive function which doesn't know anything
// about Residues
//
/// @details recursively called until all atoms in this reside are added.
/// @note The Atoms that are allocated within this function are owned by the atom_ptr
/// which is a container of AtomOP's (see AtomPointer.fwd.hh)
///
/// @note  Returns a raw pointer for internal tree links but memory management is handled by atom_ptr
///


tree::Atom*
add_atom(
	int const atomno,
	int const seqpos,
	Links const & links,
	AtomPointer1D & atom_ptr, // owns all the newly allocated atoms
	bool const add_jump_atom
)
{
	using namespace tree;

	// create new atom
	AtomOP const atom_p( add_jump_atom ? static_cast< Atom* >( new JumpAtom()) : static_cast< Atom* >(new BondedAtom()));

	// fill in the atom_ptr data
	assert( atom_ptr[ atomno ] == 0 );
	atom_ptr[ atomno ] = atom_p;

	// set the atom_id information
	atom_p->id( AtomID( atomno, seqpos ) );
	atom_p->parent( 0 );

	utility::vector1< Size > const & nbrs( links[ atomno ] );
	for ( Size i=1, i_end = nbrs.size(); i<= i_end; ++i ) {
		int const nbr( nbrs[i] );
		if ( atom_ptr[ nbr ] != 0 ) continue;
		atom_p->append_atom( add_atom( nbr, seqpos, links, atom_ptr, false ));
	}

	return atom_p();
}

/////////////////////////////////////////////////////////////////////////////
///\brief get the root atom for building residue atom-tree given the folding direction "dir"
int
get_root_atomno(
	conformation::Residue const & rsd,
	int const dir // +1, -1, or "dir_jump"
)
{
	// if dir == 1 or -1, we are building a contiguous edge ("peptide edge")
	// in the fold_tree ( ==> residue should be a polymer type? )
	int const forward( 1 );
	int const backward( -1 );


	if ( dir == forward ) {
		// N for proteins, P for DNA
		if ( rsd.is_polymer() ) {
			if ( rsd.is_lower_terminus() ) {
				// this is a little strange to be folding through a terminal residue but kinematics doesn't
				// have to correlate perfectly with chemical
				return rsd.mainchain_atom(1);
			} else {
				return rsd.lower_connect_atom(); //mainchain_atoms()[1];
			}
		} else {
			return get_root_atomno( rsd, dir_jump );
		}
	} else if ( dir == backward ) {
		if ( rsd.is_polymer() ) {
			if ( rsd.is_upper_terminus() ) {
				// this is a little strange to be folding through a terminal residue but kinematics doesn't
				// have to correlate perfectly with chemical
				return rsd.mainchain_atom( rsd.mainchain_atoms().size() );
			} else {
				// C for proteins, O3* for DNA
				return rsd.upper_connect_atom(); //mainchain_atoms()[ rsd.mainchain_atoms().size() ];
			}
		} else {
			return get_root_atomno( rsd, dir_jump );
		}
	} else {
		assert( dir == dir_jump );
		// default for jumps, use the N-terminal attachment?
		if ( rsd.mainchain_atoms().empty() ) {
			// SHORT TERM HACK -- need to add some logic for root atomno
			return 1;
		} else {
			// PB 10/29/07 - changing to use an interior mainchain atom
			//
			// old choice of mainchain_atoms()[1] meant that changing omega of jump_pos-1 would propagate
			// C-terminal to jump_pos, which is bad for loop modeling where the usual assumption was that
			// we can use loop_begin-1 and loop_end+1 as jump points
			//
			Size const nbb( rsd.n_mainchain_atoms() ); //       1  2  3  4  5  6  -- nbb
			Size const mainchain_index( ( nbb-1 )/2 + 1 ); //   1  1  2  2  3  3  -- mainchain_index
			return rsd.mainchain_atoms()[ mainchain_index ];
			//return rsd.mainchain_atoms()[1];
		}
	}
}



/// @details  Get the atom-index of the atom to which the residue at position seqpos should be anchored in
/// constructing the atomtree.

int
get_anchor_atomno( conformation::Residue const & anchor_rsd, Size const seqpos, FoldTree const & fold_tree )
{
	int anchor_atomno(0);
	assert( seqpos != (Size)fold_tree.root() );

	Edge const & edge( fold_tree.get_residue_edge( seqpos ) );

	if ( edge.is_jump() ) {
		// jump edge
		assert( (seqpos == (Size)edge.stop()) && ((Size)anchor_rsd.seqpos() == (Size)edge.start()) );
		if ( edge.has_atom_info() ) {
			anchor_atomno = anchor_rsd.atom_index( edge.upstream_atom() );
		} else {
			anchor_atomno = get_anchor_atomno( anchor_rsd, dir_jump );
		}
	} else if ( edge.is_peptide() ) {
		// peptide edge
		int const dir( edge.polymer_direction() );
		assert( anchor_rsd.seqpos() == seqpos-dir );
		anchor_atomno = get_anchor_atomno( anchor_rsd, dir );
	} else {
		// chemical edge
		assert( seqpos == static_cast< Size >( edge.stop() ) && anchor_rsd.seqpos() == static_cast< Size >( edge.start() ) && edge.has_atom_info() );
		anchor_atomno = anchor_rsd.atom_index( edge.upstream_atom() );
	}
	assert( anchor_atomno );
	return anchor_atomno;
}

///////////////////////////////////////////////////////////////////////////////
int
get_anchor_atomno(
	conformation::Residue const & rsd,
	int const dir // forward(1), backward(-1), or "dir_jump"
)
{
	// if dir == 1 or -1, we are building a contiguous edge ("peptide edge")
	// in the fold_tree ( ==> residue should be a polymer type? )
	int const forward( 1 ), backward( -1 );

	if ( dir == forward ) {
		// C for proteins, O3* for DNA
		if ( rsd.is_polymer() ) {
			if ( rsd.is_upper_terminus() ) {
				// this is a little strange to be folding through a terminal residue but kinematics doesn't
				// have to correlate perfectly with chemical
				return rsd.mainchain_atom( rsd.mainchain_atoms().size() );
			} else {
				return rsd.upper_connect_atom();
			}
		} else {
			return get_anchor_atomno( rsd, dir_jump );
		}
	} else if ( dir == backward ) {
		// N for proteins, P for DNA
		if ( rsd.is_polymer() ) {
			if ( rsd.is_lower_terminus() ) {
				// this is a little strange to be folding through a terminal residue but kinematics doesn't
				// have to correlate perfectly with chemical
				return rsd.mainchain_atom(1);
			} else {
				return rsd.lower_connect_atom();
			}
		} else {
			return get_anchor_atomno( rsd, dir_jump );
		}

	} else {
		assert( dir == dir_jump );
		// default for jumps, use the N-terminal attachment?
		if ( rsd.mainchain_atoms().empty() ) {
			// SHORT TERM HACK -- need to add some logic for root atomno
			return 1;
		} else {
			// PB 10/29/07 - changing to use an interior mainchain atom
			Size const nbb( rsd.n_mainchain_atoms() ); //       1  2  3  4  5  6  -- nbb
			Size const mainchain_index( ( nbb-1 )/2 + 1 ); //   1  1  2  2  3  3  -- mainchain_index
			return rsd.mainchain_atoms()[ mainchain_index ];
			//return rsd.mainchain_atoms()[1];
		}
	}
}

void
get_chemical_root_and_anchor_atomnos(
	conformation::Residue const & rsd_anchor,
	conformation::Residue const & rsd_root,
	Size & anchor_atom_no,
	Size & root_atom_no
)
{
	assert( rsd_anchor.is_bonded( rsd_root ) );

	Size first_connection_id = rsd_anchor.connections_to_residue( rsd_root )[ 1 ];
	chemical::ResConnID first_connection = rsd_anchor.connect_map( first_connection_id );
	anchor_atom_no = rsd_anchor.residue_connection( first_connection_id ).atomno();
	root_atom_no    = rsd_root.residue_connection( first_connection.connid() ).atomno();
}

///////////////////////////////////////////////////////////////////////////////
void
setup_dof_to_torsion_map(
	pose::Pose const & pose,
	DOF_ID_Map< TorsionID > & dof_map
)
{
	Size const n_res( pose.n_residue() );

	// Set DOF mask size and initialize to an invalid TorsionID
	core::id::initialize( dof_map, pose, BOGUS_TORSION_ID );

	for ( Size i = 1; i <= n_res; ++i ) {
		conformation::Residue const & rsd( pose.residue(i) );

		// first the backbone torsion angles
		{
			// PHIL note the Residue-based helper functions you need for this
			// PHIL also note the pose.conformation() interface
			int const n_torsions( rsd.mainchain_atoms().size() );
			for ( int j=1; j<= n_torsions; ++j ) {
				TorsionID const tor_id( i, BB, j );
				DOF_ID const & id( pose.conformation().dof_id_from_torsion_id( tor_id ) );
				if ( id.valid() ) {
					dof_map[ id ] = tor_id;
				}
			}
		}

		{
			// PHIL note the Residue-based helper functions you need for this
			// PHIL also note the pose.conformation() interface
			int const n_torsions( rsd.nchi() );
			for ( int j=1; j<= n_torsions; ++j ) {
				TorsionID const tor_id( i, CHI, j );
				DOF_ID const & id( pose.conformation().dof_id_from_torsion_id( tor_id ) );
				if ( id.valid() ) {
					dof_map[ id ] = tor_id;
				}
			} // j=1,chi-torsions
		} // scope


	} // i=1,n_res

	for ( Size i = 1; i <= pose.fold_tree().num_jump(); ++i ) {
		for ( int j=1; j<= 6; ++j ) {
			TorsionID const tor_id(i,JUMP,j);
			DOF_ID const & id( pose.conformation().dof_id_from_torsion_id( tor_id ) );
			dof_map[ id ] =tor_id;
		}
	} // i=1.num_jump
}


///////////////////////////////////////////////////////////////////////////////
///
void
setup_dof_mask_from_move_map(
	MoveMap const & mm,
	pose::Pose const & pose,
	DOF_ID_Mask & dof_mask
)
{
	Size const n_res( pose.n_residue() );

	// Set DOF mask size and initialize to false
	core::id::initialize( dof_mask, pose, false );

	// DOF_Type defaults
	// could/should do this with a loop over types?
	// currently ignoring rb types, set these individually
	// of by jump number
	dof_mask.set( PHI  , mm.get( PHI   ) );
	dof_mask.set( THETA, mm.get( THETA ) );
	dof_mask.set( D    , mm.get( D     ) );

	// bb & chi
	bool const PHI_default( mm.get( PHI ) );
	//bool const vary_omega( mm.vary_omega() );

	for ( Size i = 1; i <= n_res; ++i ) {
		conformation::Residue const & rsd( pose.residue(i));

		// first the backbone torsion angles
		//bool const bb_move( mm.get_bb(i) );
		//if ( bb_move && !PHI_default ) {
		{
			// PHIL note the Residue-based helper functions you need for this
			// PHIL also note the pose.conformation() interface
			int const n_torsions( rsd.mainchain_atoms().size() );
			for ( int j=1; j<= n_torsions; ++j ) {
				bool const mm_setting( mm.get( TorsionID(i,BB,j) ) );
				if ( mm_setting == PHI_default ) continue;
				DOF_ID const & id
					( pose.conformation().dof_id_from_torsion_id(TorsionID(i,BB,j)));
				if ( id.valid() ) {
					//if ( rsd.is_protein() ) {
					//	assert( n_torsions == 3 );
					//	if ( j == 3 && !vary_omega ) continue; // MAGIC NUMBER
					//}
					dof_mask[ id ] = mm_setting;
				} else {
					// probably just a terminal/chainbreak torsion
					//std::cout << "WARNING: Unable to find atom_tree atom for this " <<
					//" Rosetta torsion angle: " << i << ' ' << j << std::endl;
				}
			} // j=1,bb-torsions
		} // if ( bb_move )

		//bool const chi_move( mm.get_chi(i) );
		//if ( chi_move && !PHI_default ) {
		{
			// PHIL note the Residue-based helper functions you need for this
			// PHIL also note the pose.conformation() interface
			int const n_torsions( rsd.nchi() );
			for ( int j=1; j<= n_torsions; ++j ) {
				bool const mm_setting( mm.get( TorsionID(i,CHI,j) ) );
				if ( mm_setting == PHI_default ) continue;
				DOF_ID const & id
					( pose.conformation().dof_id_from_torsion_id(TorsionID(i,CHI,j)));
				if ( id.valid() ) {
					dof_mask[ id ] = mm_setting;
				} else {
					TR.Warning << "WARNING: Unable to find atom_tree atom for this " <<
						" Rosetta chi angle: " << i << ' ' << j << std::endl;
				}
			} // j=1,chi-torsions
		} // if ( chi_move )


	} // i=1,n_res

	for ( core::Size i=1; i<= pose.num_jump(); ++i ) {
		if ( mm.get_jump(i) ) {
			for ( int j=1; j<= 6; ++j ) {
				DOF_ID const & id
					( pose.conformation().dof_id_from_torsion_id(TorsionID(i,JUMP,j)));
				dof_mask[ id ] = true;
			}
		}
	} // i=1.num_jump


	/////////////////////////
	// DOF's set individually
	for ( MoveMap::DOF_ID_Map::const_iterator it=mm.dof_id_begin(), it_end=mm.dof_id_end();
				it != it_end; ++it ) {
		dof_mask[ it->first ] = it->second;
	}


} // setup_dof_mask_from_move_map


/////////////////////////////////////////////////////////////////////////////
///@details Modify a movemap so that packable residues from a PackerTask have mobile chis.  This function iterates through the task to find the packable residues and sets those residues' chis to mobile in the MoveMap.  It modifies your input movemap!
///@author Steven Lewis
void
modify_movemap_from_packertask(
	core::kinematics::MoveMap & mm,
	core::pack::task::PackerTask const & task
)
{
	int const nres = task.total_residue();

	for (int i = 1; i <= nres; ++i) {
		if ( task.pack_residue( i ) ) {
			mm.set_chi( i, true );
		}
	}

}//modify_movemap_from_packertask

/////////////////////////////////////////////////////////////////////////////
/// \brief pick a postion in n_res as the cutpoint
///
///  this is done based on probability info stored in cut_bias_sum.
/// This function is used during fold_tree construction.
int
pick_loopy_cutpoint(
	Size const n_res,
	ObjexxFCL::FArray1D_float const & cut_bias_sum
)
{

	float r = RG.uniform() * cut_bias_sum( n_res );

	int cutpoint( 0 );

	for ( Size i = 1; i <= n_res; ++i ) {
		if ( r > cut_bias_sum(i-1) && r <= cut_bias_sum(i) ) {
			cutpoint = i;
		}
	}

	if ( cutpoint == 0 ) {
		TR.Warning << "pick loopy cutpoint = 0! setting = 1" << std::endl;
		cutpoint = 1;
	}

	return cutpoint;
}


///////////////////////////////////////////////////////////////////////////////
//
// right now, this is used by the pose to setup a mapping which is
// passed in to the atomtree when replacing one residue with another
//
//
// maybe the behavior should really be to identify atoms with the same
// name in the two residues and map them...
//
// oh well, this will work for rewiring backbone connections properly;
// basically the atomtree will use this mapping when it is replacing
// the atoms of rsd1 with the subtree formed by the atoms of rsd2
// for this to work, all outgoing connections from the rsd1 atoms have
// to have their rsd1-atom represented in this map, so the atomtree
// knows where to attach the child when rsd2 is put in.

///
/// @details only map by atom number, not by identity currently.
void
setup_corresponding_atoms(
	AtomID_Map< AtomID > & atom_map,
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2
)
{

	// PB -- this whole function is going to go away soon, so I'm not really worried about all the hacks

	int const seqpos1( rsd1.seqpos() );
	int const seqpos2( rsd2.seqpos() );

	utility::vector1< Size > const &
		mainchain1( rsd1.mainchain_atoms() ),
		mainchain2( rsd2.mainchain_atoms() );

	assert( mainchain1.size() == mainchain2.size() );

	// special case for virtual residues -- fpd
	if ( mainchain1.size() == 0 &&
			 rsd1.type().name3() == "XXX" && rsd2.type().name3() == "XXX") {
		assert( rsd1.atoms().size() == rsd2.atoms().size() );

		for ( Size i=1, i_end = rsd1.atoms().size(); i<= i_end; ++i ) {
			atom_map.set( AtomID( i, seqpos1 ), AtomID( i, seqpos2 ) );
		}
	} else {
		for ( Size i=1, i_end = mainchain1.size(); i<= i_end; ++i ) {
			atom_map.set( AtomID( mainchain1[i], seqpos1 ),
										AtomID( mainchain2[i], seqpos2 ) );
		}
		if ( rsd1.is_DNA() && rsd2.is_DNA() ) {
			// special case for dna-dna jumps anchored at sidechain atoms
			for ( Size i=1; i<= 4; ++i ) {
				atom_map.set( AtomID( rsd1.chi_atoms(1)[i], seqpos1 ),
											AtomID( rsd2.chi_atoms(1)[i], seqpos2 ) );
			}
		}

		if ( rsd1.is_RNA() && rsd2.is_RNA() ) {
			// By the way, this is a total hack AND SHOULD NOT BE CHECKED IN
			// BEFORE CONSULTING WITH PHIL! -- rhiju
			// special case for rna-rna jumps anchored at sidechain atoms
			if ( rsd1.name1() == rsd2.name1() ) {
				for ( Size i=1; i<= rsd1.natoms(); ++i ) {
					atom_map.set( AtomID( i, seqpos1 ),
												AtomID( i, seqpos2 ) );
				}
			}
		}

	}

	/// Now include atoms involved in residue connections, since these might be anchor points for the atomtree
	for ( Size connid=1; connid<= rsd1.n_residue_connections() && connid <= rsd2.n_residue_connections(); ++connid ) {
		Size const atom1( rsd1.type().residue_connection( connid ).atomno() );
		Size const atom2( rsd2.type().residue_connection( connid ).atomno() );
		if ( rsd1.atom_name( atom1 ) == rsd2.atom_name( atom2 ) ) {
			atom_map.set( AtomID( atom1, seqpos1 ), AtomID( atom2, seqpos2 ) );
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
/// helper function for setup_backrub_atom_tree
tree::Atom *
setup_cloned_atom(
									tree::Atom const *, // old_atom,
									utility::vector1< id::AtomID > const & // exclude
)
{
	utility_exit_with_message("needs to be refactored to meet new tree-building guidelines");
	tree::Atom* new_atom( new tree::BondedAtom() );
	/*
	new_atom->parent(0);
	new_atom->id ( old_atom->id () );
	new_atom->xyz( old_atom->xyz() );

	// add old_atom's (non-exclude-) children
	for ( Size i=0; i< Size(old_atom->n_children()); ++i ) { // 0-indexing AAARRGGHHH!!!
		AtomID const & child_id( old_atom->child(i)->id() );
		if ( std::find( exclude.begin(), exclude.end(), child_id ) == exclude.end() ) {
			new_atom->append_atom( old_atom->child(i)->clone(0) ); // recursively copies child's children
		}
	}
	*/
	return new_atom;
}


///////////////////////////////////////////////////////////////////////////////
/// in principal, this should be the inverse of the build_tree function at the beginning
///
// void
// fold_tree_from_atom_tree(
// 	conformation::ResidueCAPs const & residues
// 	AtomTree const & atom_tree,
// 	FoldTree & fold_tree
// )
// {

// 	/// get all inter-residue connections
// 	utility::vector1< Edge

tree::Atom*
setup_backrub_atom_tree(
												utility::vector1< AtomID >, // mainchain, // make our own local copy
												AtomID const &, // downstream_id, // mainchain child of last mainchain atom
												AtomPointer2D const &, // old_atom_pointer,
												utility::vector1< std::pair< Size, Size > > const &, // edges,
												Size const //first_new_pseudo_residue
)
{

	// In the interest of making the gameguys' deadline for refactoring atomtree I'm not thinking about this right now.
	utility_exit_with_message("This code needs to be refactored to use AtomOPs" );
	/*
	using utility::vector1;

	Vector const pseudo_offset( 0,0,0 );
// 	Vector const pseudo_offset( 0.25, 0.25, 0.25 );

	// constraints on the edge list (note that edges are ordered, ie i,j means edge *from* i *to* j (i-->j)
	//
	// -- 1,nbb is the first edge
	//
	// -- for each (i,j) after the first there exists an earlier edge (x,i) (ie, have to have seen first index already)
	//
	// -- no crossing: no pairs (i,j) and (k,l) with i<k<j<l
	//
	// -- for each edge (a,b), a must be downstream of all edges that contain (a,b)
	//    ie if (i,j) contains (a,b) (meaning that the interval [min(i,j),max(i,j)] contains the corresponding ab interval
	//    then there must be a path of edges leading from j to a (j,p1) (p1,p2) (p2,p3) .... (pN,a)
	//    so eg we cant have 1,10 10,7 7,4 1,4 (violated by (1,4) edge: no path from 10 --> 1 and [1,10] contains [1,4])
	//
	// -- no edges (i,j) with |i-j|<=1
	//

	Size const nbb( mainchain.size() );
	mainchain.push_back( downstream_id ); // so that we dont clone this guy
	assert( edges[1].first == 1 && edges[1].second == nbb );

	/// we're going to add pseudo residues, one for each edge in the edges vector
	Size n(0);

	// book-keeping

	// map from each mainchain_atom to the (possibly empty) list of pseudo-residues
	vector1< vector1< Size > > mainchain_to_pseudo( nbb );

	vector1< bool > seen( nbb, false );

	vector1< Atom* > mainchain_atom_pointer( nbb, 0 );
	vector1< Atom* > pseudo_atom_pointer( edges.size(), 0 );

	// create the root of the tree
	AtomOP root( setup_cloned_atom( old_atom_pointer[ mainchain[1] ], mainchain ) );
	mainchain_atom_pointer[ 1 ] = root;
	seen[ 1 ] = true;

	while ( n < edges.size() ) {
		++n;
		std::pair< Size, Size > const & edge( edges[n] );

		Size const a( edge.first );
		Size const b( edge.second );
		assert( seen[a] );

		// what should our anchor atom be?
		// look at all pseudo rsds associated to a
		Atom * anchor_atom;
		Size anchor(0);
		vector1< Size > const & pseudo_a( mainchain_to_pseudo[a] );
		for ( Size i=1; i<= pseudo_a.size(); ++i ) {
			Size const nn( pseudo_a[i] );
			assert( edges[nn].second == a );
			if ( ( edges[nn].first < b && b < a ) || ( edges[nn].first > b && b > a ) ) {
				anchor = nn; // dont break -- want the smallest segment containing a,b
			}
		}
		if ( anchor ) {
			anchor_atom = pseudo_atom_pointer[ anchor ];
		} else {
			// connect to the authentic a
			anchor_atom = mainchain_atom_pointer[ a ];
		}

		// has b been seen before?
		Atom const * old_b_atom( old_atom_pointer[ mainchain[b] ] );
		if ( !seen[ b ] ) {
			seen[b] = true;
			// add b and b's non-mainchain children
			Atom* b_atom( setup_cloned_atom( old_b_atom, mainchain ) );
			anchor_atom->insert_atom( b_atom ); // NOTE: insert atom
			mainchain_atom_pointer[ b ] = b_atom;
		}


		// add a new pseudo rsd at b's position
		mainchain_to_pseudo[ b ].push_back( n );
		Size const pseudo_b_seqpos( first_new_pseudo_residue + n - 1 );
		AtomID const pseudo_b_id( AtomID( 1, pseudo_b_seqpos ) );
		// delete the old one
		Atom* old_pseudo_b_atom( old_atom_pointer[ pseudo_b_id ] );
		old_pseudo_b_atom->parent()->delete_atom( old_pseudo_b_atom );
		delete old_pseudo_b_atom;
		// create a new one
		Atom* pseudo_b_atom( new BondedAtom() );
		pseudo_b_atom->id( pseudo_b_id );
		pseudo_b_atom->xyz( old_b_atom->xyz() + pseudo_offset );
		anchor_atom->append_atom( pseudo_b_atom );
		pseudo_atom_pointer[ n ] = pseudo_b_atom;
		std::cout << "new pseudo! " << n << ' ' << a << ' ' << b << ' ' << mainchain[a] << ' ' << mainchain[b] << ' ' <<
			pseudo_b_seqpos << ' ' << old_b_atom->id() << ' ' <<
			old_b_atom->xyz()[0] << ' ' <<
			old_b_atom->xyz()[1] << ' ' <<
			old_b_atom->xyz()[2] << std::endl;

		// check if this edge is terminal, if so add all intervening mainchain atoms and their children
		{
			bool terminal( true );
			for ( Size n2=n+1; n2<= edges.size(); ++n2 ) {
				if ( ( edges[n2].first == b ) &&
						 ( a < b && edges[n2].second < b || a > b && edges[n2].second > b ) ) {
					terminal = false;
				}
			}

			if ( terminal ) {
				int const dir( b<a ? 1 : -1 );
				Atom * parent_atom( pseudo_b_atom );
				for ( int c=b+dir; c != (int)a; c += dir ) {
					assert( !seen[c] );

					// add c and c's non-mainchain children
					Atom const * old_c_atom( old_atom_pointer[ mainchain[c] ] );
					Atom* c_atom( setup_cloned_atom( old_c_atom, mainchain ) );
					parent_atom->insert_atom( c_atom ); // at front of list since this is mainchain. may have already added kids
					mainchain_atom_pointer[ c ] = c_atom;
					seen[c] = true;

					parent_atom = c_atom;

				} // walk from b->a adding the mainchain atoms and their children to the tree

			} // if terminal

		} // scope


	} // loop over edges, add one pseudo rsd for each edge

	// confirm that all mainchain atoms have been seen
	for ( Size i=1; i<= nbb; ++i ) {
		assert( seen[i] );
	}
	return root;
	*/
	return 0;

}

/// @details  Determines the anchor and root atom indices based on residue types and the foldtree edge.

void
get_anchor_and_root_atoms(
													conformation::Residue const & anchor_rsd,
													conformation::Residue const & root_rsd,
													Edge const & edge,
													Size & anchor_atomno,
													Size & root_atomno
													)
{
	Size const anchor_pos( anchor_rsd.seqpos() );
	Size const root_pos( root_rsd.seqpos() );

	if ( edge.is_polymer() ) {
		// POLYMER EDGE
		int const dir( edge.polymer_direction() );
		assert( dir == 1 || dir == -1 );
		assert( root_pos == anchor_pos + dir );
		anchor_atomno = get_anchor_atomno( anchor_rsd, dir );
		root_atomno   = get_root_atomno  (   root_rsd, dir );
	} else {
		assert( anchor_pos == Size(edge.start()) && root_pos == Size(edge.stop()) );
		if ( edge.has_atom_info() ) {
			// JUMP OR CHEMICAL W/ ATOM INFO
			anchor_atomno = anchor_rsd.atom_index( edge.upstream_atom() );
			root_atomno   =   root_rsd.atom_index( edge.downstream_atom() );
		} else {
			if ( edge.is_jump() ) {
				// JUMP EDGE
				anchor_atomno = get_anchor_atomno( anchor_rsd, dir_jump );
				root_atomno   =   get_root_atomno(   root_rsd, dir_jump );
			} else if ( edge.is_chemical_bond() ) {
				// CHEMICAL EDGE
				get_chemical_root_and_anchor_atomnos( anchor_rsd, root_rsd, anchor_atomno, root_atomno );
			} else {
				utility_exit_with_message( "Unrecognized edge type!" );
			}
		}
	}
	return;
}


/// @details  Get the incoming connection and all outgoing connections from a residue

void
get_residue_connections(
												conformation::Residue const & new_rsd,
												FoldTree const & fold_tree,
												conformation::ResidueCAPs const & residues,
												id::BondID & new_rsd_in,
												utility::vector1< id::BondID > & new_rsd_out
												)
{

	Size const seqpos( new_rsd.seqpos() );

	// setup incoming connection
	if ( fold_tree.is_root( seqpos ) ) {
		new_rsd_in.atom1 = BOGUS_ATOM_ID;
		new_rsd_in.atom2 = AtomID( get_root_residue_root_atomno( new_rsd, fold_tree ), seqpos );
	} else {
		Size anchor_atomno, root_atomno, anchor_pos;
		Edge const & edge( fold_tree.get_residue_edge( seqpos ) );
		if ( edge.is_polymer() ) anchor_pos = seqpos - edge.polymer_direction();
		else anchor_pos = edge.start();
		get_anchor_and_root_atoms( *residues[ anchor_pos ], new_rsd, edge, anchor_atomno, root_atomno );
		new_rsd_in.atom1 = AtomID( anchor_atomno, anchor_pos );
		new_rsd_in.atom2 = AtomID( root_atomno, seqpos );
	}


	// setup the outgoing connections
	new_rsd_out.clear();
	utility::vector1< Edge > const outgoing_edges( fold_tree.get_outgoing_edges( seqpos ) );
	for ( utility::vector1< Edge >::const_iterator it= outgoing_edges.begin(); it != outgoing_edges.end(); ++it ) {
		Size anchor_atomno, root_atomno;
		Size const root_pos( ( it->is_polymer() ) ? seqpos + it->polymer_direction() : it->stop() );
		get_anchor_and_root_atoms( new_rsd, *residues[ root_pos ], *it, anchor_atomno, root_atomno );
		new_rsd_out.push_back( BondID( AtomID( anchor_atomno, seqpos ), AtomID( root_atomno, root_pos ) ) );
	}
}

/// @details  Helper function for conformation routines.
/// Uses fold_tree to deduce the incoming/outgoing connections for the new residue and the old residue
/// We want it to be the case that the tree we get after this call is the same one that we would have gotten
/// by calling build_tree
///

void
replace_residue_in_atom_tree(
														 conformation::Residue const & new_rsd,
														 //conformation::Residue const & old_rsd,
														 FoldTree const & fold_tree,
														 conformation::ResidueCAPs const & residues,
														 AtomTree & atom_tree
														 )
{
	id::BondID new_rsd_in;
	utility::vector1< id::BondID > new_rsd_out;

	get_residue_connections( new_rsd, fold_tree, residues, new_rsd_in, new_rsd_out );

	// build the new atoms
	AtomPointer1D new_atoms;
	build_residue_tree( residues, new_rsd, fold_tree, new_atoms );

	// now replace the atoms
	atom_tree.replace_residue_subtree( new_rsd_in, new_rsd_out, new_atoms );

	// preserve same-residue jump status if necessary
	if ( fold_tree.is_jump_point( new_rsd.seqpos() ) && !fold_tree.is_root( new_rsd.seqpos() ) ) {
		Edge const & edge( fold_tree.get_residue_edge( new_rsd.seqpos() ) );
		if ( edge.is_jump() && edge.keep_stub_in_residue() ) {
			promote_sameresidue_child_of_jump_atom( edge, residues, atom_tree );
		}
	}
}

// this code assumed we were also being passed old_rsd:
	// optionally debug some things before making atom_tree call
	/**if ( debug ) {
		BondID old_rsd_in;
		vector1< BondID > old_rsd_out;
		get_residue_connections( old_rsd, fold_tree, residues, old_rsd_in, old_rsd_out );
		assert( new_rsd_in.atom1 == old_rsd_in.atom1 && new_rsd_out.size() == old_rsd_out.size() );
		for ( Size i=1; i<= new_rsd_out.size(); ++i ) {
			assert( new_rsd_out[i].atom2 == old_rsd_out[i].atom2 );
		}
		}**/


/// @brief  Inserts/ appends new residue subtree into an existing atomtree
/// @note  The foldtree must already have been changed to reflect the new residue
/// @note  The residues array should already have been inserted into
/// @note  The sequence position of the new residue is deduced from new_rsd.seqpos()
/// @note  This function handles renumbering of the atomtree if necessary
void
insert_residue_into_atom_tree(
	conformation::Residue const & new_rsd,
	FoldTree const & fold_tree,
	conformation::ResidueCAPs const & residues,
	AtomTree & atom_tree
)
{

	Size const seqpos( new_rsd.seqpos() );
	Size const nres( fold_tree.nres() );
	Size const old_nres( atom_tree.size() );
	assert( nres == residues.size() && seqpos <= nres && old_nres == nres-1 );

 	/////////////////////////////////
	// setup for renumbering atomtree
 	utility::vector1< int > old2new( old_nres, 0 );
 	for ( Size i=1; i<= old_nres; ++i ) {
 		if ( i< seqpos ) old2new[i] = i;
 		else old2new[i] = i+1;
 	}

	// renumber the atomtree, fold_tree
	atom_tree.update_sequence_numbering( nres, old2new );
	assert( atom_tree.size() == nres );

	replace_residue_in_atom_tree( new_rsd, fold_tree, residues, atom_tree );

}


///////////////////////////////////////////////////////////////////////////////


void
promote_sameresidue_child_of_jump_atom(
	Edge const & edge,
	conformation::ResidueCAPs const & residues,
	AtomPointer2D const & atom_pointer
)
{
	assert( edge.is_jump() );
	Size root_pos( edge.stop() ), anchor_atomno, root_atomno;
	get_anchor_and_root_atoms( *residues[ edge.start() ], *residues[ root_pos ], edge, anchor_atomno, root_atomno );
	tree::Atom * root_atom( atom_pointer[ AtomID( root_atomno, root_pos ) ]() );
	assert( root_atom->is_jump() );
	tree::Atom * same_residue_child( 0 );
	for ( Size i=0; i< root_atom->n_nonjump_children(); ++i ) {
		tree::Atom * child( atom_pointer[ root_atom->get_nonjump_atom( i )->id() ]() ); // want nonconst, use atom_pointer
		if ( Size(child->id().rsd()) == root_pos ) {
			same_residue_child = child;
			break;
		}
	}
	if ( same_residue_child ) {
		assert( !same_residue_child->is_jump() );
		root_atom->delete_atom( same_residue_child );
		root_atom->insert_atom( same_residue_child );
	} else {
		TR.Warning << "Unable to keep stub in residue: jump_atom has no non-jump, same-residue children!" << std::endl;
	}
}


void
promote_sameresidue_child_of_jump_atom(
	Edge const & edge,
	conformation::ResidueCAPs const & residues,
	AtomTree & atom_tree
)
{
	assert( edge.is_jump() );
	Size root_pos( edge.stop() ), anchor_atomno, root_atomno;
	get_anchor_and_root_atoms( *residues[ edge.start() ], *residues[ root_pos ], edge, anchor_atomno, root_atomno );
	atom_tree.promote_sameresidue_nonjump_child( AtomID( root_atomno, root_pos ) );
}

/// @brief prints something like this ***1***C***1*********2***C********3****C****2********3*****
void
simple_visualize_fold_tree( FoldTree const & fold_tree, std::ostream& out ) {
	for ( Size pos = 1; pos <= fold_tree.nres(); pos++ ) {
		bool special( false );
		if ( fold_tree.is_jump_point( pos ) ) {
			for ( Size jnr=1; jnr<=fold_tree.num_jump(); jnr++ ) {
				if ( (Size) fold_tree.jump_edge( jnr ).start() == pos || (Size)fold_tree.jump_edge( jnr ).stop() == pos ) {
					if ( special ) out << "/";
					out << jnr;
					special = true;
				}
			}
		}
		if ( fold_tree.is_cutpoint( pos ) ) {
			if ( special ) out << "/";
			out << "C";
			special = true;
		}
		if (!special ) out << "*";
	}
	out << std::endl;
}


/// @brief prints something like this ***1***C***1*********2***C********3****C****2********3*****
///                                   **********xxxxxxxxxxxxx************************************
void
simple_visualize_fold_tree_and_movemap( FoldTree const & fold_tree, MoveMap const& mm, std::ostream& out ) {
	std::string move;
	out << "\n";
	for ( Size pos = 1; pos <= fold_tree.nres(); pos++ ) {
		bool special( false );
		if ( fold_tree.is_jump_point( pos ) ) {
			for ( Size jnr=1; jnr<=fold_tree.num_jump(); jnr++ ) {
				if ( (Size) fold_tree.jump_edge( jnr ).start() == pos || (Size)fold_tree.jump_edge( jnr ).stop() == pos ) {
					if ( special ) {
						out << "/";
						move.push_back( '.' );
					}
					out << jnr;
					move.push_back( mm.get_bb( pos ) ? '*' : 'x' );
					special = true;
				}
			}
		}
		if ( fold_tree.is_cutpoint( pos ) ) {
			if ( special ) {
				out << "/";
				move.push_back( '.' );
			}
			out << "C";
			move.push_back( mm.get_bb( pos ) ? '*' : 'x' );
			special = true;
		}
		if (!special ) {
			out << "*";
			move.push_back( mm.get_bb( pos ) ? '*' : 'x' );
		}
	}
	out << "\n" << move;
	out << std::endl;
}

/// @brief prints something like this ***1***C***1*********2***C********3****C****2********3*****
///                                   **********xxxxxxxxxxxxx************************************
void
simple_visualize_fold_tree_and_movemap_bb_chi( FoldTree const & fold_tree, MoveMap const& mm, std::ostream& out ) {
	std::string move;
	std::string move_chi;
	out << "\n";
	for ( Size pos = 1; pos <= fold_tree.nres(); pos++ ) {
		bool special( false );
		if ( fold_tree.is_jump_point( pos ) ) {
			for ( Size jnr=1; jnr<=fold_tree.num_jump(); jnr++ ) {
				if ( (Size) fold_tree.jump_edge( jnr ).start() == pos || (Size)fold_tree.jump_edge( jnr ).stop() == pos ) {
					if ( special ) {
						out << "/";
						move.push_back( '.' );
					}
					out << jnr;
					move.push_back( mm.get_bb( pos ) ? '*' : 'x' );
					move_chi.push_back( mm.get_chi( pos ) ? '*' : 'x' );
					special = true;
				}
			}
		}
		if ( fold_tree.is_cutpoint( pos ) ) {
			if ( special ) {
				out << "/";
				move.push_back( '.' );
				move.push_back( '.' );
			}
			out << "C";
			move.push_back( mm.get_bb( pos ) ? '*' : 'x' );
			move_chi.push_back( mm.get_chi( pos ) ? '*' : 'x' );
			special = true;
		}
		if (!special ) {
			out << "*";
			move_chi.push_back( mm.get_chi( pos ) ? '*' : 'x' );
			move.push_back( mm.get_bb( pos ) ? '*' : 'x' );
		}
	}
	out << "\n" << move << "\n" << move_chi;
	out << std::endl;
}


///@brief linearizes (or defoliates, if you prefer) a FoldTree.  "default" FoldTrees produced by the PDB reader have all chains (peptide edges) starting from jumps relative to residue 1.  This code modifies the tree to instead have all the jumps be relative to the preceding edge.  It is not tested with ligands and will not work with "functional" jumps.  From A to B:
///A:FOLD_TREE  EDGE 1 78 -1  EDGE 1 79 1   EDGE 79 454 -1  EDGE 1 455 2    EDGE 455 540 -1  EDGE 1 541 3    EDGE 541 697 -1
///B:FOLD_TREE  EDGE 1 78 -1  EDGE 78 79 1  EDGE 79 454 -1  EDGE 454 455 2  EDGE 455 540 -1  EDGE 540 541 3  EDGE 541 697 -1
core::kinematics::FoldTree
linearize_fold_tree( core::kinematics::FoldTree const & tree ) {
	core::kinematics::FoldTree newtree;
	for( core::kinematics::FoldTree::const_iterator it(tree.begin()), end(tree.end()); it != end; ++it){
		//if it is not a jump, we don't modify it
		if( !it->is_jump() ) newtree.add_edge(*it);
		//if it is a jump, we move start() to stop-1.  This is naive but works for the intended case.
		else newtree.add_edge(core::kinematics::Edge(it->stop()-1, it->stop(), it->label()));
	}
	return newtree;
}


} // namespace kinematics
} // namespace core
