// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/optimization/AtomTreeMultifunc.hh
/// @brief  Atom tree multifunction class
/// @author Phil Bradley


#ifndef INCLUDED_core_optimization_AtomTreeMultifunc_HH
#define INCLUDED_core_optimization_AtomTreeMultifunc_HH


// Package headers
#include <core/optimization/types.hh>
#include <core/optimization/Multifunc.hh>
#include <core/optimization/MinimizerMap.hh>
#include <core/optimization/atom_tree_minimize.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/util/prof.hh>


namespace core {
namespace optimization {


/// @brief Atom tree multifunction class
class AtomTreeMultifunc : public Multifunc
{

public: // Creation

	// c-tor
	AtomTreeMultifunc(
		pose::Pose & pose_in,
		MinimizerMap & min_map_in,
		scoring::ScoreFunction const & scorefxn_in,
		bool const deriv_check_in = false,
		bool const deriv_check_verbose_in = false
	):
		pose_( pose_in ),
		min_map_( min_map_in ),
		score_function_( scorefxn_in ),
		deriv_check_( deriv_check_in ),
		deriv_check_verbose_( deriv_check_verbose_in )
	{}



	/// @brief Destructor
	inline
	virtual
	~AtomTreeMultifunc()
	{}


public: // Methods


	// func
	virtual
	Real
	operator ()( Multivec const & vars ) const
	{
		PROF_START( util::FUNC );
		min_map_.copy_dofs_to_pose( pose_, vars );
		Real const score( score_function_( pose_ ) );
		PROF_STOP( util::FUNC );
		return score;
	}


	// dfunc
	virtual
	void
	dfunc( Multivec const & vars, Multivec & dE_dvars ) const
	{
		PROF_START( util::DFUNC );
		// in atom_tree_minimize.cc
		atom_tree_dfunc( pose_, min_map_, score_function_, vars, dE_dvars );
		// optional derivative checking
		if ( deriv_check_ ) {
			numerical_derivative_check( min_map_, *this, vars, dE_dvars,
																	deriv_check_verbose_ );
		}
		PROF_STOP( util::DFUNC );
	}

	/// @brief Error state reached; dump out current pdb.
	virtual
	void
	dump( Multivec const & vars ) const;


protected: // accessors for subclasses
	/// non-const since pose_ is modified by calls to operator()
	pose::Pose & pose() const { return pose_; }

	MinimizerMap const & min_map() const { return min_map_; }

	scoring::ScoreFunction const & score_function() const { return score_function_; }

private: // data

	/// non-const since pose_ is modified by calls to operator()
	pose::Pose & pose_;

	/// non-const since min_map_ is modified by calls to dfunc()
	MinimizerMap & min_map_;

	scoring::ScoreFunction const & score_function_;

	bool deriv_check_;
	bool deriv_check_verbose_;

}; // AtomTreeMultifunc


} // namespace optimization
} // namespace core


#endif // INCLUDED_core_optimization_AtomTreeMultifunc_HH
