// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/optimization/LineMinimizer.cc
/// @brief  line minimizer classes
/// @author Phil Bradley
/// @author Jim Havranek


// Unit headers
#include <core/optimization/LineMinimizer.hh>

#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <core/util/Tracer.hh>

// C++ headers
#include <cmath>
//#include <cstdlib>
// #include <cstdio>
#include <iostream>
#include <algorithm>
#include <functional>

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("core.optimization");

namespace core {
namespace optimization {

void
func_1d::dump( Real displacement ) {
	for( uint i =  1 ; i <= _starting_point.size() ; ++i ) {
		_eval_point[i] = _starting_point[i] + ( displacement * _search_direction[i] );
	}
	return _func.dump( _eval_point );
}

	static util::Tracer TR( "core.optimization.LineMinimizer" );


	// Functor'ed up version of accurate line minimization
	/////////////////////////////////////////////////////////////////////////////
	Real
	BrentLineMinimization::operator()(
		Multivec & current_position,
		Multivec & search_direction
	)
	{
		int const problem_size( current_position.size() );

		_num_linemin_calls++;

		Real AX, XX, BX, FA, FX, FB;

		// check magnitude of derivative
		Real derivmax = 0.0;
		for ( int i = 1; i <= problem_size; ++i ) {
			if ( std::abs(search_direction[i]) > std::abs(derivmax) ) {
					derivmax = search_direction[i];
			}
		}

		if ( std::abs(derivmax) <= .0001 ) {
			Real final_value =_func(current_position);
			return final_value; // deriv = 0, return value
		}

		// Construct the one-dimensional projection of the function
		func_1d this_line_func( current_position, search_direction, _func );

		// initial step sizes from our options
		AX = _ax;
		XX = _xx; // initial range for bracketing
		BX = _bx; // now set in namespace in  minimize_set_func

		MNBRAK(AX,XX,BX,FA,FX,FB,this_line_func);

		Real final_value = BRENT(AX,XX,BX,FA,FX,FB,_tolerance,this_line_func);

		for ( int j = 1; j <= problem_size; ++j ) {
			search_direction[j] *= _last_accepted_step;
			current_position[j] += search_direction[j];
		}

//		TR.Info << "Linemin used " << this_line_func.get_eval_count() <<
//				" function calls" << std::endl;

		return final_value;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	BrentLineMinimization::MNBRAK(
		Real & AX,
		Real & BX,
		Real & CX,
		Real & FA,
		Real & FB,
		Real & FC,
		func_1d & func_eval
		) const
	{
		Real DUM, R, Q, U, ULIM, FU;
		Real const GOLD = { 1.618034 };
		Real const GLIMIT = { 100.0 };
		Real const TINY = { 1.E-20 };

		FA = func_eval(AX);
		FB = func_eval(BX);
		if ( FB > FA ) {
			DUM = AX;
			AX = BX;
			BX = DUM;
			DUM = FB;
			FB = FA;
			FA = DUM;
		}
		CX = BX+GOLD*(BX-AX);
		FC = func_eval(CX);

	L1:
		if ( FB >= FC ) {
			R = (BX-AX)*(FB-FC);
			Q = (BX-CX)*(FB-FA);
			U = BX-((BX-CX)*Q-(BX-AX)*R)/(2.*sign(std::max(std::abs(Q-R),TINY),Q-R));
			ULIM = BX+GLIMIT*(CX-BX);
			if ( (BX-U)*(U-CX) > 0.0 ) {
				FU = func_eval(U);
				if ( FU < FC ) {
					AX = BX;
					FA = FB;
					BX = U;
					FB = FU;
					goto L1;
				} else if ( FU > FB ) {
					CX = U;
					FC = FU;
					goto L1;
				}
				U = CX+GOLD*(CX-BX);
				FU = func_eval(U);

			} else if ( (CX-U)*(U-ULIM) > 0. ) {
				FU = func_eval(U);
				if ( FU < FC ) {
					BX = CX;
					CX = U;
					U = CX+GOLD*(CX-BX);
					FB = FC;
					FC = FU;
					FU = func_eval(U);
				}
			} else if ( (U-ULIM)*(ULIM-CX) >= 0. ) {
				U = ULIM;
				FU = func_eval(U);
			} else {
				U = CX+GOLD*(CX-BX);
				FU = func_eval(U);
			}
			AX = BX;
			BX = CX;
			CX = U;
			FA = FB;
			FB = FC;
			FC = FU;
			goto L1;
		}
	} // mnbrak

	/////////////////////////////////////////////////////////////////////////////
	//
	Real
	BrentLineMinimization::BRENT(
		Real const AX,
		Real const BX,
		Real const CX,
		Real & FA,
		Real & FB,
		Real const FC,
		Real const TOL,
		func_1d & func_eval
	)
	{
		Real const brent_abs_tolerance( _abs_tolerance );

		Real BRENT; // Return value

		Real A, B, E,TOL1,TOL2;
		Real V,W,X,FX,FV,FW,XM,R,Q,P,ETEMP,D,U,FU;

		int const ITMAX = { 100 };
		Real const CGOLD = { 0.3819660 };
		Real const ZEPS = { 1.0E-10 };
		//$$$      int func_counter;       // diagnostic only

		A = std::min(AX,CX);
		B = std::max(AX,CX);
		V = BX;
		W = V;
		X = V;
		E = 0.0;
		//     these two initializations added to remove warnings
		D = 0.0; // D will be set first time through loop, when if (E>tol) is false
		BRENT = -999.9;

		//********************************************
		FX = FB;
		if ( A == AX ) {
			FB = FC;
		} else {
			FB = FA;
			FA = FC;
		}

		// pb 3/20/07: this was already commented out:
		//      FX = F(gfrag,X);
		//********************************************
		//$$$      func_counter = 1;

		FV = FX;
		FW = FX;
		for ( int iter = 1; iter <= ITMAX; ++iter ) {
			XM = 0.5*(A+B);
			TOL1 = TOL*std::abs(X)+ZEPS;
			TOL2 = 2.*TOL1;


			//********************************************
			//     here, we exit BRENT if the function varies by less than a
			//     tolerance over the interval
			if ( ( std::abs(FB-FX) <= brent_abs_tolerance ) &&
					 ( std::abs(FA-FX) <= brent_abs_tolerance ) ) break;

			//********************************************
			//     here, we exit BRENT if we have narrowed the search down to a
			//     small change in X
			if ( std::abs(X-XM) <= (TOL2-.5*(B-A)) ) break;

			if ( std::abs(E) > TOL1 ) {
				R = (X-W)*(FX-FV);
				Q = (X-V)*(FX-FW);
				P = (X-V)*Q-(X-W)*R;
				Q = 2.*(Q-R);
				if ( Q > 0.0 ) P = -P;
				Q = std::abs(Q);
				ETEMP = E;
				E = D;
				if ( std::abs(P) >= std::abs(.5*Q*ETEMP) ||
						 P <= Q*(A-X) || P >= Q*(B-X) ) goto L1;
				D = P/Q;
				U = X+D;
				if ( U-A < TOL2 || B-U < TOL2 ) D = sign(TOL1,XM-X);
				goto L2;
			}
		L1:
			if ( X >= XM ) {
				E = A-X;
			} else {
				E = B-X;
			}
			D = CGOLD*E;
		L2:
			if ( std::abs(D) >= TOL1 ) {
				U = X+D;
			} else {
				U = X+sign(TOL1,D);
			}

			// call Minimizer object's 1D function using stored data
			FU = func_eval(U);
			//$$$        ++func_counter;

			if ( FU <= FX ) {
				if ( U >= X ) {
					A = X;
					FA = FX;
				} else {
					B = X;
					FB = FX;
				}
				V = W;
				FV = FW;
				W = X;
				FW = FX;
				X = U;
				FX = FU;
			} else {
				if ( U < X ) {
					A = U;
					FA = FU;
				} else {
					B = U;
					FB = FU;
				}
				if ( FU <= FW || W == X ) {
					V = W;
					FV = FW;
					W = U;
					FW = FU;
				} else if ( FU <= FV || V == X || V == W ) {
					V = U;
					FV = FU;
				}
			}

			if ( iter >= ITMAX ) {
				TR.Error << "BRENT exceed maximum iterations. " << iter << std::endl;
				std::exit( EXIT_FAILURE );
			}
		} // iter=1,ITMAX


		_last_accepted_step = X;
		BRENT = FX;

		return BRENT;
	}

	/////////////////////////////////////////////////////////////////////////////
	//
	Real
	ArmijoLineMinimization::operator()(
		Multivec & current_position,
		Multivec & search_direction
	)
	{

	Real const FACTOR( 0.5 );
	int const problem_size( current_position.size() );
//	Real max_step_limit( _nonmonotone ? 2.0 : 1.0 );
	Real max_step_limit( 1.0 );

	_num_linemin_calls++;

	// Construct the one-dimensional projection of the function
	func_1d this_line_func( current_position, search_direction, _func );

	// Early termination for derivatives (search_direction magnitudes) near zero
	// Please note that the search_direction vector is not normalized
	Real derivmax = 0.0;
	for ( int i = 1 ; i <= problem_size; ++i ) {
		if ( std::abs( search_direction[ i ] ) >
				std::abs( derivmax ) ) derivmax = search_direction[ i ];
	}

	// if ( runlevel >= gush) std::cout << "derivmax," << SS( derivmax ) << std::endl;
	if ( std::abs(derivmax) < .0001 ) {
		Real final_value = this_line_func( 0.0 ); // deriv = 0, return value
		return final_value;
	}

	//initial trial stepsize
	Real init_step( _last_accepted_step / FACTOR );
	if ( init_step > max_step_limit ) init_step = max_step_limit;

	Real final_value = Armijo( init_step, this_line_func );

	for ( int j = 1; j <= problem_size; ++j ) {
		search_direction[j] *= _last_accepted_step;
		current_position[j] += search_direction[j];
	}

//	std::cout << "Linemin used " << this_line_func.get_eval_count() <<
//		" function calls and returns " << final_value << " on step of " << _last_accepted_step << std::endl;

	return final_value;
}

/////////////////////////////////////////////////////////////////////////////
//
Real
ArmijoLineMinimization::Armijo(
	Real init_step,
	func_1d & func_eval
)
{
	// INPUT PARAMETERS: XX,func,gfrag,NF
	// OUTPUT PARAMETERS: NF,FRET
	//
	// Given a function FUNC, and initial stepsize XX
	// such that 0 < XX and FUNC has negative derivative _deriv_sum at 0,
	// this routine returns a stepsize XX at least as good as that
	// given by Armijo rule.
	// Reference:  D.P. Bertsekas, Nonlinear Programming, 2nd ed, 1999, page 29.
	Real const FACTOR( 0.5 );
	Real const SIGMA( 0.1 );
	Real const SIGMA2( 0.8 );

	//std::cout << "func_to_beat is " << _func_to_beat << std::endl;

	Real func_value = func_eval( init_step );
	_num_calls++;

	_last_accepted_step = init_step;

	if (func_value < _func_to_beat+init_step*SIGMA2*_deriv_sum ) {
		Real test_step = init_step/FACTOR;
		Real test_func_value = func_eval( test_step );
		_num_calls++;
		if (test_func_value < func_value) {
			_last_accepted_step = test_step;
			return test_func_value;
		}
		return func_value;
	}

	Real far_step = init_step;
	while (func_value > _func_to_beat + init_step*SIGMA*_deriv_sum) {
		// Abort if function value is unlikely to improve.
		if ( ( init_step <= 1e-5 * far_step ) ||
					(init_step < 1e-8 && func_value >= _func_to_beat)) {
			Real test_step = ( func_value - _func_to_beat ) / init_step;
			TR.Error << "Inaccurate G! step= " << ( init_step ) << " Deriv= " <<
						( _deriv_sum ) << " Finite Diff= " << ( test_step ) << std::endl;
			func_eval.dump( 0.0 );
			func_eval.dump( init_step );
			_last_accepted_step = 0.0;
			return _func_to_beat;
		}

		init_step *= FACTOR*FACTOR;		// faster step decrease
																	// init_step *= FACTOR;
		//std::cout << "func_eval( " << init_step << ")" << std::endl;
		func_value = func_eval( init_step );
		_num_calls++;
	}

	_last_accepted_step = init_step;

	if ( init_step < 0.0 ) {
		TR << "Forced to do parabolic fit!" << std::endl;
		// Parabola interpolate between 0 and init_step for refinement
		Real test_step = -_deriv_sum*init_step*init_step/
					(2*(func_value - _func_to_beat - init_step * _deriv_sum));
		if (test_step > 1e-3*far_step && test_step < far_step) {
			Real test_func_value = func_eval( test_step );
			_num_calls++;
			if ( test_func_value < func_value ) {
				_last_accepted_step = test_step;
				func_value = test_func_value;
			}
		}
	}

	return func_value;
}


} // namespace optimization
} // namespace core
