// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/optimization/LineMinimizer.hh
/// @brief  line minimizer classes
/// @author Phil Bradley
/// @author Jim Havranek


#ifndef INCLUDED_core_optimization_LineMinimizer_HH
#define INCLUDED_core_optimization_LineMinimizer_HH


// Package headers
#include <core/optimization/types.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/optimization/Multifunc.hh>


namespace core {
namespace optimization {

// Functor that stores current position and
// search direction to provide a Real->Real mapping
// for univariate minimization

class func_1d {
	private:
		Multivec const _starting_point;
		Multivec const _search_direction;
		Multivec _eval_point;
		Multifunc const & _func;
		int _eval_count;
	public:
		func_1d( Multivec & start, Multivec & dir, Multifunc const & score_fxn ):
				_starting_point( start ),
				_search_direction( dir ),
				_eval_point( dir.size(), 0.0 ),
	 			_func( score_fxn ), _eval_count( 0 ) {
			assert( _starting_point.size() == _search_direction.size() );
		};
		Real operator() ( Real displacement ) {
			_eval_count++;
			for( uint i =  1 ; i <= _starting_point.size() ; ++i ) {
				_eval_point[i] = _starting_point[i] +
												( displacement * _search_direction[i] );
			}
			return _func( _eval_point );
		};
		void reset_eval_count() { _eval_count = 0; };
		int get_eval_count() { return _eval_count; };
		void dump( Real displacement );
};

/////////////////////////////////////////////////////////
// base class / interface for line minimizers
/////////////////////////////////////////////////////////
class LineMinimizationAlgorithm
{
public:
	LineMinimizationAlgorithm( Multifunc const & score_fxn ) :
		_last_accepted_step( 1.0 ), _func_to_beat( 0.0 ),
		_deriv_sum( 0.0 ),_num_linemin_calls(0),_tolerance( 0.1 ),
		_func( score_fxn ) {};
	virtual Real operator()( Multivec & , Multivec & ){ return 0.0; };
	virtual ~LineMinimizationAlgorithm() {}

	Real _last_accepted_step;
	Real _func_to_beat;
	Real _deriv_sum;
	int _num_linemin_calls;
protected:
	Real const _tolerance;
	Multifunc const & _func;
};

/////////////////////////////////////////////////////////
// concrete line minimizer - Brent's method
/////////////////////////////////////////////////////////

class BrentLineMinimization : public LineMinimizationAlgorithm
{
public:
	BrentLineMinimization( Multifunc const & score_fxn ) :
		LineMinimizationAlgorithm( score_fxn ),
		_ax( 0.0 ), _bx( 0.2 ), _xx( 0.1 ), _abs_tolerance( 0.01 ){};
	virtual Real operator()( Multivec & curr_pos, Multivec & curr_dir );
	void MNBRAK( Real & AX, Real & BX, Real & CX, Real & FA, Real & FB,
		Real & FC, func_1d & func_eval) const;
	Real BRENT( Real const AX, Real const BX, Real const CX, Real & FA,
		Real & FB, Real const FC, Real const TOL, func_1d & func_eval);
	Real _ax;
	Real _bx;
	Real _xx;
	Real _abs_tolerance;
};

/////////////////////////////////////////////////////////
// concrete line minimizer - Armijo's method
/////////////////////////////////////////////////////////

class ArmijoLineMinimization : public LineMinimizationAlgorithm
{
public:
	ArmijoLineMinimization( Multifunc const & score_fxn, bool nonmonotone ) :
		LineMinimizationAlgorithm( score_fxn ), _nonmonotone( nonmonotone ),
		_num_calls( 0 ) {};
	virtual Real operator()( Multivec & curr_pos, Multivec & curr_dir );
	Real Armijo( Real init_step, func_1d & func_eval );

	bool _nonmonotone;
	int _num_calls;
};


} // namespace optimization
} // namespace core


#endif // INCLUDED_core_optimization_LineMinimizer_HH
