// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   core/optimization/AtomTreeMinimizer.hh
/// @brief  High-level atom tree minimizer class for symmetrical minimization
/// @author Ingemar Andre

// Unit headers
#include <core/optimization/symmetry/SymAtomTreeMinimizer.hh>

// Symmetry headers
#include <core/conformation/symmetry/util.hh>

// Package headers
#include <core/optimization/types.hh>
#include <core/optimization/Minimizer.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/optimization/MinimizerMap.hh>
#include <core/optimization/symmetry/SymAtomTreeMultifunc.hh>

#include <core/id/DOF_ID.hh>

// Project headers
#include <core/kinematics/MoveMap.fwd.hh>
#include <core/id/AtomID_Mask.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/LREnergyContainer.hh>

#include <core/util/Tracer.hh>

#include <ObjexxFCL/formatted.o.hh>

using namespace ObjexxFCL::fmt;

namespace core {
namespace optimization {
namespace symmetry {

///////////////////////////////////////////////////////////////////////////////
Real
SymAtomTreeMinimizer::run(
	pose::Pose & pose,
	kinematics::MoveMap const & move_map,
	scoring::ScoreFunction const & scorefxn,
	MinimizerOptions const & options
) const
{
	using namespace core::conformation::symmetry;

	typedef SymmetryInfo::DOF_IDs DOF_IDs;

	bool const use_nblist( options.use_nblist() );

	// it's important that the structure be scored prior to nblist setup
	Real const start_score( scorefxn( pose ) );

	// asymm
	kinematics::MoveMap asym_move_map;
	make_assymetric_movemap( pose, move_map, asym_move_map );

	// semisym -> symm along peptide, asymm in jumps
	kinematics::MoveMap semisym_move_map;
	make_semisymmetric_movemap( pose, move_map, semisym_move_map );

	// setup the map of the degrees of freedom
	MinimizerMap min_map;
	min_map.setup( pose, move_map );

	// setup the map for the assymetric min map
	MinimizerMap asym_min_map;
  asym_min_map.setup( pose, asym_move_map );

	// setup the map for the semisymetric min map
	MinimizerMap semisym_min_map;
  semisym_min_map.setup( pose, semisym_move_map );

	// if we are using the nblist, set it up
	if ( use_nblist ) {
		// setup a mask of the moving dofs
		pose.energies().set_use_nblist( pose, asym_min_map.domain_map(), options.nblist_auto_update() );
	}

	// etable functions must be initialized with asymm domain map
	scorefxn.setup_for_minimizing( pose, asym_min_map );

	// setup the function that we will pass to the low-level minimizer
	SymAtomTreeMultifunc f( pose, min_map, semisym_min_map, asym_min_map, scorefxn,
	                        options.deriv_check(),
	                        options.deriv_check_verbose() );

	// starting position -- "dofs" = Degrees Of Freedom
	Multivec dofs( min_map.nangles() );
	min_map.copy_dofs_from_pose( pose, dofs );

	Real const start_func( f( dofs ) );

	// now do the optimization with the low-level minimizer function
	Minimizer minimizer( f, options );
	minimizer.run( dofs );

	Real const end_func( f( dofs ) );

	//std::cout << "end_func:" << std::endl;
	//pose.energies().show( std::cout );

	// turn off nblist
	if ( use_nblist ) pose.energies().reset_nblist();

	// if we were doing rigid-body minimization, fold the rotation and
	// translation offsets into the jump transforms
	//
	// also sets rb dofs to 0.0, so in principle func value should be the same
	//
	min_map.reset_jump_rb_deltas( pose, dofs );

	// rescore
	Real const end_score( scorefxn( pose ) );

	//std::cout << "end_score:" << std::endl;
	//pose.energies().show( std::cout );

	// we may not really need all these extra function evaluations
	// good for diagnostics though

	static util::Tracer core_optimize( "core.optimize",  util::t_debug);
	core_optimize << "SymAtomTreeMinimizer::run: nangles= " << min_map.nangles() <<
		" start_score: " << F(12,3,start_score) <<
		" start_func: "  << F(12,3,start_func ) <<
		" end_score: "   << F(12,3,end_score  ) <<
		" end_func: "    << F(12,3,end_func   ) << std::endl;


	return end_score;
}

void
SymAtomTreeMinimizer::make_semisymmetric_movemap(
	pose::Pose & pose,
  kinematics::MoveMap const & move_map_sym,
	kinematics::MoveMap & move_map_semisym
) const
{
	typedef SymmetryInfo::DOF_IDs DOF_IDs;

	SymmetricConformation const & SymmConf (
    dynamic_cast<SymmetricConformation const &> ( pose.conformation()) );
	assert( conformation::symmetry::is_symmetric( SymmConf ) );
	SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );

	// copy bb/chi DOFs from symm movemap
	move_map_semisym = move_map_sym;

	for ( Size i=1; i<= pose.conformation().fold_tree().num_jump(); ++i ) {
	 	id::DOF_ID const & null_id
		 	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(i,id::JUMP,1)));
		if ( symm_info.get_dof_derivative_weight( null_id , SymmConf ) > 0 ) {
			// if this is not the master get the master
			if ( symm_info.jump_is_independent( i ) ) continue;

			Size master_i = symm_info.jump_follows( i );

			for ( int j=1; j<= 6; ++j ) {
   		 	id::DOF_ID const & id_master
    		 	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(master_i,id::JUMP,j)));
   		 	id::DOF_ID const & id
    		 	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(i,id::JUMP,j)));

				bool allow ( move_map_sym.get( id_master ) );
				move_map_semisym.set(id, allow );
    	}
    }
  }

}


void
SymAtomTreeMinimizer::make_assymetric_movemap(
	pose::Pose & pose,
  kinematics::MoveMap const & move_map_sym,
	kinematics::MoveMap & move_map_asym
) const
{

	typedef SymmetryInfo::DOF_IDs DOF_IDs;

	SymmetricConformation const & SymmConf (
    dynamic_cast<SymmetricConformation const &> ( pose.conformation()) );
	assert( conformation::symmetry::is_symmetric( SymmConf ) );
	SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );

	for ( Size i=1; i<= pose.conformation().size(); ++i ) {
		if ( symm_info.bb_is_independent( i ) ) {
			bool bb ( move_map_sym.get_bb(i) );
			bool chi ( move_map_sym.get_chi(i) );
			move_map_asym.set_bb ( i, bb );
    	move_map_asym.set_chi( i, chi );
			for ( std::vector< Size>::const_iterator
        clone     = symm_info.bb_clones( i ).begin(),
        clone_end = symm_info.bb_clones( i ).end();
        clone != clone_end; ++clone ){
					move_map_asym.set_bb ( *clone, bb );
    			move_map_asym.set_chi( *clone, chi );
			}
		}
	}
	for ( Size i=1; i<= pose.conformation().fold_tree().num_jump(); ++i ) {
		if ( symm_info.jump_is_independent( i ) ) {
			for ( int j=1; j<= 6; ++j ) {
   		 	id::DOF_ID const & id
    		 	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(i,id::JUMP,j)));
				DOF_IDs const & dofs( symm_info.dependent_dofs( id, SymmConf ) );
				bool allow ( move_map_sym.get( id ) );
				move_map_asym.set(id, allow );
	  		for ( DOF_IDs::const_iterator dof =dofs.begin(), dofe= dofs.end(); dof != dofe; ++dof ) {
					move_map_asym.set( *dof, allow );
    		}
    	}
    }
  }
}

} // symmetry
} // namespace optimization
} // namespace core
