// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   core/optimization/AtomTreeMultifunc.hh
/// @brief  Atom tree multifunction class for symmetrical minimization
/// @author Ingemar Andre

#ifndef INCLUDED_core_optimization_symmetry_SymAtomTreeMultifunc_HH
#define INCLUDED_core_optimization_symmetry_SymAtomTreeMultifunc_HH


// Package headers
#include <core/optimization/types.hh>
#include <core/optimization/Multifunc.hh>
#include <core/optimization/MinimizerMap.hh>
#include <core/optimization/symmetry/sym_atom_tree_minimize.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/util/prof.hh>


namespace core {
namespace optimization {
namespace symmetry {

/// @brief Atom tree multifunction class
class SymAtomTreeMultifunc : public Multifunc
{

public: // Creation

	// c-tor
	SymAtomTreeMultifunc(
		pose::Pose & pose_in,
		MinimizerMap & symm_min_map_in,
		MinimizerMap & semisymm_min_map_in,
		MinimizerMap & asymm_min_map_in,
		scoring::ScoreFunction const & scorefxn_in,
		bool const deriv_check_in = false,
		bool const deriv_check_verbose_in = false
	):
		pose_( pose_in ),
		symm_min_map_( symm_min_map_in ),
		semisymm_min_map_( semisymm_min_map_in ),
		asymm_min_map_( asymm_min_map_in ),
		score_function_( scorefxn_in ),
		deriv_check_( deriv_check_in ),
		deriv_check_verbose_( deriv_check_verbose_in )
	{}



	/// @brief Destructor
	inline
	virtual
	~SymAtomTreeMultifunc()
	{}


public: // Methods


	// func
	virtual
	Real
	operator ()( Multivec const & vars ) const
	{
		PROF_START( util::FUNC );
		symm_min_map_.copy_dofs_to_pose( pose_, vars );
		Real const score( score_function_( pose_ ) );
		PROF_STOP( util::FUNC );
		return score;
	}


	// dfunc
	virtual
	void
	dfunc( Multivec const & vars, Multivec & dE_dvars ) const
	{
		PROF_START( util::DFUNC );
		// in atom_tree_minimize.cc
		atom_tree_dfunc( pose_, symm_min_map_, semisymm_min_map_, asymm_min_map_, score_function_, vars, dE_dvars );
		// optional derivative checking
		if ( deriv_check_ ) {
			numerical_derivative_check( symm_min_map_, *this, vars, dE_dvars, deriv_check_verbose_ );
		}
		PROF_STOP( util::DFUNC );
	}

	/// @brief Error state reached; dump out current pdb.
	virtual
	void
	dump( Multivec const & vars ) const;


private: // data

	/// non-const since pose_ is modified by calls to operator()
	pose::Pose & pose_;

	/// non-const since min_map_ is modified by calls to dfunc()
	MinimizerMap & symm_min_map_;
	MinimizerMap & semisymm_min_map_;
	MinimizerMap & asymm_min_map_;

	scoring::ScoreFunction const & score_function_;

	bool deriv_check_;
	bool deriv_check_verbose_;

}; // AtomTreeMultifunc

} // symmetry
} // namespace optimization
} // namespace core


#endif // INCLUDED_core_optimization_symmetry_SymAtomTreeMultifunc_HH
