// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/interaction_graph/DensePDInteractionGraph.hh
/// @brief  Dense Edge-Matrix, Pairwise Decomposable interaction graph class header
/// This graph does not take advantage of the memory-saving AANeighborSparseMatrix
/// for the storage of RPEs (rotamer pair energies); for this reason it's a little
/// bit faster and weights a little less than the PDInteractionGraph for fixed-sequence
/// repackings -- memory becomes a significant issue for design simulations, and
/// this graph should not be used for that purpose.
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com)


#ifndef INCLUDED_core_pack_interaction_graph_DensePDInteractionGraph_HH
#define INCLUDED_core_pack_interaction_graph_DensePDInteractionGraph_HH

// Unit headers
#include <core/pack/interaction_graph/DensePDInteractionGraph.fwd.hh>

// Package Headers
#include <core/pack/types.hh>
#include <core/pack/interaction_graph/InteractionGraphBase.hh>
#include <core/pack/interaction_graph/PrecomputedInteractionGraph.hh>

//STL Headers
#include <vector>
#include <list>

//ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

namespace core {
namespace pack {
namespace interaction_graph {

// belongs in DensePDInteractionGraph.fwd.hh
class DensePDNode;
class DensePDEdge;
class DensePDInteractionGraph;

class DensePDNode : public PrecomputedPairEnergiesNode
{
public:
	DensePDNode(InteractionGraphBase * owner, int node_id, int num_states);
	virtual ~DensePDNode();
	virtual void print() const;

	virtual void set_amino_acid_types( std::vector< int > const & ) {}

	void update_one_body_energy( int state, PackerEnergy energy );
	virtual void update_one_body_energies( ObjexxFCL::FArray1DB< PackerEnergy > & energies );
	void add_to_one_body_energy( int state, PackerEnergy energy );
	virtual void add_to_one_body_energies( ObjexxFCL::FArray1DB< PackerEnergy > & energies );
	virtual void zero_one_body_energies();
	PackerEnergy get_one_body_energy( int state );

	virtual void prepare_for_simulated_annealing();
	//virtual unsigned int getMemoryUsageInBytes() const;

	void assign_zero_state();
	virtual bool state_unassigned() const { return current_state_ == 0;}
	void assign_state(int new_state);
	int get_current_state() const;
	PackerEnergy get_one_body_energy_current_state();
	inline PackerEnergy project_deltaE_for_substitution
	(
		int alternate_state,
		PackerEnergy & prev_node_energy
	);
	void commit_considered_substitution();

	// <directed_design>
	void project_deltaE_for_substitution
	(
		int alternate_state,
		PackerEnergy & deltaE_unweighted,
		PackerEnergy & prevE_unweighted,
		PackerEnergy & deltaE_weighted,
		PackerEnergy & prevE_weighted,
		ObjexxFCL::FArray2D< PackerEnergy > const& weights
	);

	PackerEnergy get_weighted_energy_with_higher_indexed_nodes(ObjexxFCL::FArray2D< PackerEnergy > const& weights) const;
	// </directed_design>

	inline
	void acknowledge_neighbors_state_substitution(
		int edge_to_altered_neighbor,
		PackerEnergy new_edge_energy,
		int other_node_new_state);

	void print_internal_energies();

	void update_internal_energy_sums();

	/*
	void prepare_to_write_to_file();
	void initialize_aa_for_state_array();
	void clean_up_after_writing_to_file();
	void prepare_to_read_energies_from_file( int num_states_for_node_in_file );
	void clean_up_after_reading_energies_from_file();

	void set_aa_for_file_state(int file_state, int aa );
	void set_instance_state_correspondence( int instance_state, int state_from_file );
	int get_correspondence_for_state( int instance_state );

	int get_num_rots_absent_from_file();
	void get_absent_rots( ObjexxFCL::FArray1DB_int & rots_absent );

	int get_num_states_in_file();
	int & get_aatypes_for_file_states();

	int & get_aatypes_for_states();
	int & get_num_file_states_for_aa();
	int & get_file_states_2_instance_states_array();

	bool get_node_corresponded_to_file_node();
	 */

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

protected:
	void update_internal_vectors();

private:

	inline
	DensePDEdge* get_incident_dpd_edge( int index ) const
	{
		return ( DensePDEdge* ) get_incident_edge( index ); // c-style cast since static_cast won't compile
	}

	inline
	DensePDNode* get_adjacent_dpd_node( int index ) const
	{
		return ( DensePDNode* ) get_adjacent_node( index );
	}

	inline
	DensePDInteractionGraph* get_dpdig_owner() const
	{
		return ( DensePDInteractionGraph* ) get_owner();  // c-style cast since static_cast won't compile
	}

	std::vector< PackerEnergy > one_body_energies_;

	std::vector< int > neighbors_curr_state_;
	std::vector< ObjexxFCL::FArray2Da< PackerEnergy > > edge_matrix_ptrs_;

	int current_state_;
	PackerEnergy curr_state_one_body_energy_;
	PackerEnergy curr_state_total_energy_;
	std::vector< PackerEnergy > curr_state_two_body_energies_;

	int alternate_state_;
	PackerEnergy alternate_state_one_body_energy_;
	PackerEnergy alternate_state_total_energy_;
	std::vector< PackerEnergy > alternate_state_two_body_energies_;

	/*
	//variables for I/O
	int num_states_in_file_;
	ObjexxFCL::FArray1D_int instance_states_2_file_states_;
	ObjexxFCL::FArray1D_int file_states_2_instance_states_;
	ObjexxFCL::FArray1D_int aa_types_for_file_states_;
	ObjexxFCL::FArray1D_int aa_types_for_instance_states_; //useful only for I/O
	ObjexxFCL::FArray1D_int num_file_states_for_aa_;
	*/

	bool alternate_state_is_being_considered_;

	//no default constructor, uncopyable
	DensePDNode();
	DensePDNode( DensePDNode const & );
	DensePDNode & operator = ( DensePDNode const & );
};

class DensePDEdge : public PrecomputedPairEnergiesEdge
{
public:
	DensePDEdge(InteractionGraphBase* owner, int first_node_ind, int second_node_ind);
	virtual ~DensePDEdge();
	virtual void set_sparse_aa_info(ObjexxFCL::FArray2DB_bool const & ) {}
	virtual bool get_sparse_aa_info( int, int) {return true;} //"all amino acids are neighbors"
	virtual void add_to_two_body_energy(int const, int const, PackerEnergy const);
	virtual void
	add_to_two_body_energies( ObjexxFCL::FArray2DB< PackerEnergy > const & res_res_energy_array );
	virtual
	void set_two_body_energy(int const, int const, PackerEnergy const);
	virtual
	void clear_two_body_energy(int const, int const);
	virtual PackerEnergy get_two_body_energy( int const, int const ) const;

	virtual void force_aa_neighbors(int, int) {} //all aa's are already neighbors -- dense representation
	virtual void force_all_aa_neighbors() {} //same thing


	virtual void declare_energies_final();
	virtual void prepare_for_simulated_annealing();
	//virtual unsigned int getMemoryUsageInBytes() const;

	PackerEnergy get_current_two_body_energy();

	void acknowledge_state_change(
			int node_ind,
			int new_state,
			PackerEnergy & new_energy
	);
	void acknowledge_state_zeroed( int node_ind );

	static
	inline
	PackerEnergy get_alternate_state_energy(
		int first_node_state,
		int second_node_state,
		ObjexxFCL::FArray2DB< PackerEnergy > & edge_energy_table
	);

	inline void acknowledge_substitution(
		int substituted_node_index,
		PackerEnergy const curr_state_energy,
		int nodes_new_state
	);

	int get_two_body_table_size() const;
	ObjexxFCL::FArray2Da< PackerEnergy > get_edge_table_ptr();

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

	virtual void set_edge_weight( Real weight );

private:
	inline
	DensePDNode* get_dpd_node( int index ) const
	{
		return (DensePDNode*) get_node( index ); //c-style cast since static_cast won't compile
	}

	inline
	DensePDInteractionGraph* get_dpdig_owner() const
	{
		return (DensePDInteractionGraph*) get_owner(); //c-style cast since static_cast won't compile
	}

	ObjexxFCL::FArray2D< PackerEnergy > two_body_energies_; //Dense matrix
	PackerEnergy curr_state_energy_;
	bool energies_updated_since_last_prep_for_simA_;

	//no default constructor, uncopyable
	DensePDEdge();
	DensePDEdge( DensePDEdge const & );
	DensePDEdge & operator = ( DensePDEdge const & );
};

class DensePDInteractionGraph : public PrecomputedPairEnergiesInteractionGraph
{
public:
	DensePDInteractionGraph(int num_nodes);
	virtual void initialize( rotamer_set::RotamerSetsBase const & rot_sets );

	virtual PackerEnergy get_one_body_energy_for_node_state( int node, int state);

	virtual void set_num_aatypes(int) {}
	virtual int  get_num_aatypes() const {return 1;}

	virtual void blanket_assign_state_0();
	virtual PackerEnergy set_state_for_node(int node_ind, int new_state);
	virtual PackerEnergy set_network_state( ObjexxFCL::FArray1DB_int & node_states);
	virtual void consider_substitution
	(
		int node_ind,
		int new_state,
		PackerEnergy & delta_energy,
		PackerEnergy & prev_energy_for_node
	);

	/// @brief Accepts (commits) the state change previously considered in a call to
	/// consider_substitution and returns the energy of the entire graph
	virtual PackerEnergy commit_considered_substitution();

	/// @brief removes all accumulated numerical drift and returns the
	/// energy for the current state assignment.
	virtual PackerEnergy get_energy_current_state_assignment();
	/// @brief returns the number of floats used in all edge two-body energy tables
	virtual int get_edge_memory_usage() const;

	/// @brief outputs the current state for each node, useful for debugging
	virtual void print_current_state_assignment() const;
	virtual void set_errorfull_deltaE_threshold( PackerEnergy ) {};

	/// @brief a user may define subsets of the vertex set for which they would like to
	/// know the internal energy sum.
	virtual PackerEnergy get_energy_sum_for_vertex_group( int group_id );

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

protected:
	//virtual unsigned int getMemoryUsageInBytes() const;

	virtual NodeBase* create_new_node( int node_index, int num_states);
	virtual EdgeBase* create_new_edge( int index1, int index2);

	/// @brief removes numerical drift that can accumulate over the course of
	/// many state assignment changes within simulated annealing
	void update_internal_energy_totals();

	inline
	DensePDNode* get_dpd_node(int index) const
	{
		return (DensePDNode*) get_node( index );
	}

private:
	int num_commits_since_last_update_;
	PackerEnergy total_energy_current_state_assignment_;
	PackerEnergy total_energy_alternate_state_assignment_;
	int node_considering_alt_state_;

	static const int COMMIT_LIMIT_BETWEEN_UPDATES = 1024; // 2^10

	//no default constructor, uncopyable
	DensePDInteractionGraph();
	DensePDInteractionGraph( DensePDInteractionGraph const & );
	DensePDInteractionGraph & operator = ( DensePDInteractionGraph const & );
};

} // namespace interaction_graph
} // namespace pack
} // namespace core

#endif
