// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/interaction_graph/PrecomputedInteractionGraph.h
/// @brief  Precomputed interaction graph class header
/// @author Andrew Leaver-Fay (leaverfa@email.unc.edu)


#ifndef INCLUDED_core_pack_interaction_graph_PrecomputedInteractionGraph_HH
#define INCLUDED_core_pack_interaction_graph_PrecomputedInteractionGraph_HH

// Unit Headers
#include <core/pack/interaction_graph/PrecomputedInteractionGraph.fwd.hh>

// Package Headers
#include <core/pack/interaction_graph/InteractionGraphBase.hh>

// ObjexxFCL Headers

// Utility Headers
#include <utility/vector1.hh>

namespace core {
namespace pack {
namespace interaction_graph {

class PrecomputedPairEnergiesNode;
class PrecomputedPairEnergiesEdge;
class PrecomputedPairEnergiesInteractionGraph;

class PrecomputedPairEnergiesNode : public NodeBase
{
public:
	virtual ~PrecomputedPairEnergiesNode() {}

	PrecomputedPairEnergiesNode(
		InteractionGraphBase * owner,
		int node_id,
		int num_states)
	:
		NodeBase( owner, node_id, num_states )
	{}

	virtual void set_amino_acid_types( std::vector< int > const & ) = 0;

};

class PrecomputedPairEnergiesEdge : public EdgeBase
{
public:
	virtual ~PrecomputedPairEnergiesEdge() {}
	PrecomputedPairEnergiesEdge(
		InteractionGraphBase* owner,
		int first_node_ind,
		int second_node_ind)
	:
		EdgeBase( owner, first_node_ind, second_node_ind )
	{}

	virtual
	void set_sparse_aa_info(ObjexxFCL::FArray2DB_bool const & sparse_conn_info) = 0;

	virtual
	bool get_sparse_aa_info( int node1aa, int node2aa) = 0;

	virtual
	void force_aa_neighbors(int node1aa, int node2aa) = 0;

	virtual
	void force_all_aa_neighbors() = 0;

	virtual
	void add_to_two_body_energy(int const, int const, PackerEnergy const) = 0;

	virtual
	void add_to_two_body_energies( ObjexxFCL::FArray2DB< PackerEnergy > const & res_res_energy_array ) = 0;

	virtual
	void set_two_body_energy(int const, int const, PackerEnergy const) = 0;

	virtual
	void clear_two_body_energy(int const, int const) = 0;

	virtual PackerEnergy get_two_body_energy( int const, int const ) const = 0;

};

class PrecomputedPairEnergiesInteractionGraph : public InteractionGraphBase
{
public:
	virtual ~PrecomputedPairEnergiesInteractionGraph() {}
	PrecomputedPairEnergiesInteractionGraph( int num_nodes )
	:
		InteractionGraphBase( num_nodes )
	{}

	/// @brief sets the amino acid types for each state of a particular node
	virtual
	void set_aatypes_for_node(int node_ind, std::vector< int > const & aatypes);

	virtual void set_num_aatypes( int ) = 0;
	virtual int get_num_aatypes() const = 0;

	/// @brief interface to PDEdge::set_sparse_aa_info
	void set_sparse_aa_info_for_edge
	(
		int node1,
		int node2,
		ObjexxFCL::FArray2DB_bool const & sparse_conn_info
	);

	/// @brief returns true if node1aa and node2aa are amino acid neighbors
	bool get_sparse_aa_info_for_edge
	(
		int node1,
		int node2,
		int node1aa,
		int node2aa
	);

	/// @brief interface to PrecomputedPairEnergiesEdge::force_aa_neighbors
	void force_aa_neighbors_for_edge
	(
		int node1,
		int node2,
		int node1aa,
		int node2aa
	);

	/// @brief interface to PDEdge::force_aa_neighbors
	void force_all_aa_neighbors_for_edge
	(
		int node1,
		int node2
	);

	/// @brief interface for PrecomputedPairEnergiesEdge::add_to_two_body_energies
	void add_to_two_body_energies_for_edge
	(
		int node1,
		int node2,
		ObjexxFCL::FArray2DB< PackerEnergy > const & res_res_energy_array
	);

	/// @brief interface to PrecomputedPairEnergiesEdge::add_to_two_body_energies
	void add_to_two_body_energies_for_edge
	(
		int node1,
		int node2,
		int state_node1,
		int state_node2,
		PackerEnergy const two_body_energy
	);

	/// @brief interface to PDEdge::set_two_body_energy
	void set_two_body_energy_for_edge
	(
		int node1,
		int node2,
		int state_node1,
		int state_node2,
		PackerEnergy const two_body_energy
	);

	/// @brief interface to PDEdge::clear_two_body_energy
	void clear_two_body_energy_for_edge
	(
		int node1,
		int node2,
		int state_node1,
		int state_node2
	);

	/// @brief interface to PrecomputedPairEnergiesEdge::get_two_body_energy
	///  - returns the state pair energy
	virtual PackerEnergy get_two_body_energy_for_edge
	(
		int node1,
		int node2,
		int state_node1,
		int state_node2
	) const;

	virtual void declare_edge_energies_final(int node1, int node2);

};

} //end namespace interaction_graph
} //end namespace pack
} //end namespace core

#endif
