// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author


#ifndef INCLUDED_core_pack_rotamer_set_WaterPackingInfo_HH
#define INCLUDED_core_pack_rotamer_set_WaterPackingInfo_HH

#include <core/types.hh>

/// MOVE TO USING FWD DECLARATIONS
#include <core/chemical/AA.hh>
#include <core/chemical/ResidueType.hh>
#include <core/util/datacache/CacheableData.hh>

// utility headers
#include <utility/vector1.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/pointer/ReferenceCount.hh>

// C++
#include <string>

namespace core {
namespace pack {
namespace rotamer_set {

class WaterAnchorInfo : public utility::pointer::ReferenceCount {
public:
	typedef chemical::AA AA;
	typedef chemical::ResidueType ResidueType;

public:

	Size
	anchor_residue() const
	{
		return anchor_residue_;
	}

	void
	anchor_residue( Size const rsd )
	{
		anchor_residue_ = rsd;
	}

	bool
	attaches_to_residue_type( ResidueType const & type ) const
	{
		return type.aa() == aa_ && type.has( anchor_atom_name_ );
	}

	Size
	anchor_atom( ResidueType const & type ) const
	{
		assert( attaches_to_residue_type( type ) );
		return type.atom_index( anchor_atom_name_ );
	}

	void
	anchor_atom( std::string const & name )
	{
		anchor_atom_name_ = name;
	}

	void
	aa( AA const & aa_in )
	{
		aa_ = aa_in;
	}

	Size
	nstep() const
	{
		return nstep_;
	}

	void
	nstep( Size const nstep_in )
	{
		nstep_ = nstep_in;
	}

private:

	Size anchor_residue_;
	std::string anchor_atom_name_;
	AA aa_;
	Size nstep_;
};

typedef utility::pointer::owning_ptr< WaterAnchorInfo > WaterAnchorInfoOP;




class WaterPackingInfo : public util::datacache::CacheableData {

public:

	WaterPackingInfo(){}

	WaterPackingInfo( WaterPackingInfo const & src ):
		CacheableData(),
		data_( src.data_ )
	{
		for ( Size i=1; i<= data_.size(); ++i ) {
			if ( data_[ i ] ) data_[i] = new WaterAnchorInfo( *data_[i] );
		}
	}

	util::datacache::CacheableDataOP
	clone() const
	{
		return new WaterPackingInfo( *this );
	}

	WaterAnchorInfo const &
	operator[] ( Size const seqpos ) const
	{
		assert( seqpos <= data_.size() && data_[ seqpos ] );
		return *( data_[ seqpos ] );
	}

	WaterAnchorInfo &
	operator[] ( Size const seqpos )
	{
		if ( seqpos > data_.size() ) data_.resize( seqpos, 0 );
		if ( data_[seqpos] == 0 ) {
			data_[seqpos] = new WaterAnchorInfo();
		}
		return *( data_[ seqpos ] );
	}


	void
	clear()
	{
		data_.clear();
	}


private:

	utility::vector1< WaterAnchorInfoOP > data_;



};


typedef utility::pointer::owning_ptr< WaterPackingInfo > WaterPackingInfoOP;

} // namespace rotamer_set
} // namespace pack
} // namespace core


#endif // INCLUDED_core_pack_RotamerSet_RotamerSet_fwd_HH
