// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/rtmin.cc
/// @brief  rotamer trials with minimization module header
/// @author Ian W. Davis (ian.w.davis@gmail.com)
/// @author Ian W. Davis (ian.w.davis@gmail.com)

// Unit headers
#include <core/pack/rtmin.hh>

// Package headers
#include <core/pack/types.hh>
#include <core/pack/packer_neighbors.hh>
#include <core/pack/rotamer_set/RotamerSet.hh>
#include <core/pack/rotamer_set/RotamerSetFactory.hh>
#include <core/pack/task/PackerTask.hh>

// Project headers
#include <core/types.hh>

#include <core/conformation/Residue.hh>
#include <core/id/AtomID.hh>
#include <core/id/AtomID_Mask.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/optimization/types.hh>
#include <core/optimization/Minimizer.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/optimization/MinimizerMap.hh>
#include <core/optimization/AtomTreeMultifunc.hh>
#include <core/optimization/SingleResidueMultifunc.hh>

#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionInfo.hh>
#include <core/scoring/hbonds/hbonds.hh>
#include <core/scoring/hbonds/HBondSet.hh>
#include <core/scoring/methods/EnergyMethod.hh>

#include <core/util/prof.hh>
#include <core/util/Tracer.hh>

#include <utility/vector1.hh>
#include <utility/vector1.functions.hh>

#include <numeric/random/random.hh>
#include <numeric/random/random_permutation.hh>

#include <ObjexxFCL/formatted.o.hh>

// STL headers
#include <ctime>

namespace core {
namespace pack {

static numeric::random::RandomGenerator rottrials_RG(206025); // <- Magic number, do not change it!!!

static util::Tracer TR( "core.pack.rtmin" );

//forward dec
utility::vector1< uint >
repackable_residues_dup( task::PackerTask const & the_task );

void
rtmin(
	pose::Pose & pose,
	scoring::ScoreFunction const & scfxn,
	task::PackerTaskCOP input_task
)
{
	using namespace numeric::random;
	using namespace core::optimization;

	PROF_START( util::ROTAMER_TRIALS );
	pack_scorefxn_pose_handshake( pose, scfxn);
	pose.update_residue_neighbors();

	utility::vector1< uint > residues_for_trials( repackable_residues_dup( *input_task ));
	random_permutation( residues_for_trials, rottrials_RG );

	task::PackerTaskOP rottrial_task( input_task->clone() );
	rottrial_task->set_bump_check( false );
	rottrial_task->or_include_current( true );
	rottrial_task->temporarily_fix_everything();

	// this will call setup fxns for each scoring method, eg HBondEnergy will
	// compute backbone hbonds to prepare for hbchecking,
	// PairEnergy will update actcoords...
	scfxn.setup_for_packing( pose, *rottrial_task );

	rotamer_set::RotamerSetFactory rsf;
	graph::GraphOP packer_neighbor_graph = create_packer_graph( pose, scfxn, input_task );

	Size const num_in_trials = residues_for_trials.size();
	for (Size ii = 1; ii <= num_in_trials; ++ii)
	{
		pose.update_residue_neighbors(); // will return if uptodate

		int const resid = residues_for_trials[ ii ];
		conformation::Residue const & trial_res = pose.residue( resid );

		//pretend this is a repacking and only this residue is being repacked
		//while all other residues are being held fixed.
		rottrial_task->temporarily_set_pack_residue( resid, true );

		rotamer_set::RotamerSetOP rotset = rsf.create_rotamer_set( trial_res );
		rotset->set_resid( resid );
		rotset->build_rotamers( pose, scfxn, *rottrial_task, packer_neighbor_graph );
		scfxn.prepare_rotamers_for_packing( pose, *rotset );
		TR.Debug << "working on " << resid << " with " << rotset->num_rotamers() << " rotamers" << std::endl;

		// All DOF start false (frozen)
		kinematics::MoveMap movemap;
		movemap.set_chi(resid, true);
		optimization::MinimizerOptions min_options(
			//"dfpmin_armijo_nonmonotone_atol", 0.1, true /*nblist*/, false /*deriv_check*/, false /*deriv_verbose*/ );
			"dfpmin", 0.1, true /*nblist*/, false /*deriv_check*/, false /*deriv_verbose*/ );

		Size best_jj = 0;
		Real best_score = 1e99;
		conformation::ResidueOP best_rsd;
		for ( Size jj = 1; jj <= rotset->num_rotamers(); ++jj ) {
			conformation::ResidueOP newresidue( rotset->rotamer( jj )->clone() );

			// Assume that protein residues' conformation is fully specified by chi angles.
			// This is NOT true for e.g. ligands, which may have e.g. various ring puckers.
			if( newresidue->is_protein() && newresidue->type().name() == pose.residue_type(resid).name() ) {
				//TR << "Setting chi angles..." << std::endl;
				for( Size kk = 1; kk <= newresidue->nchi(); ++kk ) {
					pose.set_chi(kk, resid, newresidue->chi(kk));
				}
			} else {
				pose.replace_residue( resid, *newresidue, false );
				scfxn.update_residue_for_packing( pose, resid );
			}

			// Code copied from AtomTreeMinimizer::run()
			// This has to be repeated for each residue because the ResidueType may change if we're doing design.
			// Even if not, we get a fatal error if we try to do it outside the loop,
			// which I think is related to replace_residue() modifying the structure of the atom tree.
			// It's important that the structure be scored prior to nblist setup -- why?
			// A:  required for graph state == GOOD;  triggers assert in debug mode.
			//Real const start_score = scfxn( pose );
			// Actually, this appears to be sufficient, and is much cheaper (no twobody energy calc)
			pose.scoring_begin( scfxn );
			pose.scoring_end( scfxn );
			// setup the map of the degrees of freedom
			MinimizerMap min_map;
			min_map.setup( pose, movemap );
			// if we are using the nblist, set it up
			if ( min_options.use_nblist() ) {
				// setup a mask of the moving dofs
				pose.energies().set_use_nblist( pose, min_map.domain_map(), min_options.nblist_auto_update() );
			}
			scfxn.setup_for_minimizing( pose, min_map );
			// setup the function that we will pass to the low-level minimizer
			//AtomTreeMultifunc f( pose, min_map, scfxn, min_options.deriv_check(), min_options.deriv_check_verbose() );
			SingleResidueMultifunc f( pose, resid, min_map, scfxn, packer_neighbor_graph, min_options.deriv_check(), min_options.deriv_check_verbose() );
			// starting position -- "dofs" = Degrees Of Freedom
			Multivec dofs( min_map.nangles() );

			// Code copied from AtomTreeMinimizer::run()
			min_map.copy_dofs_from_pose( pose, dofs );
			//Real const start_func = f( dofs );

			// This actually caches the hbonds, etc.
			for ( scoring::ScoreFunction::AllMethodsIterator it=scfxn.all_energies_begin(),
					it_end = scfxn.all_energies_end(); it != it_end; ++it ) {
				(*it)->setup_for_scoring( pose, scfxn );
			}

			// now do the optimization with the low-level minimizer function
			Minimizer minimizer( f, min_options );
			Real const score = minimizer.run( dofs );
			//Real const end_func = f( dofs );
			TR.Trace << "Rotamer " << jj << " " << newresidue->name3() <<
			" nangles= " << min_map.nangles() <<
			//" start_score: " << F(12,3,start_score) <<
			//" start_func: " << F(12,3,start_func) <<
			" score: "      << F(12,3,score     ) <<
			//" end_func: "   << F(12,3,end_func  ) <<
			std::endl;

			if ( min_options.use_nblist() ) pose.energies().reset_nblist();

			if(score < best_score) {
				best_jj = jj;
				best_score = score;
				best_rsd = pose.residue(resid).clone();
			}
		}

		if ( best_jj > 0 ) {
			pose.replace_residue ( resid, *best_rsd, false );
			scfxn.update_residue_for_packing( pose, resid );
		}

		rottrial_task->temporarily_set_pack_residue( resid, false );
	}
	PROF_STOP ( util::ROTAMER_TRIALS );
}

utility::vector1< uint >
repackable_residues_dup( task::PackerTask const & the_task )
{
	utility::vector1< int > to_be_packed( the_task.num_to_be_packed() );
	uint count = 0;
	for (uint ii = 1; ii <= the_task.total_residue(); ++ii )
	{
		if ( the_task.pack_residue( ii ) )
		{
			++count;
			to_be_packed[ count ] = ii;
		}
	}
	return to_be_packed;
}


} //end namespace core
} //end namespace pack
