// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/operation/ResFilters.hh
/// @brief  core-level (very general) classes that take a pose and a residue index, and return true or false
/// @author ashworth

// Unit Headers
#include <core/pack/task/operation/ResFilters.hh>

// Project Headers
#include <core/chemical/ResidueType.hh>
#include <core/pose/Pose.hh>

#include <utility/Tag/Tag.hh>

namespace core {
namespace pack {
namespace task {
namespace operation {

ResidueHasProperty::ResidueHasProperty()
	: parent()
{}

ResidueHasProperty::ResidueHasProperty( std::string const & str )
	: parent(),
		property_( str )
{}

bool ResidueHasProperty::operator() ( Pose const & pose, Size index ) const
{
	runtime_assert( index > 0 && index <= pose.total_residue() );
	return pose.residue_type(index).has_property( property_ );
}

ResFilterOP ResidueHasProperty::clone() const { return new ResidueHasProperty( *this ); }

void ResidueHasProperty::parse_tag( TagPtr tag )
{
	if ( tag->hasOption("property") ) property_ = tag->getOption<std::string>("property");
}

ResidueLacksProperty::ResidueLacksProperty()
	: parent()
{}

ResidueLacksProperty::ResidueLacksProperty( std::string const & str )
	: parent(),
		property_( str )
{}

bool ResidueLacksProperty::operator() ( Pose const & pose, Size index ) const
{
	runtime_assert( index > 0 && index <= pose.total_residue() );
	return ! pose.residue_type(index).has_property( property_ );
}

ResFilterOP ResidueLacksProperty::clone() const { return new ResidueLacksProperty( *this ); }

void ResidueLacksProperty::parse_tag( TagPtr tag )
{
	if ( tag->hasOption("property") ) property_ = tag->getOption<std::string>("property");
}

ResidueName3Is::ResidueName3Is()
	: parent()
{}

ResidueName3Is::ResidueName3Is( std::string const & str )
	: parent(),
		name3_( str )
{}

bool ResidueName3Is::operator() ( Pose const & pose, Size index ) const
{
	runtime_assert( index > 0 && index <= pose.total_residue() );
	return pose.residue_type(index).name3() == name3_;
}

ResFilterOP ResidueName3Is::clone() const { return new ResidueName3Is( *this ); }

void ResidueName3Is::parse_tag( TagPtr tag )
{
	if ( tag->hasOption("name3") ) name3_ = tag->getOption<std::string>("name3");
}

} //namespace operation
} //namespace task
} //namespace pack
} //namespace core
