// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/operation/ResLvlTaskOperations.cc
/// @brief
/// @author ashworth

// Unit Headers
#include <core/pack/task/operation/ResLvlTaskOperations.hh>

#include <core/pack/task/PackerTask.hh>

#include <core/chemical/AA.hh>

#include <utility/Tag/Tag.hh>
#include <utility/exit.hh>

#include <sstream>

namespace core {
namespace pack {
namespace task {
namespace operation {

RestrictToRepackingRLT::~RestrictToRepackingRLT() {}

ResLvlTaskOperationOP
RestrictToRepackingRLT::clone() const { return new RestrictToRepackingRLT( *this ); }

void RestrictToRepackingRLT::apply( ResidueLevelTask & rlt ) const
{
	rlt.restrict_to_repacking();
}

RestrictAbsentCanonicalAASRLT::RestrictAbsentCanonicalAASRLT()
	: canonical_aas_to_keep_( chemical::num_canonical_aas, false )
{}

RestrictAbsentCanonicalAASRLT::~RestrictAbsentCanonicalAASRLT() {}

ResLvlTaskOperationOP
RestrictAbsentCanonicalAASRLT::clone() const { return new RestrictAbsentCanonicalAASRLT( *this ); }

void RestrictAbsentCanonicalAASRLT::apply( ResidueLevelTask & rlt ) const
{
	rlt.restrict_absent_canonical_aas( canonical_aas_to_keep_ );
}

void RestrictAbsentCanonicalAASRLT::aas_to_keep( utility::vector1< bool > const & aa_flags )
{
	runtime_assert( aa_flags.size() == chemical::num_canonical_aas );
	canonical_aas_to_keep_ = aa_flags;
}

void RestrictAbsentCanonicalAASRLT::aas_to_keep( std::string const & aastring )
{
	using namespace chemical;
	runtime_assert( canonical_aas_to_keep_.size() == num_canonical_aas );
	for ( std::string::const_iterator it( aastring.begin() ), end( aastring.end() );
		it != end; ++it ) {
		if ( oneletter_code_specifies_aa( *it ) ) {
			canonical_aas_to_keep_[ aa_from_oneletter_code( *it ) ] = true;
		} else {
			std::ostringstream os;
			os << "aa letter " << *it << " does not not correspond to a canonical AA";
			utility_exit_with_message( os.str() );
		}
	}
}

void RestrictAbsentCanonicalAASRLT::parse_tag( TagPtr tag )
{
	runtime_assert( tag );
	if ( tag->hasOption("aas") ) aas_to_keep( tag->getOption<std::string>("aas") );
	else utility_exit_with_message("no aas tag option by which restrict absent canonical aas.");
}

PreventRepackingRLT::~PreventRepackingRLT() {}

ResLvlTaskOperationOP
PreventRepackingRLT::clone() const { return new PreventRepackingRLT( *this ); }

void PreventRepackingRLT::apply( ResidueLevelTask & rlt ) const
{
	rlt.prevent_repacking();
}

AddBehaviorRLT::AddBehaviorRLT() : parent() {}

AddBehaviorRLT::AddBehaviorRLT( std::string const & behavior )
	: parent(),
		behavior_( behavior )
{}

AddBehaviorRLT::~AddBehaviorRLT() {}

ResLvlTaskOperationOP
AddBehaviorRLT::clone() const { return new AddBehaviorRLT( *this ); }

void AddBehaviorRLT::apply( ResidueLevelTask & rlt ) const
{
	runtime_assert( ! behavior_.empty() );
	rlt.add_behavior( behavior_ );
}

void AddBehaviorRLT::parse_tag( TagPtr tag )
{
	runtime_assert( tag );
	if ( tag->hasOption("behavior") ) behavior_ = tag->getOption<std::string>("behavior");
	else utility_exit_with_message("AddBehaviorRLT tag needs to define option \"behavior\".");
}

} //namespace operation
} //namespace task
} //namespace pack
} //namespace core
