// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/operation/TaskOperationFactory.cc
/// @brief
/// @author ashworth

#include <core/pack/task/operation/TaskOperationFactory.hh>

#include <core/pack/task/operation/TaskOperation.hh>
#include <core/pack/task/operation/ResLvlTaskOperation.hh>
#include <core/pack/task/operation/ResLvlTaskOperationFactory.hh>
#include <core/pack/task/operation/ResFilter.hh>
#include <core/pack/task/operation/ResFilterFactory.hh>

// very general operations that are defined in the local namespace
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/operation/OperateOnCertainResidues.hh>
#include <core/pack/task/operation/ResLvlTaskOperations.hh>
#include <core/pack/task/operation/ResFilters.hh>

#include <utility/exit.hh> // runtime_assert, utility_exit_with_message
#include <utility/Tag/Tag.hh>

namespace core {
namespace pack {
namespace task {
namespace operation {

TaskOperationFactory::TaskOperationFactory()
	: parent(),
		rlto_factory_(0),
		res_filter_factory_(0)
{
	// initialize all TaskOperations that are present in the core::pack::task namespace.
	// for additional TaskOperations (such as in protocols), use add_type where needed (but not here)
	add_type( new RestrictToRepacking );
	add_type( new RestrictResidueToRepacking );
	add_type( new RestrictAbsentCanonicalAAS );
	add_type( new InitializeFromCommandline );
	add_type( new InitializeExtraRotsFromCommandline );
	add_type( new IncludeCurrent );
	add_type( new ReadResfile );
	add_type( new SetRotamerCouplings );
	add_type( new AppendRotamer );
	add_type( new AppendRotamerSet );
	add_type( new PreserveCBeta );
	add_type( new PreventRepacking );
	add_type( new RestrictYSDesign );
	add_type( new OperateOnCertainResidues );
	// ResLvlTaskOperations
	add_type( new RestrictToRepackingRLT );
	add_type( new RestrictAbsentCanonicalAASRLT );
	add_type( new PreventRepackingRLT );
	add_type( new AddBehaviorRLT );
	// ResFilters
	add_type( new ResidueHasProperty );
	add_type( new ResidueLacksProperty );
	add_type( new ResidueName3Is );
}

TaskOperationFactory::~TaskOperationFactory(){}

///@brief add a TaskOperation prototype, using its default type name as the map key
void
TaskOperationFactory::add_type(
	TaskOperationOP task_operation,
	std::string type /* = std::string() */
)
{
	runtime_assert( task_operation );
	if ( type.empty() ) type = task_operation->type();
	if ( type == "UNIMPLEMENTED TYPE STRING" ) {
		utility_exit_with_message("Can't map derived TaskOperation with unimplemented type() method.");
	}
	task_operation_map_[ type ] = task_operation;
}

bool TaskOperationFactory::has_type( std::string const & type ) const
{
	return ( task_operation_map_.find( type ) != task_operation_map_.end() );
}

///@brief adds a ResLvlTaskOperation prototype to the child ResLvlTaskOperationFactory
void
TaskOperationFactory::add_type(
	ResLvlTaskOperationOP rlt_operation,
	std::string type /* = std::string() */
)
{
	if ( ! rlto_factory_ ) rlto_factory_ = new ResLvlTaskOperationFactory;
	rlto_factory_->add_type( rlt_operation, type );
}

///@brief adds a ResFilter prototype to the child ResFilterFactory
void
TaskOperationFactory::add_type(
	ResFilterOP res_filter,
	std::string type /* = std::string() */
)
{
	if ( ! res_filter_factory_ ) res_filter_factory_ = new ResFilterFactory;
	res_filter_factory_->add_type( res_filter, type );
}

///@brief return new TaskOperation by key lookup in task_operation_map_ (new TaskOperation parses Tag if provided)
/*!
Example Tag syntax for parser as of Summer 2009

<ReadResfile name=rrf filename=myresfile/>

or

<OperateOnCertainResidues name=PROTEINnopack>
	<PreventRepackingRLT/>
	<ResidueHasProperty property=PROTEIN/>
</OperateOnCertainResidues>

/*/
TaskOperationOP
TaskOperationFactory::newTaskOperation(
	std::string const & type,
	TagPtr tag /* = boost::shared_ptr< Tag >() */
) const
{
	TaskOperationMap::const_iterator iter( task_operation_map_.find( type ) );
	if ( iter != task_operation_map_.end() ) {
		TaskOperationOP task_operation( iter->second->clone() );
		// parse tag if tag pointer is pointing to one
		if ( tag.get() != NULL ) task_operation->parse_tag( tag, rlto_factory_, res_filter_factory_ );
		return task_operation;
	} else {
		utility_exit_with_message( type + " is not known to the TaskOperationFactory. TaskOperations outside of core::pack::task must be registered via the add_type method." );
		return NULL;
	}
}

} //namespace operation
} //namespace task
} //namespace pack
} //namespace core
