// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/operation/TaskOperations.cc
/// @brief
/// @author Andrew Leaver-Fay (leaverfa@email.unc.edu)

// Unit Headers
#include <core/pack/task/operation/TaskOperations.hh>

#include <core/pack/task/PackerTask.hh>

// Project Headers
#include <core/pack/rotamer_set/RotamerCouplings.hh>
#include <core/pack/rotamer_set/RotamerSetOperation.hh>
#include <core/pack/task/RotamerSampleOptions.hh>

#include <core/options/option.hh>
#include <core/pose/Pose.hh>
#include <core/util/Tracer.hh>

// Utility Headers
#include <utility/exit.hh>
#include <utility/Tag/Tag.hh>

// option key includes
#include <core/options/keys/packing.OptionKeys.gen.hh>

namespace core {
namespace pack {
namespace task {
namespace operation {

using util::t_warning;
using util::t_info;
using util::t_debug;
static util::Tracer TR("core.pack.task.operation.TaskOperations",t_info);
using namespace utility::Tag;

/// BEGIN RestrictToRepacking

RestrictToRepacking::~RestrictToRepacking() {}

TaskOperationOP RestrictToRepacking::clone() const
{
	return new RestrictToRepacking( *this );
}

void
RestrictToRepacking::apply( pose::Pose const &, PackerTask & task ) const
{
	task.restrict_to_repacking();
}

/// BEGIN RestrictResidueToRepacking
RestrictResidueToRepacking::~RestrictResidueToRepacking() {}

TaskOperationOP RestrictResidueToRepacking::clone() const
{
	return new RestrictResidueToRepacking( *this );
}

void
RestrictResidueToRepacking::apply( pose::Pose const &, PackerTask & task ) const
{
	for(utility::vector1< core::Size >::const_iterator it(residues_to_restrict_to_repacking_.begin()), end(residues_to_restrict_to_repacking_.end());
			it != end; ++it)
		{task.nonconst_residue_task(*it).restrict_to_repacking();}
	return;
}

void
RestrictResidueToRepacking::include_residue( core::Size resid ) { residues_to_restrict_to_repacking_.push_back(resid); }

void
RestrictResidueToRepacking::clear() { residues_to_restrict_to_repacking_.clear(); }


/// BEGIN RestrictAbsentCanonicalAAS
RestrictAbsentCanonicalAAS::RestrictAbsentCanonicalAAS()
	:	parent(),
		resid_(1)
{}

RestrictAbsentCanonicalAAS::RestrictAbsentCanonicalAAS( core::Size resid, utility::vector1< bool > keep_aas  )
	:
	parent(),
	resid_(resid),
	keep_aas_( keep_aas )
{}

RestrictAbsentCanonicalAAS::~RestrictAbsentCanonicalAAS(){}

TaskOperationOP RestrictAbsentCanonicalAAS::clone() const
{
	return new RestrictAbsentCanonicalAAS( *this );
}

void
RestrictAbsentCanonicalAAS::apply( pose::Pose const &, PackerTask & task ) const
{
	task.nonconst_residue_task( resid_ ).restrict_absent_canonical_aas( keep_aas_ );
}


RotamerExplosion::RotamerExplosion(){}

RotamerExplosion::RotamerExplosion( core::Size const resid, ExtraRotSample const sample_level, core::Size const chi ) :
	resid_( resid ),
	chi_( chi ),
	sample_level_( sample_level )
{}

RotamerExplosion::~RotamerExplosion() {}

TaskOperationOP
RotamerExplosion::clone() const
{
	return( new RotamerExplosion( *this ) );
}

void
RotamerExplosion::apply( core::pose::Pose const &, PackerTask & task ) const
{
	ResidueLevelTask & restask( task.nonconst_residue_task( resid_ ) );
  if( chi_ > 0 ) restask.or_ex1_sample_level( sample_level_ );
	if( chi_ > 1 ) restask.or_ex2_sample_level( sample_level_ );
	if( chi_ > 2 ) restask.or_ex3_sample_level( sample_level_ );
	if( chi_ > 3 ) restask.or_ex4_sample_level( sample_level_ );
	restask.or_include_current( false );
}

/// BEGIN InitializeFromCommandline

InitializeFromCommandline::~InitializeFromCommandline() {}

TaskOperationOP InitializeFromCommandline::clone() const
{
	return new InitializeFromCommandline( *this );
}

void
InitializeFromCommandline::apply( pose::Pose const &, PackerTask & task ) const
{
	task.initialize_from_command_line();
}

/// BEGIN InitializeFromCommandline

InitializeExtraRotsFromCommandline::~InitializeExtraRotsFromCommandline() {}

TaskOperationOP InitializeExtraRotsFromCommandline::clone() const
{
	return new InitializeExtraRotsFromCommandline( *this );
}

void
InitializeExtraRotsFromCommandline::apply( pose::Pose const &, PackerTask & task ) const
{
	task.initialize_extra_rotamer_flags_from_command_line();
}


/// BEGIN IncludeCurrent

IncludeCurrent::~IncludeCurrent() {}

TaskOperationOP IncludeCurrent::clone() const
{
	return new IncludeCurrent( *this );
}

void
IncludeCurrent::apply( pose::Pose const &, PackerTask & task ) const
{
	task.or_include_current(true);
}

/// BEGIN ReadResfile

ReadResfile::ReadResfile() : parent()
{
	default_filename();
}

ReadResfile::ReadResfile( std::string const & filename )
	: parent(),
		resfile_filename_( filename )
{}

ReadResfile::~ReadResfile() {}

TaskOperationOP ReadResfile::clone() const
{
	return new ReadResfile( *this );
}

void
ReadResfile::apply( pose::Pose const &, PackerTask & task ) const
{
	task.read_resfile( resfile_filename_ );
}

void
ReadResfile::default_filename()
{
	///.value().at(1) gets the value of the first resfile in the FileVector (like [1])
	resfile_filename_ = core::options::option[ core::options::OptionKeys::packing::resfile].value().at(1);
}

void
ReadResfile::filename( std::string const & filename )
{
	resfile_filename_ = filename;
}

std::string const & ReadResfile::filename() const
{
	return resfile_filename_;
}

void
ReadResfile::parse_tag(
	TagPtr tag,
	ResLvlTaskOperationFactoryCOP,
	ResFilterFactoryCOP
)
{
	if ( tag->hasOption("filename") ) resfile_filename_ = tag->getOption<std::string>("filename");
	// special case: if "COMMANDLINE" string specified, use commandline option setting
	if ( resfile_filename_ == "COMMANDLINE" ) default_filename();
}


/// BEGIN SetRotamerCouplings

SetRotamerCouplings::SetRotamerCouplings()
{}

SetRotamerCouplings::~SetRotamerCouplings()
{}

SetRotamerCouplings::SetRotamerCouplings( SetRotamerCouplings const & src )
:
	parent(),
	rotamer_couplings_( src.rotamer_couplings_ )
{}

SetRotamerCouplings const &
SetRotamerCouplings::operator = ( SetRotamerCouplings const & rhs )
{
	rotamer_couplings_ = rhs.rotamer_couplings_;
	return *this;
}

TaskOperationOP SetRotamerCouplings::clone() const
{
	return new SetRotamerCouplings( *this );
}

void
SetRotamerCouplings::apply( pose::Pose const &, PackerTask & task ) const
{
	task.rotamer_couplings( rotamer_couplings_ );
}

void
SetRotamerCouplings::set_couplings( rotamer_set::RotamerCouplingsOP couplings )
{
	rotamer_couplings_ = couplings;
}

/// BEGIN AppendRotamer

AppendRotamer::AppendRotamer()
	: rotamer_operation_(0)
{}

AppendRotamer::~AppendRotamer()
{}

AppendRotamer::AppendRotamer( rotamer_set::RotamerOperationOP rotamer_operation )
 : rotamer_operation_( rotamer_operation )
{}

AppendRotamer::AppendRotamer( AppendRotamer const & src )
: parent(), rotamer_operation_( src.rotamer_operation_ )
{}

TaskOperationOP AppendRotamer::clone() const
{
	return new AppendRotamer( *this );
}

void
AppendRotamer::apply( pose::Pose const &, PackerTask & task ) const
{
	task.append_rotamer_operation( rotamer_operation_ );
}

void
AppendRotamer::set_rotamer_operation(
	rotamer_set::RotamerOperationOP rotamer_operation
)
{
	rotamer_operation_ = rotamer_operation;
}


/// BEGIN AppendRotamerSet

AppendRotamerSet::AppendRotamerSet()
	: rotamer_set_operation_(0)
{}

AppendRotamerSet::~AppendRotamerSet()
{}

AppendRotamerSet::AppendRotamerSet( rotamer_set::RotamerSetOperationOP rotamer_set_operation )
 : rotamer_set_operation_( rotamer_set_operation )
{}

AppendRotamerSet::AppendRotamerSet( AppendRotamerSet const & src )
: parent(), rotamer_set_operation_( src.rotamer_set_operation_ )
{}

TaskOperationOP AppendRotamerSet::clone() const
{
	return new AppendRotamerSet( *this );
}

void
AppendRotamerSet::apply( pose::Pose const &, PackerTask & task ) const
{
	task.append_rotamerset_operation( rotamer_set_operation_ );
}

void
AppendRotamerSet::set_rotamer_set_operation(
	rotamer_set::RotamerSetOperationOP rotamer_set_operation
)
{
	rotamer_set_operation_ = rotamer_set_operation;
}



/// BEGIN PreserveCBeta

PreserveCBeta::~PreserveCBeta() {}

TaskOperationOP PreserveCBeta::clone() const
{
	return new PreserveCBeta( *this );
}

void
PreserveCBeta::apply( pose::Pose const &, PackerTask & task ) const
{
	task.or_preserve_c_beta( true );
}

/// BEGIN PreventRepacking
PreventRepacking::~PreventRepacking() {}

TaskOperationOP PreventRepacking::clone() const
{
	return new PreventRepacking( *this );
}

void
PreventRepacking::apply( pose::Pose const &, PackerTask & task ) const
{
	for(utility::vector1< core::Size >::const_iterator it(residues_to_prevent_.begin()), end(residues_to_prevent_.end());
			it != end; ++it)
		{task.nonconst_residue_task(*it).prevent_repacking();}
	return;
}

void
PreventRepacking::include_residue( core::Size resid ) { residues_to_prevent_.push_back(resid); }

void
PreventRepacking::clear() { residues_to_prevent_.clear(); }


// BEGIN RestrictYSDesign
RestrictYSDesign::~RestrictYSDesign() {}
RestrictYSDesign::RestrictYSDesign() : gly_switch_( false )  {}
RestrictYSDesign::RestrictYSDesign( RestrictYSDesign const & src ) :
	parent(), YSresids_( src.YSresids_ ), gly_switch_( src.gly_switch_ )
{}
RestrictYSDesign::RestrictYSDesign( utility::vector1< core::Size > const & resids ) {
	gly_switch_ = false;
	YSresids_ = resids;
}

void
RestrictYSDesign::apply( pose::Pose const &, PackerTask & task ) const {
	utility::vector1<bool> restrict_to_aa( 20, false );
	for( utility::vector1<core::Size>::const_iterator res_it=YSresids_.begin(); res_it!=YSresids_.end(); ++res_it ) {
		if( gly_switch_ ) restrict_to_aa[chemical::aa_from_name( "GLY" )] = true;;
		restrict_to_aa[chemical::aa_from_name( "TYR" )] = true;
		restrict_to_aa[chemical::aa_from_name( "SER" )] = true;
		task.nonconst_residue_task(*res_it).restrict_absent_canonical_aas( restrict_to_aa );
	}
}

TaskOperationOP
RestrictYSDesign::clone() const {
	return new RestrictYSDesign( *this );
}

void
RestrictYSDesign::include_resid( core::Size const resid ) { YSresids_.push_back(resid); }

void
RestrictYSDesign::include_gly( bool const gly ) { gly_switch_ = gly; }

} //namespace operation
} //namespace task
} //namespace pack
} //namespace core
