// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/operation/TaskOperations.hh
/// @brief  An operation to perform on a packer task --
///         usually, by a PackerTaskFactory right after the task's construction
///         This is an incomplete list.  Freely define your own TaskOperation and hand it
///         to a PackerTaskFactory.
/// @author Andrew Leaver-Fay (leaverfa@email.unc.edu)

#ifndef INCLUDED_core_pack_task_operation_TaskOperations_HH
#define INCLUDED_core_pack_task_operation_TaskOperations_HH

// Unit Headers
#include <core/pack/task/operation/TaskOperation.hh> // abstract base classes
#include <core/pack/task/operation/TaskOperations.fwd.hh>
#include <core/pack/task/operation/ResLvlTaskOperationFactory.hh>
#include <core/pack/task/operation/ResFilterFactory.hh>

// Project Headers
#include <core/pose/Pose.fwd.hh>
#include <core/pack/task/PackerTask.fwd.hh>
#include <core/pack/rotamer_set/RotamerCouplings.fwd.hh>
#include <core/pack/rotamer_set/RotamerSetOperation.fwd.hh>
#include <core/pack/task/RotamerSampleOptions.hh>
#include <core/types.hh>

// Utility Headers
#include <utility/Tag/Tag.fwd.hh>
#include <utility/vector1.hh>
#include <utility/pointer/ReferenceCount.hh>

// C++ Headers
#include <string>

namespace core {
namespace pack {
namespace task {
namespace operation {

class RestrictToRepacking : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	virtual ~RestrictToRepacking();

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	virtual std::string type() const { return "RestrictToRepacking"; }
};

/// @brief RestrictResidueToRepacking
class RestrictResidueToRepacking : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	virtual ~RestrictResidueToRepacking();

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	void include_residue( core::Size resid );
	void clear();

	virtual std::string type() const { return "RestrictResidueToRepacking"; }

private:
	utility::vector1< core::Size > residues_to_restrict_to_repacking_;
};

// @brief  RestrictAbsentCanonicalAAS
class RestrictAbsentCanonicalAAS : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	RestrictAbsentCanonicalAAS();
	RestrictAbsentCanonicalAAS( core::Size resid, utility::vector1< bool > keep_aas );
	virtual ~RestrictAbsentCanonicalAAS();

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	virtual std::string type() const { return "RestrictAbsentCanonicalAAS"; }

private:
	core::Size resid_;
	utility::vector1< bool > keep_aas_;

};

/// @brief rotamer explosion for a residue
class RotamerExplosion : public TaskOperation
{
public:
	typedef TaskOperation parent;
public:
	RotamerExplosion();
	RotamerExplosion( core::Size const resid, ExtraRotSample const sample_level, core::Size const chi );
	virtual ~RotamerExplosion();
	virtual TaskOperationOP clone() const;
	virtual void apply( pose::Pose const &, PackerTask & ) const;
	virtual std::string type() const { return "RotamerExplosion"; }
private:
	core::Size resid_, chi_;
	ExtraRotSample sample_level_;
};

class InitializeFromCommandline : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	virtual ~InitializeFromCommandline();

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	virtual std::string type() const { return "InitializeFromCommandline"; }
};

class InitializeExtraRotsFromCommandline : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	virtual ~InitializeExtraRotsFromCommandline();

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	virtual std::string type() const { return "InitializeExtraRotsFromCommandline"; }
};

class IncludeCurrent : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	virtual ~IncludeCurrent();

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	virtual std::string type() const { return "IncludeCurrent"; }
};

class ReadResfile : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	ReadResfile();
	ReadResfile( std::string const & );
	virtual ~ReadResfile();

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	void filename( std::string const & filename );

	///@brief queries options system for resfile name
	void default_filename();

	std::string const & filename() const;

	virtual void parse_tag( TagPtr, ResLvlTaskOperationFactoryCOP, ResFilterFactoryCOP );

	virtual std::string type() const { return "ReadResfile"; }

private:
	std::string resfile_filename_;
};


class SetRotamerCouplings : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	SetRotamerCouplings();
	SetRotamerCouplings(SetRotamerCouplings const & );
	virtual ~SetRotamerCouplings();
	SetRotamerCouplings const & operator = ( SetRotamerCouplings const & );

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	void
	set_couplings( rotamer_set::RotamerCouplingsOP couplings );

	virtual std::string type() const { return "SetRotamerCouplings"; }

private:
	rotamer_set::RotamerCouplingsCOP rotamer_couplings_;
};

///@brief when a PackerTask is created by the Factory, the RotamerOperation will be given to it
class AppendRotamer : public TaskOperation
{
public:
	typedef TaskOperation parent;
public:
	AppendRotamer();
	virtual ~AppendRotamer();
	AppendRotamer( rotamer_set::RotamerOperationOP rotamer_operation );
	AppendRotamer( AppendRotamer const & );
	virtual TaskOperationOP clone() const;
	virtual void apply( pose::Pose const &, PackerTask & ) const;
	void set_rotamer_operation( rotamer_set::RotamerOperationOP rotamer_operation );

	virtual std::string type() const { return "AppendRotamer"; }

private:
	rotamer_set::RotamerOperationOP rotamer_operation_;
};


///@brief when a PackerTask is created by the Factory, the RotamerSetOperation will be given to it
///@author Barak Raveh (Dec 2008)
class AppendRotamerSet : public TaskOperation
{
public:
	typedef TaskOperation parent;
public:
	AppendRotamerSet();
	virtual ~AppendRotamerSet();
	AppendRotamerSet( rotamer_set::RotamerSetOperationOP rotamer_set_operation );
	AppendRotamerSet( AppendRotamerSet const & );
	virtual TaskOperationOP clone() const;
	virtual void apply( pose::Pose const &, PackerTask & ) const;
	void set_rotamer_set_operation( rotamer_set::RotamerSetOperationOP rotamer_operation );

	virtual std::string type() const { return "AppendRotamerSet"; }

private:
	rotamer_set::RotamerSetOperationOP rotamer_set_operation_;
};


class PreserveCBeta : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	virtual ~PreserveCBeta();

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	virtual std::string type() const { return "PreserveCBeta"; }
};


/// @brief PreventRepacking allows you to prevent repacking (NATRO behavior) through the Factory.  Useful if you do not know the residue numbers when the resfile is created.  Note that this is unlike RestrictToRepacking; you have to specify which residues.  If PreventRepacking worked on the entire Task you'd have a do-nothing task.
class PreventRepacking : public TaskOperation
{
public:
	typedef TaskOperation parent;

public:
	virtual ~PreventRepacking();

	virtual TaskOperationOP clone() const;

	virtual
	void
	apply( pose::Pose const &, PackerTask & ) const;

	void include_residue( core::Size resid );
	void clear();

	virtual std::string type() const { return "PreventRepacking"; }

private:
	utility::vector1< core::Size > residues_to_prevent_;
};


/// @brief RestrictYSDesign restricts positions to a binary Tyr/Ser alphabet
class RestrictYSDesign : public TaskOperation {

public:
	typedef TaskOperation parent;

public:
	RestrictYSDesign();
	RestrictYSDesign( RestrictYSDesign const & src );
	RestrictYSDesign( utility::vector1< core::Size > const & resids );
	virtual ~RestrictYSDesign();

	virtual TaskOperationOP clone() const;
	virtual void apply( pose::Pose const &, PackerTask & task) const;
	void include_resid( core::Size const resid );
	void include_gly( bool const gly );
	// void clear();  // Not defined, commenting out to make Python binding compile

	virtual std::string type() const { return "RestrictYSDesign"; }

private:
	utility::vector1< core::Size > YSresids_;
	bool gly_switch_;
};

} //namespace operation
} //namespace task
} //namespace pack
} //namespace core

#endif
