// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pose/util.cc
/// @brief  Pose class utilities
/// @author Phil Bradley
/// @author Modified by Sergey Lyskov

// Unit headers
#include <core/pose/util.hh>
#include <core/pose/PDBInfo.hh>
#include <core/pose/Pose.hh>
#include <core/pose/datacache/CacheableDataType.hh>
#include <core/conformation/Conformation.hh>
#include <core/conformation/ResidueFactory.hh>
#include <core/conformation/util.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/ResidueTypeSet.hh>

#include <core/util/datacache/BasicDataCache.hh>

#include <protocols/jumping/Dssp.hh> //for setting ss of start-structure

// Utility headers
#include <core/util/Tracer.hh>
#include <core/util/datacache/CacheableStringFloatMap.hh>
#include <core/util/datacache/CacheableStringMap.hh>
#include <core/options/option.hh>
#include <utility/io/izstream.hh>

// option key includes

#include <core/options/keys/in.OptionKeys.gen.hh>

// Project headers

namespace core {
namespace pose {

static core::util::Tracer TR("core.pose.util");
////////////////////////////////////////////////////////////////////////////
void
create_subpose(
	Pose const & src,
	utility::vector1< Size > const & positions,
	kinematics::FoldTree const & f,
	Pose & pose
)
{
	Size const nres( f.nres() );
	assert( nres == positions.size() );

	pose.clear();

	// first add all as jumps
	for ( Size i=1; i<= nres; ++i ) {
		Size const seqpos( positions[i] );
		conformation::Residue const & rsd( src.residue( seqpos ) );
		pose.append_residue_by_jump( rsd, 1 );
		if ( i>1 ) {
			// check if this residue should be in a new chain. not a perfect check...
			conformation::Residue const & prev_rsd( src.residue( positions[i-1] ) );
			if ( prev_rsd.is_upper_terminus() || rsd.is_lower_terminus() || prev_rsd.chain() != rsd.chain() ) {
				assert( pose.total_residue() == i );
				pose.conformation().insert_chain_ending( i-1 );
			}
		}
	}

	// now set the desired foldtree
	pose.fold_tree(f);

}
////////////////////////////////////////////////////////////////////////////
void
partition_pose_by_jump(
	pose::Pose const & src,
	int const jump_number,
	pose::Pose & partner1,
	pose::Pose & partner2
)
{
	Size const nres( src.total_residue() );

	// split src pose's foldtree
	kinematics::FoldTree f1,f2;
	src.fold_tree().partition_by_jump( jump_number, f1, f2 );

	TR << src.fold_tree() << '\n' << f1 << '\n' << f2 << '\n';

	// identify residues in the two partners
	ObjexxFCL::FArray1D_bool partner1_pos( nres, false ); // FARRAY! DOH!!
	src.fold_tree().partition_by_jump( jump_number, partner1_pos );

	utility::vector1< Size > partner1_pos_list, partner2_pos_list;
	for ( Size i=1; i<= nres; ++i ) {
		if ( partner1_pos(i) ) partner1_pos_list.push_back( i );
		else partner2_pos_list.push_back( i );
	}

	create_subpose( src, partner1_pos_list, f1, partner1 );

	create_subpose( src, partner2_pos_list, f2, partner2 );

	src.dump_pdb( "complex.pdb" );
	partner1.dump_pdb( "partner1.pdb" );
	partner2.dump_pdb( "partner2.pdb" );
}
////////////////////////////////////////////////////////////////////////////////////////////////////
/// @details Crude way to guess secondary structure given a pose. This function
/// sets the sec_struct array of pose.conformation_ to the result of the
/// guesswork. This has been ported directly from rosetta++.
void
set_ss_from_phipsi(
	pose::Pose & pose
)
{
	// ss       :ss = 1 helix, ss = 2 sheet, ss = 3 other
	const int sstemp_offset=3;
	utility::vector1 < int > sstemp( sstemp_offset*2 + pose.total_residue() );
	utility::vector1 < int > ss( pose.total_residue() );

	// make sure phi,psi angles are between -180.0 and 180.0
	//for ( int i = 1; i <= pose.total_residue() ; ++i ) {
	//	pose.phi(i) = mod(pose.phi(i)+180.0f,360.0f)-180.0f;
	//	pose.psi(i) = mod(pose.psi(i)+180.0f,360.0f)-180.0f;
	//}

	sstemp[sstemp_offset-1] = 3; // assign loop to fictious residues at ends of chain
	sstemp[sstemp_offset+0] = 3;
	sstemp[sstemp_offset+pose.total_residue()+1] = 3;
	sstemp[sstemp_offset+pose.total_residue()+2] = 3;

	for ( Size i = 1; i <= pose.total_residue(); ++i ) {

	// <murphp>
		if( !pose.residue_type(i).is_protein() ) { // make sure we don't inquire about the phi/psi of a non-protein residue
			sstemp[sstemp_offset+i] = 3;
		}
		else {
			// </murphp>

			if ( pose.phi(i) < -20.0 && pose.psi(i) > -90.0 && pose.psi(i) < -10.0 ) {
				sstemp[sstemp_offset+i] = 1;
			} else if ( pose.phi(i) < -20.0 && (pose.psi(i) > 20.0 || pose.psi(i) < -170.0) ) {
				sstemp[sstemp_offset+i] = 2;
			} else {
				sstemp[sstemp_offset+i] = 3;
			}

		}
	}

	for ( Size i = 1; i <= pose.total_residue(); ++i ) {
		if ( sstemp[sstemp_offset+i] == 2 ) {
			if ( sstemp[sstemp_offset+i-1] == 2 && sstemp[sstemp_offset+i+1] == 2 ) {
				ss[i] = 2;
			} else if ( sstemp[sstemp_offset+i-1] == 2 && sstemp[sstemp_offset+i-2] == 2 ) {
				ss[i] = 2;
			} else if ( sstemp[sstemp_offset+i+1] == 2 && sstemp[sstemp_offset+i+2] == 2 ) {
				ss[i] = 2;
			} else {
				ss[i] = 3;
			}
		} else if ( sstemp[sstemp_offset+i] == 1 ) {
			if ( sstemp[sstemp_offset+i-1] == 1 && sstemp[sstemp_offset+i+1] == 1 ) {
				ss[i] = 1;
			} else if ( sstemp[sstemp_offset+i-1] == 1 && sstemp[sstemp_offset+i-2] == 1 ) {
				ss[i] = 1;
			} else if ( sstemp[sstemp_offset+i+1] == 1 && sstemp[sstemp_offset+i+2] == 1 ) {
				ss[i] = 1;
			} else {
				ss[i] = 3;
			}
		} else {
			ss[i] = 3;
		}
	}

	for ( Size i = 1; i <= pose.total_residue(); ++i ) {
		if ( ss[i] == 1 ) {
			pose.set_secstruct(i,'H');
		} else if ( ss[i] == 2 ) {
			pose.set_secstruct(i,'E');
		} else {
			pose.set_secstruct(i,'L');
		}
	}


}

// void
// set_ss_from_phipsi_with_dssp(
// 	pose::Pose & pose
// ) {
// 		// initialize secondary structure from DSSP.
// undefined reference to stuff from protocols...
// 	protocols::jumping::Dssp dssp_obj( pose );
// 	dssp_obj.insert_ss_into_pose( pose );
// }


////////////////////////////////////////////////////////////////////////////////////////////////////
/// @details Adds a virtual residue to a pose as the root.
/// Jump is to the residue closest to the center of mass.
/// If the pose is already rooted on a VRT res, do nothing.
void addVirtualResAsRoot( core::pose::Pose & pose ) {
	int nres = pose.total_residue();

	// if already rooted on virtual residue , return
	if ( pose.residue( pose.fold_tree().root() ).aa() == core::chemical::aa_vrt ) {
		TR.Warning << "addVirtualResAsRoot() called but pose is already rooted on a VRT residue ... continuing." << std::endl;
		return;
	}

	// check for terminal ligands
	int last_peptide_res = nres;
	while ( !pose.residue( last_peptide_res ).is_protein() )
		last_peptide_res--;

	// find residue closest to center-of-mass
	numeric::xyzVector< core::Real > massSum(0.0,0.0,0.0);
	int nAtms=0;
	for ( int i=1; i<= nres; ++i ) {
		conformation::Residue const & rsd( pose.residue(i) );
		if (rsd.aa() == core::chemical::aa_vrt) continue;
		for ( Size j=1; j<= rsd.nheavyatoms(); ++j ) {
			conformation::Atom const & atom( rsd.atom(j) );
			massSum += atom.xyz();
			nAtms++;
		}
	}
	massSum /= nAtms;

	// attach virt res there
	int i_min = 1;
	core::Real d_min = 99999, this_d;
	for ( int i=1; i<= last_peptide_res; ++i ) {
		conformation::Residue const & rsd( pose.residue(i) );

		if (rsd.aa() == core::chemical::aa_vrt) continue;
		if (!rsd.is_protein() ) continue;

		conformation::Atom const & atom( rsd.atom("CA") );
		this_d = (atom.xyz() - massSum).length();
		if (this_d < d_min) {
			d_min = this_d;
			i_min = i;
		}
	}
	bool fullatom = pose.is_fullatom();
	core::chemical::ResidueTypeSetCAP const &residue_set(
								          core::chemical::ChemicalManager::get_instance()->residue_type_set
	                            ( fullatom ? core::chemical::FA_STANDARD : core::chemical::CENTROID )
	                                                    );
	core::chemical::ResidueTypeCAPs const & rsd_type_list( residue_set->name3_map("VRT") );
	core::conformation::ResidueOP new_res( core::conformation::ResidueFactory::create_residue( *rsd_type_list[1] ) );
	pose.append_residue_by_jump( *new_res , i_min );

	// make the virt atom the root
	kinematics::FoldTree newF( pose.fold_tree() );
	newF.reorder( nres+1 );
	pose.fold_tree( newF );
}


////////////////////////////////////////////////////////////////////////////////////////////////////
bool getPoseExtraScores(
	core::pose::Pose & pose,
	std::string name,
	float & value
) {
	using core::pose::datacache::CacheableDataType::ARBITRARY_FLOAT_DATA;
	using core::util::datacache::CacheableStringFloatMap;

	// make sure that the pose has one of these.
	if( !pose.data().has( ARBITRARY_FLOAT_DATA ) ){
		return false;
	}

	CacheableStringFloatMap *data
		= dynamic_cast< CacheableStringFloatMap* >
			( pose.data().get_raw_ptr( ARBITRARY_FLOAT_DATA ) );

	assert( data != NULL );

	value = data->map()[ name ];
	return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
void setPoseExtraScores(
	core::pose::Pose & pose,
	std::string name,
	float value
) {
	using core::pose::datacache::CacheableDataType::ARBITRARY_FLOAT_DATA;
	using core::util::datacache::CacheableStringFloatMap;

	// make sure that the pose has one of these.
	if ( !pose.data().has( ARBITRARY_FLOAT_DATA ) ) {
		pose.data().set(
			ARBITRARY_FLOAT_DATA,
			new CacheableStringFloatMap()
		);
	}

	CacheableStringFloatMap *data
		=  dynamic_cast< CacheableStringFloatMap* >
			( pose.data().get_raw_ptr(ARBITRARY_FLOAT_DATA) );

	runtime_assert( data != NULL );
	data->map()[name] = value;
}

void add_comment(
	core::pose::Pose & pose,
	std::string const & key,
	std::string const & val
) {
	using core::pose::datacache::CacheableDataType::STRING_MAP;
	using core::util::datacache::CacheableStringMap;

	// make sure that the pose has a map of strings
	if ( !pose.data().has( STRING_MAP ) ) {
		pose.data().set(
			STRING_MAP,
			new CacheableStringMap()
		);
	}

	CacheableStringMap *data
		=  dynamic_cast< CacheableStringMap* >
			( pose.data().get_raw_ptr(STRING_MAP) );

	runtime_assert( data != NULL );
	data->map()[key] = val;
} // add_comment

void add_score_line_string(
	core::pose::Pose & pose,
	std::string const & key,
	std::string const & val
) {
	using core::pose::datacache::CacheableDataType::SCORE_LINE_STRINGS;
	using core::util::datacache::CacheableStringMap;

	// make sure that the pose has a map of strings
	if ( !pose.data().has( SCORE_LINE_STRINGS ) ) {
		pose.data().set(
			SCORE_LINE_STRINGS,
			new CacheableStringMap()
		);
	}

	CacheableStringMap *data
		=  dynamic_cast< CacheableStringMap* >
			( pose.data().get_raw_ptr(SCORE_LINE_STRINGS) );

	runtime_assert( data != NULL );
	data->map()[key] = val;
}

void clearPoseExtraScores(
	core::pose::Pose & pose
) {
	using core::pose::datacache::CacheableDataType::ARBITRARY_FLOAT_DATA;
	using core::util::datacache::CacheableStringFloatMap;
	pose.data().set(
		ARBITRARY_FLOAT_DATA,
		new CacheableStringFloatMap()
	);
}

bool get_comment(
	core::pose::Pose const & pose,
	std::string const & key,
	std::string & val
) {
	using std::map;
	using std::string;
	map< string, string > comment_map = get_all_comments( pose );

	if ( comment_map.find( key ) == comment_map.end() ) {
		return false;
	}

	val = comment_map[ key ];
	return true;
}

bool get_score_line_string(
	core::pose::Pose const & pose,
	std::string const & key,
	std::string & val
) {
	using std::map;
	using std::string;
	map< string, string > score_line_strings_map = get_all_score_line_strings( pose );

	if ( score_line_strings_map.find( key ) == score_line_strings_map.end() ) {
		return false;
	}

	val = score_line_strings_map[ key ];
	return true;
}

void delete_comment(
	core::pose::Pose & pose,
	std::string const & key
) {
	using core::pose::datacache::CacheableDataType::STRING_MAP;
	using core::util::datacache::CacheableStringMap;

	if ( pose.data().has( STRING_MAP ) ) {
		CacheableStringMap *data
			=  dynamic_cast< CacheableStringMap* >
				( pose.data().get_raw_ptr(STRING_MAP) );
		std::map< std::string, std::string >::iterator it;
		it = data->map().find(key);
		if ( it != data->map().end() ) {
			data->map().erase(it);
		}
	}
}

std::map< std::string, std::string > get_all_score_line_strings(
	core::pose::Pose const & pose
) {
	using core::pose::datacache::CacheableDataType::SCORE_LINE_STRINGS;
	using core::util::datacache::CacheableStringMap;

	std::map< std::string, std::string > score_line_strings;
	if ( pose.data().has( SCORE_LINE_STRINGS ) ) {
		const CacheableStringMap *data
			= dynamic_cast< const CacheableStringMap* >
				( pose.data().get_raw_const_ptr( SCORE_LINE_STRINGS ) );
		score_line_strings = data->map();
		runtime_assert( data != NULL );
	}

	return score_line_strings;
}

std::map< std::string, std::string > get_all_comments(
	core::pose::Pose const & pose
) {
	using core::pose::datacache::CacheableDataType::STRING_MAP;
	using core::util::datacache::CacheableStringMap;

	std::map< std::string, std::string > comments;
	if ( pose.data().has( STRING_MAP ) ) {
		const CacheableStringMap *data
			= dynamic_cast< const CacheableStringMap* >
				( pose.data().get_raw_const_ptr( STRING_MAP ) );
		comments = data->map();
		runtime_assert( data != NULL );
	}

	return comments;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
utility::vector1< char > read_psipred_ss2_file( pose::Pose const & pose ) {
	using namespace core::options;

	std::string filename( option[ OptionKeys::in::file::psipred_ss2 ]().name() );

	utility::vector1< char > secstructs;
	utility::io::izstream data( filename );

	if ( !data ) {
		TR.Warning << "Cannot open psipred_ss2 file " << filename << std::endl;
		return secstructs;
	}

	std::string line;
	Size count(0);
	while( getline( data, line ) ) {
		if( line[0] == '#' || line == "" )
			continue;
		std::istringstream line_stream( line );
		Size pos;
		char aa, sec;
		line_stream >> pos >> aa >> sec;
		count++;
		if ( sec != 'H' && sec != 'E' && sec != 'C' ) {
			TR.Warning << "unrecognized secstruct char : " << sec << " at seqpos " << count << std::endl;
		}
		if ( sec == 'C' ) {
			secstructs.push_back( 'L' );
		} else {
			secstructs.push_back( sec );
		}
	}

	// chu get number of protein residues
	Size nres=0;
	for ( Size i =1 ; i <= pose.total_residue(); ++i ) {
		if ( pose.residue(i).is_protein() ) nres++;
	}

	assert( secstructs.size() == nres);
	if( secstructs.size() != nres )
		secstructs.clear();

	return secstructs;
}
////////////////////////////////////////////////////////////////////////////////////////////////////


/// @brief get Conformation chain -> PDBInfo chain mapping
/// @remarks Any chains whose PDBInfo chain records are marked entirely as
///  PDBInfo::empty_record() will be mapped to that character.  Note that
///  Conformation -> PDBInfo is always unique, but the reverse may not be true.
/// @return the mapping if PDBInfo available and chains appear consistent,
///  otherwise returns an empty mapping
std::map< int, char > conf2pdb_chain( core::pose::Pose const & pose ) {
	using core::Size;
	using core::pose::PDBInfo;
	typedef std::map< int, char > Conf2PDB;

	Conf2PDB conf2pdb;

	if ( !pose.pdb_info().get() ) {
		TR.Warning << "WARNING: conf2pdb_chain(): PDBInfo does not exist, returning empty map" << std::endl;
		return conf2pdb;
	}

	for ( Size i = 1, ie = pose.n_residue(); i <= ie; ++i ) {
		int const conf = pose.chain( i );
		char const pdb = pose.pdb_info()->chain( i );

		Conf2PDB::iterator c2p = conf2pdb.find( conf );
		if ( c2p != conf2pdb.end() ) { // must check if existing record inconsistent
			if ( c2p->second != pdb ) {

				// three cases:
				//  (1) replace an existing empty record
				//  (2) it's an unneeded empty record, so continue
				//  (3) there is an actual problem
				if ( pdb != PDBInfo::empty_record() && c2p->second == PDBInfo::empty_record() ) {
					// replace the record
					c2p->second = pdb;
				} else if ( pdb == PDBInfo::empty_record() ) {
					continue; // skip empty record
				} else {
					// something is inconsistent
					TR.Warning << "WARNING: conf2pdb_chain(): chain mapping inconsistent, returning empty map; ";
					TR.Warning << "existing " << c2p->first << " -> " << c2p->second << "  |  ";
					TR.Warning << "new " << conf << " -> " << pdb << std::endl;
					conf2pdb.clear();
					return conf2pdb;
				}

			}
		} else { // record doesn't exist yet
			conf2pdb[ conf ] = pdb;
		}

	} // foreach residue

	assert( conf2pdb.size() == pose.conformation().num_chains() );
	return conf2pdb;
}


/// @brief renumber PDBInfo based on Conformation chains; each chain starts from 1
/// @param[in,out] pose The Pose to modify.
/// @param[in] fix_chains If true, the procedure will attempt to fix any empty record
///  characters it finds in the PDBInfo. (default true)
/// @param[in] start_from_existing_numbering If true, will attempt to start each
///  chain from the existing numbering in the PDBInfo.  E.g. if the first residue
///  of chain 2 in the Conformation is 27, then the renumbering of the chain in
///  PDBInfo will start from 27. (default true)
/// @param[in] keep_insertion_codes If true, will maintain insertion codes and
///  will not increment the pdb residue numbering for those residues.  This means
///  new numbering with insertion codes will only reflect properly if the
///  old numbering included the base numbering of the insertion code residues,
///  i.e. 100 100A 100B and not just 100A 100B (with 100 never appearing).
///  (default false)
/// @param[in] rotate_chain_ids If true, allows support for more than 26 pdb chains
///  by rotating [A,Z] continuously.  WARNING: This will break the assumption
///  made by the PDBPoseMap that each pdb chain id is unique, so make sure you
///  are not using the PDBPoseMap feature downstream in your code path without
///  corrections! (default false)
/// @remarks If fixing chains and there is only one chain and the PDBInfo exists
///  but all records are marked as empty, will renumber and set the PDBInfo chain
///  to 'A'.
/// @return true if renumbering successful, false otherwise
bool renumber_pdbinfo_based_on_conf_chains(
	core::pose::Pose & pose,
	bool fix_chains,
	bool const start_from_existing_numbering,
	bool const keep_insertion_codes,
	bool const rotate_chain_ids
)
{
	using core::Size;
	using core::pose::PDBInfo;
	typedef std::map< int, char > Conf2PDB;

	if ( !pose.pdb_info().get() ) {
		TR.Warning << "WARNING: renumber_pdbinfo_based_on_conf_chains(): no PDBInfo, returning" << std::endl;
		return false;
	}

	Conf2PDB conf2pdb = conf2pdb_chain( pose );

	if ( fix_chains ) {
		if ( conf2pdb.empty() ) { // something is wrong with chain consistency
			TR.Warning << "WARNING: renumber_pdbinfo_based_on_conf_chains(): Request to fix PDBInfo chains, but ";
			TR.Warning << "chain mapping is inconsistent, so that step will be skipped." << std::endl;
			fix_chains = false;
		} else { // Try to fill in any empty record characters.

			// two different schemes: rotating and fixed length
			// WARNING: Rotating will break assumption of unique chain ids
			// inside PDBPoseMap, so make sure you are not using the PDBPoseMap
			// feature after calling this function without correcting!
			// First either remove or rotate any existing chains to the end of
			// the list.
			std::string letters( "ABCDEFGHIJKLMNOPQRSTUVWXYZ" );
			for ( Conf2PDB::iterator i = conf2pdb.begin(), ie = conf2pdb.end(); i != ie; ++i ) {
				if ( i->second != PDBInfo::empty_record() ) {
					std::string::size_type const j = letters.find( i->second );
					if ( j != std::string::npos ) {
						if ( rotate_chain_ids ) { // rotating
							letters.push_back( letters.at( j ) );
						}
						letters.erase( j, 1 );
					}
				}
			}

			// Now fill in empty records.
			Size lidx = 0;
			for ( Conf2PDB::iterator i = conf2pdb.begin(), ie = conf2pdb.end(); i != ie; ++i ) {
				if ( i->second == PDBInfo::empty_record() ) {
					if ( rotate_chain_ids ) { // rotating
						i->second = letters.at( lidx % letters.size() );
					} else { // fixed length
						runtime_assert( lidx < letters.size() );
						i->second = letters.at( lidx );
					}
					++lidx;
				}
			}

		} // if conf2pdb.empty()
	} // if fix_chains

	PDBInfo & pdbinfo = *pose.pdb_info();

	// grab all the chain endings
	utility::vector1< Size > chain_endings = pose.conformation().chain_endings();
	chain_endings.push_back( pose.n_residue() ); // add the last res, which is not in the list

	Size res = 1;
	for ( utility::vector1< Size >::const_iterator i = chain_endings.begin(), ie = chain_endings.end(); i != ie; ++i ) {
		Size const chain_end = *i;
		int pdb_res = 0; // new chain, so reset pdb_res counter
		char chain;
		char icode;
		if ( start_from_existing_numbering && pdbinfo.chain( res ) != PDBInfo::empty_record() ) {
			pdb_res = pdbinfo.number( res ) - 1;
		}

		// find the corresponding pdb chain
		Conf2PDB::const_iterator c2p = conf2pdb.find( pose.chain( chain_end ) );
		assert( ( fix_chains && c2p != conf2pdb.end() ) || !fix_chains ); // otherwise something's very wrong

		for ( ; res <= chain_end; ++res ) {
			// handle the pdb chain only if necessary
			chain = pdbinfo.chain( res );
			if ( pdbinfo.chain( res ) == PDBInfo::empty_record() && fix_chains && c2p != conf2pdb.end() ) {
				chain = c2p->second;
			}

			// If keeping insertion codes, increment pdb_res counter only if
			// no insertion code or we're at position 1, in case there's an
			// insertion code at 1.
			icode = pdbinfo.icode( res );
			if ( keep_insertion_codes && ( pdbinfo.icode( res ) == ' ' || res == 1 ) ) {
				++pdb_res;
			} else if ( !keep_insertion_codes ) { // always increment and clear insertion code
				icode = ' ';
				++pdb_res;
			}

			// The new pdb info for this residue must be setup in one shot.
			// The way we're redoing the info in this function can cause the
			// pdb2pose map to become out-of-sync if we attempt to make the
			// changes by separately calling chain(), icode(), and number().
			pdbinfo.set_resinfo( res, chain, pdb_res, icode );
		}

	} // foreach chain ending
	assert( res == pose.n_residue() + 1 );

	return true;
}

/// @brief checks if the pose geometry is ideal
/// @param[in] pose The Pose to check.
/// @return true if all pose positions have ideal bond lengths and angles
///  up to some very small epsilon
bool is_ideal_pose(
	core::pose::Pose const & pose
) {
	bool is_ideal=true;
	for(core::Size i=1 ; i <= pose.total_residue(); i++)
		{
			if(!is_ideal_position(i, pose))
				{
					is_ideal=false;
					break;
				}
		}
	return is_ideal;
}

/// @brief checks if the pose geometry is ideal in position seqpos
/// @param[in] pose The Pose to check.
/// @return true if position seqpos has ideal bond lengths and angles
///  up to some very small epsilon
bool is_ideal_position(
	core::Size seqpos,
	core::pose::Pose const & pose
)
{
	return conformation::is_ideal_position(
		seqpos,	pose.conformation()	);
}

///@brief this function removes all residues from the pose which are not protein residues.  This removal includes, but is not limited to, metals, DNA, RNA, and ligands.  It will NOT remove ligands which are canonical residues (for example, if a protein binds an alanine monomer, the monomer will be untouched).
void remove_nonprotein_residues( core::pose::Pose & pose)
{
	core::Size i(1);
	while(i <= pose.total_residue()) {
		if(!pose.residue_type(i).is_protein()) pose.conformation().delete_residue_slow(i);
		else ++i;
	}
}

///@brief this function removes all residues with both UPPER and LOWER terminus types.  This is intended for removing ligands that are canonical residues.
void remove_ligand_canonical_residues( core::pose::Pose & pose)
{
	core::Size i(1);
	while(i <= pose.total_residue()) {
		if(pose.residue_type(i).is_upper_terminus() && pose.residue_type(i).is_lower_terminus())
			pose.conformation().delete_residue_slow(i);
		else ++i;
	}
}

} // pose
} // core
