// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/InterchainEnergy.cc
/// @brief  Statistically derived rotamer pair potentials
/// @detailed For docking (or between chains) only those residues at the interface
///						and between the two interfaces need to be evaluated
/// @author Monica Berrondo


// Unit headers
#include <core/scoring/InterchainPotential.hh>

#include <core/scoring/AtomVDW.hh>
#include <core/scoring/EnvPairPotential.hh>
#include <core/scoring/ScoringManager.hh>

// Package headers

#include <core/scoring/EnergyGraph.hh>

// Project headers
#include <core/chemical/AA.hh>
#include <core/chemical/VariantType.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/conformation/Residue.hh>
#include <core/io/database/open.hh>
#include <core/pose/Pose.hh>
#include <core/pose/datacache/CacheableDataType.hh>
#include <core/util/datacache/BasicDataCache.hh>

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/utility.functions.hh>

// just for debugging
//#include <ObjexxFCL/formatted.o.hh>

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TC("core.scoring.InterchainPotential");



// C++

namespace core {
namespace scoring {

static Size max_interchain_sites ( 2 );


InterfaceInfo::InterfaceInfo( InterfaceInfo const & src ) :
	CacheableData(), calculated_(false)
{
	interface6A_ = src.interface6A_;
	interface8A_ = src.interface8A_;

	pair_list6A_ = src.pair_list6A_;
	pair_list8A_ = src.pair_list8A_;
}

void
InterfaceInfo::initialize( pose::Pose const & pose )
{
	Size rb_jump( rbjump_from_pose( pose ) );

	interface6A_ = new conformation::Interface( rb_jump );
	interface8A_ = new conformation::Interface( rb_jump );

	pair_list6A_.clear();
	pair_list6A_.resize( max_interchain_sites );
	pair_list8A_.clear();
	pair_list8A_.resize( max_interchain_sites );
}

/// @details If there is no rb jump in the pose, it is set to jump number 1
Size
InterfaceInfo::rbjump_from_pose( pose::Pose const & ) const
{
	// TODO currently not working
//	if ( pose.data().has( util::RB_JUMP ) ) {
//		return ( static_cast< int >( pose.data().get( util::RB_JUMP ) ) );
//	}
	return size_t(1);
}


InterchainPotential::InterchainPotential():
	EnvPairPotential(),
	atom_vdw_( ScoringManager::get_instance()->get_AtomVDW( chemical::CENTROID ) ) // need to make the table choice configurable
{

	// load the data
	Size const max_aa( 20 ); // just the standard aa's for now
	Size const env_log_table_size( 4 );

	std::string tag,line;
	chemical::AA aa;

	{ // interchain_env_log
		interchain_env_log_.dimension( max_aa, env_log_table_size );

		utility::io::izstream stream;
		io::database::open( stream, "interchain_env_log.txt" );
		while ( getline( stream, line ) ) {
			std::istringstream l(line);
			l >> tag >> aa;
			for ( Size i=1; i<= env_log_table_size; ++i ){
				l >> interchain_env_log_(aa,i);
			}
			if ( l.fail() || tag != "INT_CHAIN_ENV_LOG:"  ) utility_exit_with_message("bad format for interchain_env_log.txt");
		}
	}
	{ // interchain_pair_log
		interchain_pair_log_.dimension( max_aa, max_aa );

		utility::io::izstream stream;
		io::database::open( stream, "interchain_pair_log.txt" );
		while ( getline( stream, line ) ) {
			std::istringstream l(line);
			l >> tag >> aa;
			for ( Size i=1; i<= max_aa; ++i ) {
				l >> interchain_pair_log_(aa,i);
			}
			if ( l.fail() || tag != "INT_CHAIN_PAIR_LOG:"  ) utility_exit_with_message("bad format for interchain_pair_log.txt");
		}
	}
}

void
InterchainPotential::compute_interface(
	pose::Pose & pose
	) const
{
	InterfaceInfo & interface( nonconst_interface_from_pose( pose ) );

	/// initialize the cenlist info:
	/// only if they have not been calculated since the last score
	if ( !interface.calculated() ) {

		// initialize values
		interface.initialize( pose );

		// compute interpolated number of neighbors at various distance cutoffs
		interface.interface6A()->distance( 6.0 );
		interface.interface6A()->calculate( pose );
		interface.pair_list6A() = interface.interface6A()->pair_list();
		//interface.interface6A()->print( pose );
		//interface.interface8A()->distance( 8.0 );
		//interface.interface8A()->calculate( pose );
		//interface.pair_list8A() = interface.interface8A()->pair_list();
		//interface.interface8A()->print( pose );
	}

	interface.calculated() = true;
}

void
InterchainPotential::finalize( pose::Pose & pose ) const
{
	CenListInfo & cenlist( nonconst_cenlist_from_pose( pose ));
	cenlist.calculated() = false;

	InterfaceInfo & interface( nonconst_interface_from_pose( pose ) );
	interface.calculated() = false;
}

////////////////////////////////////////////////////////////////////////////////////
void
InterchainPotential::evaluate_env_score(
	pose::Pose const & pose,
	conformation::Residue const & rsd,
	Real & env_score
) const
{
	int env;
	Real const fcen10 ( cenlist_from_pose( pose ).fcen10(rsd.seqpos()) );

	InterfaceInfo const & interface( interface_from_pose( pose ) );

	if ( rsd.is_protein() ) {
		if ( interface.interface6A()->is_interface( rsd ) ) {
			if ( fcen10 > 16 )
				env = 1;
			else
				env = 2;
		} else {
			if ( fcen10 > 16 )
				env = 3;
			else
				env = 4;
		}

		env_score = interchain_env_log_( rsd.aa(), env );
		//std::cout << " residue: " << rsd.seqpos() << " res(i) " << rsd.aa()
		//	<< " env " << env << " score " << env_score << "\n";
	} else { // amino acid check
		env_score = 0.0;
	}
}


///////////////////////////////////////////////////////////////////////////////////////////////

void
InterchainPotential::evaluate_pair_and_vdw_score(
	pose::Pose const & pose,
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	Real & pair_contribution,
	Real & vdw_contribution
) const
{
	InterfaceInfo const & interface( interface_from_pose( pose ) );

	pair_contribution = 0.0;
	vdw_contribution = 0.0;

	if ( !interface.interface6A()->is_pair( rsd1, rsd2 ) ) return;

	if ( !rsd1.is_protein() || !rsd2.is_protein() ) return;

	// is this going to double count?
	pair_contribution = interchain_pair_log_( rsd1.aa(), rsd2.aa() );

	// calculation for vdw?
	// atoms between two chains are guaranteed to not be bonded
	// no countpair!
	for ( Size i=1, i_end = rsd1.natoms(); i<= i_end; ++i ) {
		Vector const & i_xyz( rsd1.xyz(i) );
		Size const i_type( rsd1.atom_type_index(i) );
		utility::vector1< Real > const & i_atom_vdw( atom_vdw_( i_type ) );
		for ( Size j=1, j_end = rsd2.natoms(); j<= j_end; ++j ) {
			Real const bump_dis( i_atom_vdw[ rsd2.atom_type_index(j) ] );
			Real const clash( bump_dis - i_xyz.distance_squared( rsd2.xyz(j) ) );
			if ( clash > 0.0 ) {
				vdw_contribution += ( clash * clash ) / bump_dis;
			}
		}
	}
}

Size
InterchainPotential::interface_residues( pose::Pose const & pose ) const
{
	InterfaceInfo const & interface( interface_from_pose( pose ) );
	Size partner1 (1), partner2 (2);
	return interface.pair_list6A_size(partner1) + interface.pair_list6A_size(partner2);
}

/// @details Pose must already contain a Interface object or this method will fail
InterfaceInfo const &
InterchainPotential::interface_from_pose( pose::Pose const & pose ) const
{
	using core::pose::datacache::CacheableDataType::INTERFACE_INFO;
	return *( static_cast< InterfaceInfo const * >(pose.data().get_const_ptr( INTERFACE_INFO )() ) );
}

/// @details Either returns a non-const reference to the Interface object that already exists
/// in the pose, or creates a new Interface object, places it in the pose, and then returns
/// a non-const reference to it
InterfaceInfo &
InterchainPotential::nonconst_interface_from_pose( pose::Pose & pose ) const
{
	using core::pose::datacache::CacheableDataType::INTERFACE_INFO;

	if ( pose.data().has( INTERFACE_INFO ) ) {
		return *( static_cast< InterfaceInfo * >(pose.data().get_ptr( INTERFACE_INFO )() ) );
	}
	// else
	InterfaceInfoOP interface = new InterfaceInfo;
	pose.data().set( INTERFACE_INFO, interface );
	return *interface;
}

}
}
