// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/InterchainEnergy.cc
/// @brief  Statistically derived rotamer pair potentials
/// @detailed For docking (or between chains) only those residues at the interface
///						and between the two interfaces need to be evaluated
/// @author Monica Berrondo

#ifndef INCLUDED_core_scoring_InterchainPotential_HH
#define INCLUDED_core_scoring_InterchainPotential_HH

#include <core/types.hh>

// Unit headers
#include <core/scoring/AtomVDW.fwd.hh>
#include <core/scoring/InterchainPotential.fwd.hh>
#include <core/scoring/EnvPairPotential.hh>

// Package headers
#include <core/scoring/EnergyGraph.fwd.hh>
#include <core/conformation/Interface.hh>
#include <core/conformation/Residue.fwd.hh>

#include <core/util/datacache/CacheableData.hh>


// Project headers

// Utility headers
#include <utility/vector1.hh>

#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
// C++


namespace core {
namespace scoring {
using namespace conformation;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief Keep track of the interface information

class InterfaceInfo : public util::datacache::CacheableData {
public:


	InterfaceInfo(): calculated_(false) {};

	InterfaceInfo( InterfaceInfo const & src );

	util::datacache::CacheableDataOP
	clone() const
	{
		return new InterfaceInfo( *this );
	}

	InterfaceOP
	interface6A()
	{
		return interface6A_;
	}

	InterfaceOP
	interface6A() const
	{
		return interface6A_;
	}

	InterfaceOP
	interface8A()
	{
		return interface8A_;
	}

	InterfaceOP
	interface8A() const
	{
		return interface8A_;
	}

	utility::vector1 < utility::vector1_int > &
	pair_list6A()
	{
		return pair_list6A_;
	}

	utility::vector1 < utility::vector1_int >
	pair_list6A() const
	{
		return pair_list6A_;
	}

	Size
	pair_list6A_size( Size pos ) const
	{
		return pair_list6A_[pos].size();
	}

	utility::vector1 < utility::vector1_int > &
	pair_list8A()
	{
		return pair_list8A_;
	}

	utility::vector1 < utility::vector1_int >
	pair_list8A() const
	{
		return pair_list8A_;
	}

	Size
	pair_list8A_size( Size pos ) const
	{
		return pair_list8A_[pos].size();
	}

	bool
	calculated() const
	{
		return calculated_;
	}

	bool &
	calculated()
	{
		return calculated_;
	}

	void
	initialize( pose::Pose const & );

	Size rbjump_from_pose( pose::Pose const & ) const;

private:
	bool calculated_;
	InterfaceOP interface6A_;
	InterfaceOP interface8A_;

	utility::vector1 < utility::vector1_int > pair_list6A_;
	utility::vector1 < utility::vector1_int > pair_list8A_;
};

class InterchainPotential : public EnvPairPotential {

public:
	InterchainPotential();

	void
	compute_interface( pose::Pose & pose ) const;

	void
	finalize( pose::Pose & pose ) const;

	///
	void
	evaluate_env_score(
		pose::Pose const & pose,
		conformation::Residue const & rsd,
		Real & env_score
	) const;

	///
	void
	evaluate_pair_and_vdw_score(
		pose::Pose const & pose,
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		Real & pair_score,
		Real & vdw_score
	) const;

	Size
	interface_residues( pose::Pose const & pose ) const;

	InterfaceInfo const & interface_from_pose( pose::Pose const & ) const;
	InterfaceInfo & nonconst_interface_from_pose(  pose::Pose & ) const;

private: // data

	// const-ref to scoring database
	AtomVDW const & atom_vdw_;

	ObjexxFCL::FArray2D< Real > interchain_env_log_;
	ObjexxFCL::FArray2D< Real > interchain_pair_log_;
};

} // ns scoring
} // ns core

#endif
