// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/EnvPairPotential.cc
/// @brief  Membrane Potential
/// @author Bjorn Wallner
///


// Unit headers
#include <core/scoring/MembranePotential.hh>
#include <core/scoring/MembranePotential.fwd.hh>
#include <core/scoring/MembraneTopology.hh>

// Package headers

#include <core/scoring/EnergyGraph.hh>
#include <core/scoring/EnvPairPotential.hh>

// Project headers
#include <core/chemical/AA.hh>
#include <core/chemical/VariantType.hh>
#include <core/conformation/Conformation.hh>
#include <core/conformation/Residue.hh>
#include <core/io/database/open.hh>
#include <core/pose/Pose.hh>
#include <core/pose/datacache/CacheableDataType.hh>
#include <core/util/datacache/BasicDataCache.hh>

#include <core/options/option.hh>
#include <core/options/after_opts.hh>
#include <core/options/keys/membrane.OptionKeys.gen.hh>

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/utility.functions.hh>
#include <core/types.hh>


#include <utility/vector1.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/random/random.hh>
#include <numeric/conversions.hh>

#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>



// just for debugging
//#include <ObjexxFCL/formatted.o.hh>

// C++


namespace core {
namespace scoring {
static numeric::random::RandomGenerator RG(280628);  // <- Magic number, do not change it!

MembraneEmbed::MembraneEmbed( MembraneEmbed const & src ) :
	CacheableData()
{
	depth_=src.depth_;
	center_=src.center_;
	normal_=src.normal_;
	spanning_=src.spanning_;
	calculated_ = src.calculated_;
}

void
MembraneEmbed::initialize(pose::Pose const & pose)
{
	Size const nres( pose.total_residue() );
	depth_.resize(nres,0.0);
	std::fill(depth_.begin(),depth_.end(),30.0); //bw why not?
	center_.assign(0,0,0);
	normal_.assign(0,0,1);

}




MembranePotential::MembranePotential():
/*	cen_dist_cutoff2( 144.0 ),

	cen_dist6sqr_( 6 * 6 ),
	cen_dist10sqr_( 10 * 10 ),
	cen_dist12sqr_( 12 * 12 ),
*/
	//cems transition regions between environment bins
	//cems transition is from +/- sqrt(36+pad6) +/- sqrt(100+pad10) etc
	cen_dist5_pad( 0.5 ),
	cen_dist6_pad( 0.6 ),
	cen_dist7_pad( 0.65 ),
	cen_dist10_pad( 1.0 ),
	cen_dist12_pad( 1.2 ),

	cen_dist5_pad_plus ( cen_dist5_pad  + 25.0 ),
	cen_dist6_pad_plus( cen_dist6_pad + 36.0 ),
	cen_dist7_pad_plus ( cen_dist7_pad  + 56.25 ),
	cen_dist10_pad_plus( cen_dist10_pad + 100.0 ),
	cen_dist12_pad_plus( cen_dist12_pad + 144.0 ),

	cen_dist5_pad_minus ( cen_dist5_pad  - 25.0 ),
	cen_dist7_pad_minus ( cen_dist7_pad  - 56.25 ),
	cen_dist10_pad_minus( cen_dist10_pad - 100.0 ),
	cen_dist12_pad_minus( cen_dist12_pad - 144.0 ),

	cen_dist5_pad_hinv ( 0.5 / cen_dist5_pad ),
	cen_dist6_pad_hinv ( 0.5 / cen_dist6_pad ),
	cen_dist7_pad_hinv ( 0.5 / cen_dist7_pad ),
	cen_dist10_pad_hinv( 0.5 / cen_dist10_pad ),
	cen_dist12_pad_hinv( 0.5 / cen_dist12_pad ),
	calculated_(false)

{

	// load the data
	Size const max_aa( 20 ); // just the standard aa's for now
	//Size const env_log_table_size( 40 );
	Size const env_log_table_cen6_bins( 6 );
	Size const env_log_table_cen10_bins( 6 );

	Size const pair_log_table_size( 5 );
	Size const cbeta_den_table_size( 45 );
	Size const max_mem_layers( 4 );
	Size const min_mem_layers( 2 );


		//init some variables from command line
	membrane_normal_cycles_=core::options::option[core::options::OptionKeys::membrane::normal_cycles]();
	membrane_normal_magnitude_=numeric::conversions::radians(core::options::option[ core::options::OptionKeys::membrane::normal_mag ]());
	membrane_center_magnitude_=core::options::option[ core::options::OptionKeys::membrane::center_mag ]();
	smooth_move_frac_=core::options::option[ core::options::OptionKeys::membrane::smooth_move_frac ]();
	no_interpolate_Mpair_=core::options::option[ core::options::OptionKeys::membrane::no_interpolate_Mpair ]();
	Menv_penalties_=core::options::option[ core::options::OptionKeys::membrane::Menv_penalties ]();

	std::string tag,line;
	chemical::AA aa;

	{ // mem_env_log
		mem_env_log6_.dimension( max_aa, max_mem_layers,env_log_table_cen6_bins );
		mem_env_log10_.dimension( max_aa, max_mem_layers,env_log_table_cen10_bins );

		utility::io::izstream stream;
		io::database::open( stream, "mem_env_log.txt" );
		for ( Size i=1; i<= max_aa; ++i ){
			getline( stream, line );
			std::istringstream l(line);
			l >> tag >> aa;
			//	std::cout << i << " " << line << std::endl;
			if ( l.fail() || tag != "MEM_ENV_LOG_CEN6:"  ) utility_exit_with_message("bad format for mem_env_log.txt (cen6)");
			for( Size j=1;j<=max_mem_layers;++j) {
				for( Size k=1;k<=env_log_table_cen6_bins;++k) {
					l >> mem_env_log6_(aa,j,k);
				}
			}

		}
		for ( Size i=1; i<= max_aa; ++i ){
			getline( stream, line );
			std::istringstream l(line);
			l >> tag >> aa;
			if ( l.fail() || tag != "MEM_ENV_LOG_CEN10:"  ) utility_exit_with_message("bad format for mem_env_log.txt (cen10)");
			for( Size j=1;j<=max_mem_layers;++j) {
				for( Size k=1;k<=env_log_table_cen6_bins;++k) {
						l >> mem_env_log10_(aa,j,k);
				}
			}
		}
	}

	{ // cbeta_den_6/12
		mem_cbeta_den6_.dimension( cbeta_den_table_size );
		mem_cbeta_den12_.dimension( cbeta_den_table_size );
		mem_cbeta_2TM_den6_.dimension( cbeta_den_table_size );
		mem_cbeta_2TM_den12_.dimension( cbeta_den_table_size );
		mem_cbeta_4TM_den6_.dimension( cbeta_den_table_size );
		mem_cbeta_4TM_den12_.dimension( cbeta_den_table_size );

		utility::io::izstream stream;
		io::database::open( stream, "memcbeta_den.txt" );

		{ // den6
			getline( stream, line );
			{
				std::istringstream l(line);
				l >>	tag;
				for ( Size i=1; i<= cbeta_den_table_size; ++i ){
					l >>mem_cbeta_den6_(i);
				}
			if ( l.fail() || tag != "MEMCBETA_DEN6:"  ) utility_exit_with_message("bad format for cbeta_den.txt (DEN6)");
			}
			getline( stream, line );
			{
				std::istringstream l(line);
				l >> tag;
				for ( Size i=1; i<= cbeta_den_table_size; ++i ){
					l >> mem_cbeta_2TM_den6_(i);
				}
				if ( l.fail() || tag != "MEMCBETA_2TM_DEN6:"  ) utility_exit_with_message("bad format for cbeta_den.txt (2TM_DEN6)");
			}
			getline( stream, line );
			{
				std::istringstream l(line);
				l >> tag;
				for ( Size i=1; i<= cbeta_den_table_size; ++i ){
					l >> mem_cbeta_4TM_den6_(i);
				}
				if ( l.fail() || tag != "MEMCBETA_4TM_DEN6:"  ) utility_exit_with_message("bad format for cbeta_den.txt (4TM_DEN6)");
			}
		}

		{
			{ // den12
				getline( stream, line );
				std::istringstream l(line);
				l >> tag;
				for ( Size i=1; i<= cbeta_den_table_size; ++i ){
					l >> mem_cbeta_den12_(i);
				}
				if ( l.fail() || tag != "MEMCBETA_DEN12:"  ) utility_exit_with_message("bad format for cbeta_den.txt (DEN12)");
			}
			getline( stream, line );
		    {
				std::istringstream l(line);
				l >> tag;
				for ( Size i=1; i<= cbeta_den_table_size; ++i ){
					l >> mem_cbeta_2TM_den12_(i);
				}
				if ( l.fail() || tag != "MEMCBETA_2TM_DEN12:"  ) utility_exit_with_message("bad format for cbeta_den.txt (2TM_DEN12)");
			}
			getline( stream, line );
			{
				std::istringstream l(line);
				l >> tag;
				for ( Size i=1; i<= cbeta_den_table_size; ++i ){
					l >> mem_cbeta_4TM_den12_(i);
				}
				if ( l.fail() || tag != "MEMCBETA_4TM_DEN12:"  ) utility_exit_with_message("bad format for cbeta_den.txt (4TM_DEN12)");
			}

		}
	}


	{ // pair_log
		mem_pair_log_.dimension( min_mem_layers,pair_log_table_size,max_aa, max_aa );

		utility::io::izstream stream;
		io::database::open( stream, "mem_pair_log.txt" );
		for ( Size i=1; i<= min_mem_layers;++i) {
			for ( Size j=1; j<= pair_log_table_size; ++j ) {
				for ( Size k=1; k<= max_aa; ++k ) {
					getline( stream, line );
					std::istringstream l(line);
					Size ii,jj;
					l >> tag >> ii >> jj >> aa;
					assert( Size(aa) == k );
					for ( Size n=1;n<=max_aa;++n)
						{
							l >> mem_pair_log_(i,j,aa,n);
						}
					if ( l.fail() || ii != i || jj != j || tag != "MEM_PAIR_LOG:"  ) utility_exit_with_message("bad format for mem_pair_log.txt");
				}

			}
		}
	}


}

void
MembranePotential::finalize( pose::Pose & pose ) const
{
	CenListInfo & cenlist( nonconst_cenlist_from_pose( pose ));
	cenlist.calculated() = false;
	MembraneEmbed & membrane_embed( nonconst_MembraneEmbed_from_pose( pose ));
	membrane_embed.calculated() = false;
}

////////////////////////////////////////////////////////////////////////////////////
void
MembranePotential::evaluate_env( //_and_cbeta_scores(
	pose::Pose const & pose,
	conformation::Residue const & rsd,
	Real & membrane_env_score
) const
{
	//core::util::MetricValue< utility::vector1< core::Real > > depth;
	if(MembraneEmbed_from_pose( pose ).spanning()) {
		Real const MembraneDepth (MembraneEmbed_from_pose( pose ).depth(rsd.seqpos() ) );
		evaluate_env(pose,rsd,MembraneDepth,membrane_env_score);
	} else
	{
		membrane_env_score=100;
	}
}



////////////////////////////////////////////////////////////////////////////////////
void
MembranePotential::evaluate_env( //_and_cbeta_scores(
	pose::Pose const & pose,
	conformation::Residue const & rsd,
	Real const MembraneDepth,
	Real & membrane_env_score
) const
{
	//using ObjexxFCL::fmt::F; // debugging
	//using ObjexxFCL::fmt::I;

	Real const env6_weight=1.0;
	Real const env10_weight=1.0;

	Real t1 = 4.0; //thickness of interpolation phase
  Real t2 = 2.0;
  Real t3 = 2.0;
  int s1 = 10; //steepness of interpolation function
  int s2 = 14;
  int s3 = 14;

	int layer1,layer2,layer,B_layer; // B_layer will be removed just keep it for now...
	Real f,z,zn,low;

	//Real hbond_penalty=20.0;
	//CenListInfo const & cenlist( cenlist_from_pose( pose ));

	//int const position ( rsd.seqpos() );

	Real const fcen6  ( cenlist_from_pose( pose ).fcen6( rsd.seqpos() ) );
	Real const fcen10 ( cenlist_from_pose( pose ).fcen10( rsd.seqpos() ) );
	//Real const fcen12 ( cenlist.fcen12(position) );
	Size bur6,bur10;
	find_buried_class(fcen6,fcen10,bur6,bur10);

	//Real const MembraneDepth (MembraneEmbed_from_pose( pose ).depth(rsd.seqpos() ) ); // TODO Will be set by the Pose metric, which was too slow...
	if ( rsd.is_protein() ) {

		if ( ( MembraneDepth < -2.0 ) || ( MembraneDepth > 62.0 ) ) {
    //pure water layer
		layer = 1;
		B_layer = 1;
    membrane_env_score = env6_weight*mem_env_log6_( rsd.aa(), layer, bur6 )
     + env10_weight*mem_env_log10_( rsd.aa(), layer, bur10 );

  } else if ( ( MembraneDepth >= -2.0 && MembraneDepth <= 2.0 ) || ( MembraneDepth >= 58.0 &&  MembraneDepth <= 62.0 ) ) {
    //interpolate between water and polar phases
    layer1 = 1;
    layer2 = 2;
    B_layer = 1;
    if ( MembraneDepth <= 2.0 ) {
      low = 2.0;
    } else {
      low = 58.0;
    }
    z = 2*std::abs( (MembraneDepth - low) ) / t1;
    zn = std::pow( z, s1 );
    f = zn/(1 + zn);

    membrane_env_score = f * ( env6_weight*mem_env_log6_( rsd.aa(), layer1, bur6 )
     + env10_weight*mem_env_log10_( rsd.aa(), layer1, bur10 ) ) +
    ( 1 - f ) * ( env6_weight*mem_env_log6_( rsd.aa(), layer2, bur6 )
     + env10_weight*mem_env_log10_( rsd.aa(), layer2, bur10 ) );

    if ( MembraneDepth <= 0.0 || MembraneDepth >= 60.0 ) {
      layer = 1;
    } else {
      layer = 2;
    }

	} else if ( ( MembraneDepth > 2.0 && MembraneDepth < 11.0 ) || ( MembraneDepth > 49.0 && MembraneDepth < 58.0 ) ) {
    //pure polar layer
		layer = 2;
		B_layer = 1;
    membrane_env_score = env6_weight*mem_env_log6_( rsd.aa(), layer, bur6 )
     + env10_weight*mem_env_log10_( rsd.aa(), layer, bur10 );

  } else if ( ( MembraneDepth >= 11.0 && MembraneDepth <= 13.0 ) || ( MembraneDepth >= 47.0 && MembraneDepth <= 49.0 ) ) {
    //interpolate between polar and interface phases
    layer1 = 2;
    layer2 = 3;
    B_layer = 1;

   if ( MembraneDepth <= 13.0 ) {
      low = 13.0;
    } else {
      low = 47.0;
    }
    z = 2*std::abs( (MembraneDepth - low) ) / t2;
    zn = std::pow( z, s2 );
    f = zn/(1 + zn);

    membrane_env_score = f * ( env6_weight*mem_env_log6_( rsd.aa(), layer1, bur6 )
     + env10_weight*mem_env_log10_( rsd.aa(), layer1, bur10 ) ) +
    ( 1 - f ) * ( env6_weight*mem_env_log6_( rsd.aa(), layer2, bur6 )
     + env10_weight*mem_env_log10_( rsd.aa(), layer2, bur10 ) );

    if ( MembraneDepth <= 12.0 || MembraneDepth >= 48.0 ) {
      layer = 2;
    } else {
      layer = 3;
    }

  } else if ( ( MembraneDepth > 13.0 && MembraneDepth < 17.0 ) || ( MembraneDepth > 43.0 && MembraneDepth < 47.0 ) ) {
    //pure interface phase
		layer = 3;
		B_layer = 1;
    membrane_env_score = env6_weight*mem_env_log6_( rsd.aa(), layer, bur6 )
     + env10_weight*mem_env_log10_( rsd.aa(), layer, bur10 );

  } else if ( ( MembraneDepth >= 17.0 && MembraneDepth <= 19.0 ) || ( MembraneDepth >= 41.0 && MembraneDepth <= 43.0 ) ) {
    //interpolate between interface and hydrophobic phases
    layer1 = 3;
    layer2 = 4;

    if ( MembraneDepth <= 19.0 ) {
      low = 19.0;
    } else {
      low = 41.0;
    }
    z = 2*std::abs( (MembraneDepth - low) ) / t3;
    zn = std::pow( z, s3 );
    f = zn/(1 + zn);

    membrane_env_score = f * ( env6_weight*mem_env_log6_( rsd.aa(), layer1, bur6 )
     + env10_weight*mem_env_log10_( rsd.aa(), layer1, bur10 ) ) +
    ( 1 - f ) * ( env6_weight*mem_env_log6_( rsd.aa(), layer2, bur6 )
     + env10_weight*mem_env_log10_( rsd.aa(), layer2, bur10 ) );

    if ( MembraneDepth <= 18.0 || MembraneDepth >= 42.0 ) {
      layer = 3;
      B_layer = 1;
    } else {
      layer = 4;
      B_layer = 2;
    }

    } else {
    //pure hydrophobic phase
		layer = 4;
		B_layer = 2;
		membrane_env_score = env6_weight*mem_env_log6_( rsd.aa(), layer, bur6 ) + env10_weight*mem_env_log10_( rsd.aa(), layer, bur10 );
		//std::cout << rsd.seqpos() << " " << pose.secstruct(rsd.seqpos()) << "\n";
		//if(pose.secstruct(rsd.seqpos()) != 'H')
		//{
		//	membrane_env_score+=hbond_penalty;
		//}
	}
		membrane_env_score*=0.5; //bw membrane_embed_weight...

		//env_score = env_log_( rsd.aa(), static_cast< int >( fcen10 ) );

//		// interp1 rounds down to nearest (non-negative) integer.
//		int interp1 = static_cast< int >( fcen6 );
//		// note cen6 is always at least 1.0
//
//		// fraction remainder after nearest lower integer is removed
//		Real interp2 = fcen6 - interp1;
//
//		//  use interp2 to linearly interpolate the two nearest bin values
//		cb_score6 =
//			( ( 1.0 - interp2 ) * cbeta_den6_( interp1 ) +
//			(         interp2 ) * cbeta_den6_( interp1+1 ) );
//
//		interp1 = static_cast< int >( fcen12 );
//		// note cen12 is always at least 1.0 -- this is in fact false for fcen12
//		interp2 = fcen12 - interp1;
//		cb_score12 =
//			( ( 1.0 - interp2 ) * cbeta_den12_( interp1   ) +
//			(         interp2 ) * cbeta_den12_( interp1+1 ) );
//
//		//std::cout << "eval_env_cbeta: " << I(4,rsd.seqpos()) << F(9,3,fcen6) << F(9,3,fcen10) << F(9,3,fcen12) <<
//		//	F(9,3,env_score) << F(9,3,cb_score6) << F(9,3,cb_score12) << ' ' << rsd.name() << std::endl;
//		//std::cout << "fcen6( " << position << " ) = " << fcen6 << " fcen10( " <<  position << " ) " << fcen10 << " fcen12( " << position << " ) = ";
//		//std::cout << fcen12 << " "; //<< std::endl;
//		// " interp1: " << interp1 << " interp2: " << interp2 << std::endl;
//		Size old_bur10(0);
//		Size ffcen10=fcen10+0.5;
//		if ( ffcen10 <= 12 ) {
//			old_bur10 =  12;
//		} else if ( ( ffcen10 > 12 ) && ( ffcen10 <= 15 ) ) {
//			old_bur10 = 15;
//		} else if ( ( ffcen10 > 15 ) && ( ffcen10 <= 18 ) ) {
//			old_bur10 = 18;
//		} else if ( ( ffcen10 > 18 ) && ( ffcen10 <= 21 ) ) {
//			old_bur10 = 21;
//		} else if ( ( ffcen10 > 21 ) && ( ffcen10 <= 24 ) ) {
//			old_bur10 = 24;
//		} else {
//			old_bur10 = 27;
//		}

//		Size old_bur6(0);

//		if ( fcen6 <= 2 ) {
//			old_bur6 = 2;
//		} else if ( fcen6 >= 7 ) {
//			old_bur6 = 7;
//		} else {
//			old_bur6 = fcen6;
//		}

//		std::cout << MembraneDepth << " " << membrane_env_score << " " << bur6+1 << " " << old_bur10 << " " << fcen6 << " " << fcen10 << "\n";
	} else { // amino acid check
		membrane_env_score = 0.0;
		//		cb_score6  = 0.0;
		//cb_score12 = 0.0;
	}
}


///////////////////////////////////////////////////////////////////////////////////////////////
void
MembranePotential::evaluate_cbeta(
		pose::Pose const & pose,
		conformation::Residue const & rsd,
		Real & membrane_cb_score
	) const
{
	//CenListInfo const & cenlist( cenlist_from_pose( pose ));
	//int const position ( rsd.seqpos() );
	//Real const fcen6  ( cenlist.fcen6( position) );
	////Real const fcen10 ( cenlist.fcen10(position) );
	//Real const fcen12 ( cenlist.fcen12(position) );

	membrane_cb_score=0;
	//if(!MembraneTopology_from_pose( pose ).allow_scoring(rsd.seqpos())
	//   return;
	Real const fcen6 ( cenlist_from_pose( pose ).fcen6( rsd.seqpos() ) );
	Real const fcen12 ( cenlist_from_pose( pose ).fcen12( rsd.seqpos() ) );


	Real membrane_cb_score6,membrane_cb_score12;

	Size const TMHs (MembraneTopology_from_pose( pose ).tmh_inserted() );
	//if ( fcen10 >= 31 ) fcen10 = 30.9999;
	//if(TMHs==1)
//		return;

	// interp1 rounds down to nearest (non-negative) integer.
	int const interp1 = static_cast< int >( fcen6 );
	int const interp3 = static_cast< int >( fcen12 );
	// note cen6 is always at least 1.0

	// fraction remainder after nearest lower integer is removed
	Real const interp2 = fcen6-interp1;
	Real const interp4 = fcen12-interp3;

	//std::cout << TMHs << " " << rsd.seqpos() << " " << interp1 << " " << interp2 << " " << interp3 << " " << interp4 << " " << mem_cbeta_2TM_den6_( interp1 ) << " " << mem_cbeta_4TM_den6_( interp1 ) << " " << mem_cbeta_den6_( interp1 )<< " " << mem_cbeta_den12_( interp3 ) << "\n";
	//vyy  NEW- Cbeta score evaluation depends on number of transmembrane helices inserted in the membrane:
	if ( TMHs <= 2 ) {
		membrane_cb_score6 =
			( 1.0-interp2 ) * mem_cbeta_2TM_den6_( interp1 )+
			interp2         * mem_cbeta_2TM_den6_( interp1+1 );

	} else if ( TMHs <= 4 ) {
		membrane_cb_score6 =
			(1.0-interp2) * mem_cbeta_4TM_den6_( interp1 )+
			interp2       * mem_cbeta_4TM_den6_( interp1+1 );

	} else {
		membrane_cb_score6 =
			(1.0-interp2) * mem_cbeta_den6_( interp1 )+
			interp2       * mem_cbeta_den6_( interp1+1 );
						//pba compute density with 12A probe only for TMH bundles or more than 4 helices
						//membrane_cb_score12 += (1.0-interp4)*membrane_cbeta_den12( interp3 )+
						//  interp4*membrane_cbeta_den12( interp3+1 );
	}
	// bw was better to always include this term than only for TMHs gt 4


	membrane_cb_score12 =
		(1.0-interp4) * mem_cbeta_den12_( interp3 )+
		interp4       * mem_cbeta_den12_( interp3+1 );

	//std::cout << "pos fcen6 fcen12 " << TMHs << " " << rsd.seqpos() << " " << fcen6 <<  " " << fcen12 << " : " << membrane_cb_score6 << " " << membrane_cb_score12 << "\n";
	//membrane_cb_score = 2.667*(membrane_cb_score6+membrane_cb_score12) *0.3;
	membrane_cb_score = (membrane_cb_score6+membrane_cb_score12);
}

///////////////////////////////////////////////////////////////////////////////////////////////
void
MembranePotential::evaluate_pair(
								 pose::Pose const & pose,
								 conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	Real const cendist,
		Real & membrane_pair_score
	) const
{

	membrane_pair_score = 0.0;

	if ( !rsd1.is_protein() || !rsd2.is_protein() ) return;

	chemical::AA const aa1( rsd1.aa() );
	chemical::AA const aa2( rsd2.aa() );

	//CAR  no pair score if a disulfide
	if ( aa1 == chemical::aa_cys && aa2 == chemical::aa_cys &&
			 rsd1.is_bonded( rsd2 ) && rsd1.polymeric_sequence_distance( rsd2 ) > 1 &&
			 rsd1.has_variant_type( chemical::DISULFIDE ) && rsd2.has_variant_type( chemical::DISULFIDE ) ) return;

	// no pair score for residues closer than 9 in sequence
	if ( rsd1.polymeric_sequence_distance( rsd2 ) /* j - i */ <= 8 ) return;

	//$$$  we now try to find which bin the pair distance lies in
	//$$$  I note this could in principle be calculated and updatded
	//$$$  just like cen_dist is if there is a need for speed.
	//$$$  this function interpolates between bins.
	//$$$  An important(!) requirement on pair_log is that the
	//$$$  value should approach zero as the radius increases.
	//$$$  this fact permits us not to have to compute and score pairs are larger
	//$$$  than cen_dist > cutoff.

	int icon = 5;
	Real interp2( 0.0 );
	//Real interp1( 0.0);
	Real const MembraneDepth1 (MembraneEmbed_from_pose( pose ).depth(rsd1.seqpos() ) );
	Real const MembraneDepth2 (MembraneEmbed_from_pose( pose ).depth(rsd2.seqpos() ) );


	int hydro_layer=1;  //1 not_hydrophobic_core 2 hydrophobic core
	Real AverageDepth=(MembraneDepth1+MembraneDepth2)/2;
	 if(MembraneDepth1 > 18 &&
	   MembraneDepth1 < 42 &&
	   MembraneDepth2 >18 &&
	   MembraneDepth2 <42) //bw currently both residues have to be in the hydrophobic core
		{
			hydro_layer=2;
		}


	if ( cendist > cen_dist10_pad_plus ) {
		icon = 4;
		interp2 = ( cendist + cen_dist12_pad_minus ) * cen_dist12_pad_hinv;
	} else {
		if ( cendist > cen_dist7_pad_plus ) {
			icon = 3;
			interp2 = ( cendist + cen_dist10_pad_minus ) * cen_dist10_pad_hinv;
			//std::cout << "interp2 (" << cendist << " + " << cen_dist10_pad_minus << " ) * " << cen_dist10_pad_hinv << " " << interp2 << "\n";
		} else {
			if ( cendist > cen_dist5_pad_plus ) {
				icon = 2;
				interp2 = ( cendist + cen_dist7_pad_minus ) * cen_dist7_pad_hinv;
			} else {
				icon = 1;
				interp2 = ( cendist + cen_dist5_pad_minus ) * cen_dist5_pad_hinv;
			}
		}
	}
	if ( interp2 < 0.0 ) interp2 = 0.0;

	// note in theory this will never happen but in practice round off
	// error can cause problem
	if ( interp2 > 1.0 ) interp2 = 1.0;

	// handle last bin specially since icon+1 would be past array end
	// pb note -- I don't think icon will ever be 5 here, wonder if it has always been this way?
  // bw true
	//std::cout << "PAIR " << rsd1.seqpos() << " " << rsd2.seqpos() << " " << MembraneDepth1 << " " <<MembraneDepth2 << " " << hydro_layer << " " <<  icon <<  " " << interp2 << " " << aa1 << " " << aa2 << " " << mem_pair_log_( hydro_layer, icon  , aa1, aa2 ) << "\n";
	Real f(0);
	if ( icon != 5 ) {

		if(!no_interpolate_Mpair_) { //bw new mini specfic, true by default.
			//std::cout << "interpolate Mpair\n";
			if( std::abs(AverageDepth - 18)<4)
			{
				f=1/(1+std::exp(1.5*(18-AverageDepth)));
				membrane_pair_score = ( ( 1.0f - interp2 ) * ((1-f)*mem_pair_log_( 1, icon  , aa1, aa2 ) + f*mem_pair_log_( 2, icon  , aa1, aa2 )) +
									   (      	interp2 ) *  ((1-f)*mem_pair_log_( 1, icon+1, aa1, aa2 ) + f*mem_pair_log_( 2, icon+1, aa1, aa2 )));
			} else if(std::abs(AverageDepth - 42)<4) {

				f=1/(1+std::exp(1.5*(AverageDepth-42)));

				membrane_pair_score = ( ( 1.0f - interp2 ) * ((1-f)*mem_pair_log_( 1, icon  , aa1, aa2 ) + f*mem_pair_log_( 2, icon  , aa1, aa2 )) +
								   (      	interp2 ) *  ((1-f)*mem_pair_log_( 1, icon+1, aa1, aa2 ) + f*mem_pair_log_( 2, icon+1, aa1, aa2 )));
			//std::cout << AverageDepth << " " << f << " " << membrane_pair_score_tmp << " " << membrane_pair_score << "\n";

			}  else {

				membrane_pair_score = ( ( 1.0f - interp2 ) * mem_pair_log_( hydro_layer, icon  , aa1, aa2 ) +
									    (      	interp2 ) *  mem_pair_log_( hydro_layer, icon+1, aa1, aa2 ));
			}
		} else {
			//std::cout << "No interpolate Mpair\n";
			membrane_pair_score = ( ( 1.0f - interp2 ) * mem_pair_log_( hydro_layer, icon  , aa1, aa2 ) +
									(      	interp2 ) *  mem_pair_log_( hydro_layer, icon+1, aa1, aa2 ));
		}




	} else {

		membrane_pair_score =   ( 1.0f - interp2 ) * mem_pair_log_( hydro_layer,icon  , aa1, aa2 );
	}
	membrane_pair_score*=2.019;
	//std::cout << "PAIR " << rsd1.seqpos() << " " << rsd2.seqpos() << " " << AverageDepth << " " << hydro_layer << " " <<  icon <<  " " << aa1 << " " << aa2 << " " << interp2 << " " << f << " " << mem_pair_log_( hydro_layer,icon  , aa1, aa2 ) << " " << membrane_pair_score << "\n";

	/*	// Adding a term that should help reproduce pairwise correlation function between centroids
	//      as observed in the PDB.
	int cendist_bin = static_cast <int> ( sqrt( cendist ) * 10 + 1); //Binned with 0.1 A width.

	if (cendist_bin > 120)   cendist_bin = 120;
	if (cendist_bin <   1)   cendist_bin = 1;

	cenpack_contribution = cenpack_log_( cendist_bin );
	*/
	return;
}


void
MembranePotential::compute_membrane_embedding(pose::Pose & pose)
const
{
	using namespace core::options;
	using namespace core::options::OptionKeys;
	MembraneEmbed & membrane_embed(nonconst_MembraneEmbed_from_pose( pose ));
	if(!membrane_embed.calculated())
	   {

		   membrane_embed.initialize( pose );
		   Vector init_normal,init_center;
		   init_membrane_center_normal(pose,init_normal,init_center);

		   //	core::scoring::MembraneTopology & topology( nonconst_MembraneTopology_from_pose(pose).depth() );
		   //utility::vector1< core::Real > & depth(nonconst_MembraneTopology_from_pose(this_pose).depth() );
		   //bw project each CA to the membrane normal.
//		   Real total_score(0),
		   Real score(0),best_score(999999),accepted_score(99999);
		   //Size tm_proj(0),best_tm_proj(0);
		   //Vector best_normal,best_center,accepted_normal,accepted_center;
		   Real temperature=2.0;
		   Vector trial_normal(init_normal);
		   Vector trial_center(init_center);
		   Vector best_normal(init_normal);
		   Vector best_center(init_center);
		   Vector accepted_center(init_center);
		   Vector accepted_normal(init_normal);
		   score_normal_center(pose,trial_normal,trial_center,best_score);
		   accepted_score=best_score;

		   bool spanning(check_spanning(pose,trial_normal,trial_center));
		   bool best_spanning(spanning);
///numeric::random::RandomGenerator RG_mem(2771);
		   bool debug=option[ membrane::debug ]();

		   Real normal_mag=membrane_normal_magnitude_; //bw tmp
		   Real center_mag=membrane_center_magnitude_; //bw tmp
		   //Real normal_mag_fine=numeric::conversions::radians(1.0);
		   //Real center_mag_fine=1;
		   //Real normal_mag_coarse=numeric::conversions::radians(5.0);
		   //Real center_mag_coarse=5;
		   Size counter=0;
		   Size accepted=0;
		   Size thermally_accepted=0;

//		   Size total_cycles= option[ membrane::normal_cycles ](); //1000;
//		   Size finer_moves=(Size)(smooth_move_frac_*membrane_normal_cycles_);
		   for( Size cycles=1;cycles<=membrane_normal_cycles_;++cycles)
		   {

			 /*  if(cycles==finer_moves) {

				   normal_mag*=0.25;
				   center_mag*=0.25;
			   }
				*/
			   if(RG.uniform()<0.5) // change center
			   {
				   /*
				   if(debug) {
					   std::cout << trial_center.x() << " " << trial_center.y() << " " << trial_center.z() << std::endl;
					   std::cout << "center mag " << center_mag << std::endl;
					}
					*/
				   rigid_perturb_vector(trial_center,center_mag);
				   /*
				   if(debug)
					   std::cout << trial_center.x() << " " << trial_center.y() << " " << trial_center.z() << std::endl;
				   */
			   } else { // change normal

				   /*if(debug)	{
				   std::cout << trial_normal.x() << " " << trial_normal.y() << " " << trial_normal.z() << std::endl;
					   std::cout << "normal mag " << numeric::conversions::degrees(normal_mag) << std::endl;
				   }
					*/
					rot_perturb_vector(trial_normal,normal_mag);
				   /*
				   if(debug)
					std::cout << trial_normal.x() << " " << trial_normal.y() << " " << trial_normal.z() <<std::endl;
				   */
			   }
			   if(!check_spanning(pose,trial_normal,trial_center)){
				   if(debug)
					   std::cout << "Cycle " << cycles << " not spanning\n" ;
					trial_center=accepted_center;
					trial_normal=accepted_normal;
					continue;
				  }

			   score_normal_center(pose,trial_normal,trial_center,score); //,tm_ratio_projection);
			  // if(debug)
			//	   std::cout << "CYCLES SCORE " << cycles << " " << score << " " << accepted_score << " " << best_score << "\n";
			   if(score<accepted_score)
			   {
				   if(score<best_score)
				   {
						best_score=score;
					   best_center=trial_center;
					   best_normal=trial_normal;
					   //best_tm_proj=tm_proj;
					   best_spanning=true; //bw if you are here it is spanning....
					   //std::cout << "Best Sofar!\n";
				   }
				   accepted_score=score;
				   accepted_center=trial_center;
				   accepted_normal=trial_normal;
				   ++accepted;
				  // std::cout << "Accepted: Energy Global Best\n";
			   }
			   else
			   {
				   ++counter;
				   Real const boltz_factor=(accepted_score-score)/temperature;
				   Real const probability = std::exp( std::min ((core::Real)40.0, std::max((core::Real)-40.0,boltz_factor)) );
				   if(RG.uniform()<probability)
				   {
					   accepted_score=score;
					   accepted_center=trial_center;
					   accepted_normal=trial_normal;
					   ++thermally_accepted;
					   ++accepted;
					  // std::cout << "Accepted: Thermally accepted " << boltz_factor << " " << probability << "\n";
				   }
				   else
				   {
					   trial_center=accepted_center;
					   trial_normal=accepted_normal;
					 //  std::cout << "Rejected: Thermally rejected " << boltz_factor << " " << probability << "\n";
				   }
			   }
		   }


		   //save best projection...
		   Size nres=pose.total_residue();
		   for ( Size i = 1; i <= nres; ++i ) {
			   Vector const & xyz( pose.residue( i ).atom( 2 ).xyz());
			   membrane_embed.depth(i)=dot(xyz-best_center,best_normal)+30; //bw check that the values used are shifted 30.
//			   std::cout << "DEPTH " << i << " " << membrane_embed.depth(i) << "\n";
			}
		   membrane_embed.set_normal(best_normal);
		   membrane_embed.set_center(best_center);
		   membrane_embed.spanning()=best_spanning;
		   //membrane_embed.tm_projection()=best_tm_proj;
		   membrane_embed.calculated()=true;

		 /*/  std::cout << "INIT CENTER " << init_center.x() <<  " " << init_center.y() << " " << init_center.z() << "\n";
		   std::cout << "INIT NORMAL " << init_normal.x() <<  " " << init_normal.y() << " " << init_normal.z() << "\n";


		   std::cout << "INIT CENTER " << best_center.x() <<  " " << best_center.y() << " " << best_center.z() << "\n";
		   std::cout << "INIT NORMAL " << best_normal.x() <<  " " << best_normal.y() << " " << best_normal.z() << "\n";

		   Real acceptance_rate=(Real)accepted/membrane_normal_cycles_;
		   Real thermally_acceptance_rate=(Real)thermally_accepted/counter;
		   */
		   //std::cout << "TOTAL ACCEPT " << accepted << " " << membrane_normal_cycles_ << " " << acceptance_rate << "\n";
		   //std::cout << "THERMAL ACCEPT " << thermally_accepted << " " << counter << " " << thermally_acceptance_rate << "\n";
//		   std::cout << "SCORE: " << best_score << "\n";
	   }

}


void
MembranePotential::init_membrane_center_normal(pose::Pose const & pose,
													  Vector & normal,
													  Vector & center) const
	{
		//using namespace core::scoring;
		//		normal.assign(0,0,1);
		//center.assign(0,0,0);
	//	bool debug=core::options::option[ core::options::OptionKeys::membrane::debug ]();

		MembraneTopology const & topology( MembraneTopology_from_pose(pose) );

		//Define vectors for inside and outside cap residue

		Vector inside(0);
		Vector outside(0);
		//Size tmh_inserted
		for(Size i=1;i<=topology.tmhelix();++i)
		{
			if(!topology.allow_tmh_scoring(i)) continue;
			/*if(debug)
			{
				std::cout << "ALLOW TMH SCORING " << i << "\n";
			}
			*/
			Vector const & start( pose.residue( topology.span_begin(i) ).atom( 2 ).xyz());
			Vector const & end( pose.residue( topology.span_end(i) ).atom( 2 ).xyz());
			//std::cout << i << " " << topology.span_begin(i) << " start: " << start.x() << " " << start.y() << " " << start.z() << "\n";
			//std::cout << i << " " << topology.span_end(i) << " end  : " << end.x() << " " << end.y() << " " << end.z() << "\n";
			// all odd helices goes from outside in (from c++)
			if( topology.helix_id(i) % 2 == 0)
			{
			//	std::cout << i << " " << topology.helix_id(i) << " even\n";
				inside+=start;
				outside+=end;
			}
			else
			{
			//	std::cout << i << " " << topology.helix_id(i) << " odd\n";
				outside+=start;
				inside+=end;
			}
		}
		normal=outside-inside;
		normal.normalize();
		center=0.5*(outside+inside)/topology.tmh_inserted();
		//std::cout << "INIT CENTER " << center.x() <<  " " << center.y() << " " << center.z() << "\n";
		//std::cout << "INIT NORMAL " << normal.x() <<  " " << normal.y() << " " << normal.z() << "\n";

	}


void
MembranePotential::score_normal_center(pose::Pose const & pose,
										   Vector const & normal,
										   Vector const & center,
									       Real & score) const
{
	Size const nres=pose.total_residue();
	MembraneTopology const & topology( MembraneTopology_from_pose(pose) );
	score=0;
	Real residue_score(0);
	Real tm_projection(0);
	Real hbond_pen(0);
	Real termini_pen(0);
	for ( Size i = 1; i <= nres; ++i ) {
		//CA coords
		if(!topology.allow_scoring(i)) continue;
		Vector const & xyz( pose.residue( i ).atom( 2 ).xyz());
		core::Real depth=dot(xyz-center,normal)+30;
		evaluate_env(pose,pose.residue(i),depth,residue_score);
		score+=residue_score;
		//		std::cout << "projection: " << i << " " << prod << "\n";
		//membrane_embed.depth(i)=dot(xyz-trial_center,trial_normal)+30; //bw check that the values used are shifted 30 was -30.

		//std::cout << "projection: " << i << " " << xyz.x() << " " << xyz.y() << " " << xyz.z() << " " << depth << " " <<  residue_score << "\n";
	}
	if(Menv_penalties_) {
		tm_projection_penalty(pose,normal,center,tm_projection);
		hbond_penalty(pose,normal,center,hbond_pen);
		termini_penalty(pose,normal,center,termini_pen);
		score+=tm_projection+hbond_pen+termini_pen; // bw skipping term_penalty+50.0*term_penalty; //bw 0.5*c++ version.
	}
}

void
MembranePotential::rot_perturb_vector(Vector & v,
								       Real const & std_dev) const
{
	Vector u(numeric::random::gaussian(),numeric::random::gaussian(),numeric::random::gaussian()); //bw rotation_matrix will normalize.
	Real alpha(numeric::random::gaussian()*std_dev);
	//std::cout << "alpha " << alpha << std::endl;
	v=rotation_matrix(u,alpha)*v;
}

void
MembranePotential::rigid_perturb_vector(Vector & v,
										Real const & std_dev) const
{
	Vector u(numeric::random::gaussian(),numeric::random::gaussian(),numeric::random::gaussian());
	u.normalize();
	v=v+std_dev*u;
}

bool
MembranePotential::check_spanning(pose::Pose const & pose, Vector const & normal,Vector const & center) const
{
	MembraneTopology const & topology( MembraneTopology_from_pose(pose) );

	// check that helices have ends on either side of the center.
	// and that directory is changing.

	//std::cout << "=====\n";
	//std::cout << "center " << center.x() << " " << center.y() << " " << center.z() << "\n";
	//std::cout << "normal " << normal.x() << " " << normal.y() << " " << normal.z() << "\n";
	for(Size i=1;i<=topology.tmhelix()-1;++i)
	{
		if(!topology.allow_tmh_scoring(i)) continue;
		Vector const & start_i( pose.residue( topology.span_begin(i) ).atom( 2 ).xyz());
	//	std::cout << "start_i " << i << " " << topology.span_begin(i) << " " << start_i.x() << " " << start_i.y() << " " << start_i.z() << "\n";

		bool start_i_side=(dot(start_i-center,normal) > 0);
	//	Real depth_i=dot(start_i-center,normal);
		bool span_check=false;
		for(Size j=i+1;j<=topology.tmhelix();++j)
		{
			if(!topology.allow_tmh_scoring(j) || span_check) continue;
			span_check=true;
			Vector const & start_j( pose.residue( topology.span_begin(j) ).atom( 2 ).xyz());
	//		std::cout << "start_j " << j << " " << topology.span_begin(j) << " " << start_j.x() << " " << start_j.y() << " " << start_j.z() << "\n";
			bool start_j_side=(dot(start_j-center,normal) > 0);
	//		Real depth_j=dot(start_j-center,normal);
			bool coord_para=(start_i_side==start_j_side);
			//bool tmh_para=(topology.helix_id(i)-topology.helix_id(j) % 2 == 0);
	//		std::cout << i << " " << j << " " << start_i_side << " " << depth_i << " " << start_j_side << " " << depth_j << "\n";
//			if(topology.relative_tmh_ori(i,j)
			if(topology.helix_id(i)-topology.helix_id(j) % 2 == 0) // both should be on the same side (parallel)
			{
				if(!(coord_para))
				{
					//not spanning;
					//std::cout << "Parallel helices " << i << " and " << j << " ARE Antiparallel\n";
					return false;
				}

			} else { // should be on opposite sides.
				if(coord_para)
				{
					//std::cout << "AntiParallel helices " << i << " and " << j << " ARE parallel\n"; //on the same side\n";
					return false;
				}

			}
		}
		//start_i_side=start_j_side;
	}
//	std::cout << "===== spanning\n";
	return true;
}


void
MembranePotential::tm_projection_penalty(pose::Pose const & pose,Real & tm_proj) const
{
	tm_proj=0.0;
	if(!Menv_penalties_)
		return;
	Vector const normal(MembraneEmbed_from_pose( pose ).normal());
	Vector const center(MembraneEmbed_from_pose( pose ).center());

	tm_projection_penalty(pose,normal,center,tm_proj);
}
void
MembranePotential::tm_projection_penalty(pose::Pose const & pose, Vector const & normal,Vector const & center,Real & tm_proj) const
{
	tm_proj=0.0;
	if(!Menv_penalties_)
		return;
	MembraneTopology const & topology( MembraneTopology_from_pose(pose) );

	//Define vectors for inside and outside cap residue

	Vector inside(0);
	Vector outside(0);
	tm_proj=0;

	for(Size i=1;i<=topology.tmhelix();++i)
	{
		if(!topology.allow_tmh_scoring(i)) continue;
		Vector const & start( pose.residue( topology.span_begin(i) ).atom( 2 ).xyz());
		Vector const & end( pose.residue( topology.span_end(i) ).atom( 2 ).xyz());
		Real tm_length=std::abs(dot(start-center,normal)-dot(end-center,normal));
		Real ratio=tm_length/(topology.span_end(i)-topology.span_begin(i)+1);
		if(tm_length<15)
			tm_proj++;
		if(ratio<1 || ratio > 1.5)
			tm_proj++;

	}
	tm_proj*=50; //total_embed weight is 0.5 in membrane_score_quick.cc
}

void
MembranePotential::hbond_penalty(pose::Pose const & pose, Real & hbond_pen) const
{
	hbond_pen=0.0;
	if(!Menv_penalties_)
		return;
	Vector const normal(MembraneEmbed_from_pose( pose ).normal());
	Vector const center(MembraneEmbed_from_pose( pose ).center());
	hbond_penalty(pose,normal,center,hbond_pen);
}

void
MembranePotential::hbond_penalty(pose::Pose const & pose, Vector const & normal,Vector const & center,Real & hbond_pen) const
{
	hbond_pen=0.0;
	if(!Menv_penalties_)
		return;
	MembraneTopology const & topology( MembraneTopology_from_pose(pose) );
	for(Size i=1;i<=pose.total_residue();++i)
	{
		if(!topology.allow_scoring(i)) continue;
		if(topology.tmregion(i) && pose.conformation().secstruct(i)!='H') {
			Vector const & xyz( pose.residue( i ).atom( 2 ).xyz());
			Real depth=dot(xyz-center,normal)+30;
			if(depth>18 &&
			   depth<42) {
				hbond_pen++;
			}
		}
	}
	hbond_pen*=10; //total_embed weight is 0.5 in membrane_score_quick.cc
}

void
MembranePotential::termini_penalty(pose::Pose const & pose, Real & termini_pen) const
{
	termini_pen=0.0;
	if(!Menv_penalties_)
		return;
	Vector const normal(MembraneEmbed_from_pose( pose ).normal());
	Vector const center(MembraneEmbed_from_pose( pose ).center());
	termini_penalty(pose,normal,center,termini_pen);
}

void
MembranePotential::termini_penalty(pose::Pose const & pose, Vector const & normal,Vector const & center,Real & termini_pen) const
{
	termini_pen=0.0;
	if(!Menv_penalties_)
		return;
	MembraneTopology const & topology( MembraneTopology_from_pose(pose) );

	for(Size i = 1;i<=pose.total_residue();i=i+pose.total_residue()-1) {
		if(topology.allow_scoring(i))
		{
			Vector const & xyz( pose.residue( i ).atom( 2 ).xyz());
			Real depth=dot(xyz-center,normal)+30;
			if(depth>18 &&
			   depth<42) {
				termini_pen++;
			}
		}
	}
   termini_pen*=50;
}
//	( get_residue_weight_by_ss( pose.conformation().secstruct( rsd1.seqpos() ) ) +


/*void
MembranePotential::mc_normal_center(pose::Pose const & pose, Vector & normal,Vector & center,Real const temperature, Size const cycles, Real const scale_factor) const
{

}*/

void
MembranePotential::find_buried_class(
	Real const fcen6,
	Real const fcen10,
	Size & bur6,
	Size & bur10
) const
{
	if ( fcen6 <= 2 ) {
		bur6 = 2;
	} else if ( fcen6 >= 7 ) {
		bur6 = 7;
	} else {
		bur6 = static_cast <Size> (fcen6+0.5); //bw to round off correctly.
	}
	bur6=bur6-1;
	//	Size cen10=fcen10+0.5;
	Size cen10=(core::Size)(fcen10+0.5);
	if ( cen10 <= 12 ) {
		bur10 = 1; // 12;
	} else if ( ( cen10 > 12 ) && ( cen10 <= 15 ) ) {
		bur10 = 2; //15;
	} else if ( ( cen10 > 15 ) && ( cen10 <= 18 ) ) {
		bur10 = 3; //18;
	} else if ( ( cen10 > 18 ) && ( cen10 <= 21 ) ) {
		bur10 = 4; //21;
	} else if ( ( cen10 > 21 ) && ( cen10 <= 24 ) ) {
		bur10 = 5; //24;
	} else {
		bur10 = 6; //27;
	}
}

/// @details Pose must already contain a cenlist object or this method will fail.
MembraneEmbed const &
MembranePotential::MembraneEmbed_from_pose( pose::Pose const & pose ) const
{
	using core::pose::datacache::CacheableDataType::MEMBRANE_EMBED;
	return *( static_cast< MembraneEmbed const * >( pose.data().get_const_ptr( MEMBRANE_EMBED )() ));
}

/// @details Either returns a non-const reference to the cenlist object already stored
/// in the pose, or creates a new cenist object, places it in the pose, and returns
/// a non-const reference to it.
MembraneEmbed &
MembranePotential::nonconst_MembraneEmbed_from_pose( pose::Pose & pose ) const
{
	using core::pose::datacache::CacheableDataType::MEMBRANE_EMBED;

	if ( pose.data().has( MEMBRANE_EMBED ) ) {
		return *( static_cast< MembraneEmbed * >( pose.data().get_ptr( MEMBRANE_EMBED )() ));
	}
	// else
	MembraneEmbedOP membrane_embed = new MembraneEmbed;
	pose.data().set( MEMBRANE_EMBED, membrane_embed );
	return *membrane_embed;
}

/// @details Pose must already contain a cenlist object or this method will fail.
MembraneTopology const &
MembranePotential::MembraneTopology_from_pose( pose::Pose const & pose ) const
{
	using core::pose::datacache::CacheableDataType::MEMBRANE_TOPOLOGY;
	return *( static_cast< MembraneTopology const * >( pose.data().get_const_ptr( MEMBRANE_TOPOLOGY )() ));
}

/// @details Either returns a non-const reference to the cenlist object already stored
/// in the pose, or creates a new cenist object, places it in the pose, and returns
/// a non-const reference to it.
MembraneTopology &
MembranePotential::nonconst_MembraneTopology_from_pose( pose::Pose & pose ) const
{
	using core::pose::datacache::CacheableDataType::MEMBRANE_TOPOLOGY;

	if ( pose.data().has( MEMBRANE_TOPOLOGY ) ) {
		return *( static_cast< MembraneTopology * >( pose.data().get_ptr( MEMBRANE_TOPOLOGY )() ));
	}
	// else
	MembraneTopologyOP membrane_topology = new MembraneTopology;
	pose.data().set( MEMBRANE_TOPOLOGY, membrane_topology );
	return *membrane_topology;
}


}
}
