// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/MembranePotential.hh
/// @brief  Membrane Potential
/// @author Bjorn Wallner


#ifndef INCLUDED_core_scoring_MembranePotential_HH
#define INCLUDED_core_scoring_MembranePotential_HH

#include <core/types.hh>

// Unit headers
#include <core/scoring/MembranePotential.fwd.hh>
#include <core/scoring/MembraneTopology.fwd.hh>
#include <core/scoring/MembraneTopology.hh>
#include <core/scoring/EnvPairPotential.hh>

// Package headers
#include <core/scoring/EnergyGraph.fwd.hh>
#include <core/conformation/Residue.fwd.hh>

// Project headers
#include <core/pose/Pose.fwd.hh>

#include <core/util/datacache/CacheableData.hh>

// Utility headers
#include <utility/vector1.hh>
#include <numeric/xyzVector.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>

// C++


namespace core {
namespace scoring {


class MembraneEmbed : public util::datacache::CacheableData {


public:
	MembraneEmbed(): calculated_(false), spanning_(false) {};
	MembraneEmbed( MembraneEmbed const & src );


	util::datacache::CacheableDataOP
	clone() const
	{
		return new MembraneEmbed( *this );
	}

	Size
	size() const{
		return depth_.size();
	}

	Real &
	depth(Size const seqpos)
	{
		return depth_[seqpos];
	}

	void
	set_normal(Vector const & v)
	{
		normal_=v;
		//normal_.(1)=v(1);
		//normal_.(2)=v(2);
		//normal_.(3)=v(3);
	}
	void
	set_center(Vector const & v)
	{
		center_(1)=v(1);
		center_(2)=v(2);
		center_(3)=v(3);
	}

	Real
	depth(Size const seqpos) const
	{
		return depth_[seqpos];
	}

	bool &
	spanning()
	{
		return spanning_;
	}

	bool
	spanning() const
	{
		return spanning_;
	}

	bool
	calculated() const
	{
		return calculated_;
	}

	bool &
	calculated()
	{
		return calculated_;
	}

	Vector const &
	normal() const
	{
		return normal_;
	}

	Vector const &
	center() const
	{
		return center_;
	}


	void
	initialize( pose::Pose const & pose );

private:
	utility::vector1 < Real > depth_;
	Vector normal_;
	Vector center_;
	bool calculated_;
	bool spanning_;
	Size tm_projection_;

};

class MembranePotential : public EnvPairPotential {

public:
	MembranePotential();

	///
	void
	evaluate_env(
		pose::Pose const & pose,
		conformation::Residue const & rsd,
		Real const MembraneDepth,
		Real & membrane_env_score
	) const;
	//

	void
	evaluate_env(
		pose::Pose const & pose,
		conformation::Residue const & rsd,
		Real & membrane_env_score
	) const;

		///
	void
	evaluate_cbeta(
		pose::Pose const & pose,
		conformation::Residue const & rsd,
		Real & membrane_cb_score
	) const;

	///
	void
	evaluate_pair(
				  pose::Pose const & pose,
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		Real const cendist,
		Real & membrane_pair_score
	) const;

	void
	finalize( pose::Pose & pose ) const;

/*  bool
	calculated() const
	{
		return calculated_;
	}

	bool &
	calculated()
	{
		return calculated_;
	}
*/


void
compute_membrane_embedding( pose::Pose & pose) const;

void
init_membrane_center_normal(pose::Pose const & pose,
							Vector & normal,
							Vector & center) const;


void tm_projection_penalty(pose::Pose const & pose,Real & tm_proj) const;
void tm_projection_penalty(pose::Pose const & pose, Vector const & normal,Vector const & center,Real & tm_proj) const;


void hbond_penalty(pose::Pose const & pose, Real & hbond_penalty) const;
void hbond_penalty(pose::Pose const & pose, Vector const & normal,Vector const & center,Real & hbond_penalty) const;

void termini_penalty(pose::Pose const & pose, Real & termini_penalty) const;
void termini_penalty(pose::Pose const & pose, Vector const & normal,Vector const & center,Real & termin_penalty) const;

	bool
	Menv_penalties() const
	{
		return Menv_penalties_;
	}

	/*bool &
	Menv_penalties() const
	{
		return Menv_penalties_;
	}*/

protected:



/*	Real const cen_dist_cutoff2;

*/
	MembraneEmbed const & MembraneEmbed_from_pose( pose::Pose const & ) const;
	MembraneEmbed & nonconst_MembraneEmbed_from_pose( pose::Pose & ) const;
	MembraneTopology const & MembraneTopology_from_pose( pose::Pose const & ) const;
	MembraneTopology & nonconst_MembraneTopology_from_pose( pose::Pose & ) const;



//*/


private:


	void score_normal_center(pose::Pose const & pose,
							 Vector const & normal,
							 Vector const & center,
							 Real & score) const;

	void rot_perturb_vector(Vector & v, Real const & std_dev) const;  //bw does not belong here
	void rigid_perturb_vector(Vector & v, Real const & std_dev) const;  //bw does not belong here
	bool check_spanning(pose::Pose const & pose, Vector const & normal,Vector const & center) const;

	//void mc_normal_center(pose::Pose const & pose, Vector & normal,Vector & center, Real const temperature,Size const cycles,Real const scale_factor) const;
	void
	find_buried_class(
		Real const fcen6,
		Real const fcen10,
		Size & bur6,
		Size & bur10
	) const;

private: // data

	ObjexxFCL::FArray3D< Real > mem_env_log6_;
	ObjexxFCL::FArray3D< Real > mem_env_log10_;
	ObjexxFCL::FArray1D< Real > mem_cbeta_den6_;
	ObjexxFCL::FArray1D< Real > mem_cbeta_den12_;
	ObjexxFCL::FArray1D< Real > mem_cbeta_2TM_den6_;
	ObjexxFCL::FArray1D< Real > mem_cbeta_2TM_den12_;
	ObjexxFCL::FArray1D< Real > mem_cbeta_4TM_den6_;
	ObjexxFCL::FArray1D< Real > mem_cbeta_4TM_den12_;
	ObjexxFCL::FArray4D< Real > mem_pair_log_;
//	ObjexxFCL::FArray1D< Real > cenpack_log_;
/*
	Real const cen_dist6sqr_;
	Real const cen_dist10sqr_;
	Real const cen_dist12sqr_;
*/
	//cems transition regions between environment bins
	//cems transition is from +/- sqrt(36+pad6) +/- sqrt(100+pad10) etc
	Real const cen_dist5_pad;
	Real const cen_dist6_pad;
	Real const cen_dist7_pad;
	Real const cen_dist10_pad;
	Real const cen_dist12_pad;

	Real const cen_dist5_pad_plus ;
	Real const cen_dist6_pad_plus ;
	Real const cen_dist7_pad_plus ;
	Real const cen_dist10_pad_plus;
	Real const cen_dist12_pad_plus;

	Real const cen_dist5_pad_minus ;
	Real const cen_dist7_pad_minus ;
	Real const cen_dist10_pad_minus;
	Real const cen_dist12_pad_minus;

	Real const cen_dist5_pad_hinv ;
	Real const cen_dist6_pad_hinv ;
	Real const cen_dist7_pad_hinv ;
	Real const cen_dist10_pad_hinv;
	Real const cen_dist12_pad_hinv;
	bool calculated_;
	bool no_interpolate_Mpair_;
	bool Menv_penalties_;
	Size membrane_normal_cycles_;
	Real membrane_normal_magnitude_;
	Real membrane_center_magnitude_;
	Real smooth_move_frac_;

};



} // ns scoring
} // ns core

#endif
