// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/MembraneTopology.cc
/// @brief  MembraneTopology
/// @author Bjorn Wallner
///


// Unit headers
#include <core/scoring/MembraneTopology.hh>

// Package headers

#include <core/scoring/EnergyGraph.hh>

// Project headers
#include <core/chemical/AA.hh>
#include <core/chemical/VariantType.hh>
#include <core/conformation/Residue.hh>
#include <core/io/database/open.hh>
#include <core/pose/Pose.hh>
#include <core/options/option.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/after_opts.hh>

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/utility.functions.hh>

// just for debugging
//#include <ObjexxFCL/formatted.o.hh>

// C++


namespace core {
namespace scoring {

MembraneTopology::MembraneTopology( MembraneTopology const & src ) :
	CacheableData()
{
	helix_id_=src.helix_id_;
	span_=src.span_;
	full_span_=src.full_span_;
	relative_tmh_ori_=src.relative_tmh_ori_;
	total_tmhelix_=src.total_tmhelix_;
	N_term_inside_=src.N_term_inside_;
	beta_barrel_=src.beta_barrel_;
	depth_=src.depth_;
	LipidExposure_=src.LipidExposure_;
	LipidBurial_=src.LipidBurial_;
	LipoDefined_=src.LipoDefined_;
	tmregion_=src.tmregion_;
	allow_scoring_=src.allow_scoring_;
	allow_tmh_scoring_=src.allow_tmh_scoring_;
	tmh_inserted_=src.tmh_inserted_;
	init_=src.init_;
}

void
MembraneTopology::initialize(std::string const & spanfile)
{
    using namespace core::options;
	using namespace OptionKeys;
		//	std::string spanfile("BRD7.span");
	std::cout << "Initialize Membrane spanning regions with " << spanfile << "\n";
	std::string line;
	utility::io::izstream stream (spanfile);
	getline(stream,line);
	std::cout << line << "\n";
	getline(stream,line);
	Size total_residue;
	{
		std::istringstream l(line);
		std::cout << line << "\n";
		l>> total_tmhelix_ >> total_residue;
	}

	Size const max_tmhelix(total_tmhelix_);
	span_.dimension(max_tmhelix,2);
	full_span_.dimension(max_tmhelix,2);
	relative_tmh_ori_.dimension(max_tmhelix,max_tmhelix);
	helix_id_.dimension(max_tmhelix);


	getline(stream,line); //n2c
	std::cout << line << "\n";
	getline(stream,line); //parallel
	std::cout << line << "\n";
	for(Size i=1;i<=total_tmhelix_;++i)
		{
			getline(stream,line);
			{
				std::istringstream l(line);
				l >> span_(i,1) >> span_(i,2);
			}

			std::cout << line << "\n" ;
			helix_id_(i)=i;
		}
	if(total_tmhelix_==0)
		{
			utility_exit_with_message("bad format for spanfile total_tmhelix=0");
		}
	stream.close();
	stream.clear();



	for ( Size reg1 = 1; reg1 <= total_tmhelix_; ++reg1 ) {
		for ( Size reg2 = 1; reg2 <= total_tmhelix_; ++reg2 ) {
				relative_tmh_ori_(reg1,reg2)=1;
		}
	}

	full_span_(1,1)=1;
	full_span_(total_tmhelix_,2)=total_residue;

	for ( Size reg1 = 2; reg1 <= total_tmhelix_; ++reg1 ) {
		full_span_(reg1,1)=span_(reg1,1);
		full_span_(reg1-1,2)=span_(reg1,1);
	}
	depth_.resize(total_residue,0.0);
	tmregion_.resize(total_residue,false);
	allow_scoring_.resize(total_residue,true);
	allow_tmh_scoring_.resize(total_tmhelix_,true);
	tmh_inserted_=total_tmhelix_;

	for(Size i=1;i<=total_residue;++i) {
		for ( Size reg1 = 1; reg1 <= total_tmhelix_; ++reg1 ) {
			if(i>=span_(reg1,1) && i<=span_(reg1,2)) {
				tmregion_[i]=true;
				continue;
			}
		}
		//std::cout << "tmregion " << i << " " << tmregion_[i] << std::endl;
	}

	//init Lipo
	if(option[in::file::lipofile].user())
	{
		Size NumberOfConstraints(0);
		Size resnum;
		Real exposure;

		std::string lipofile(option[OptionKeys::in::file::lipofile]());
		std::cout << "init lipo using " << lipofile << "\n";
		LipidExposure_.resize(total_residue,0.0);
		LipidBurial_.resize(total_residue,0.0);
		stream.open(lipofile);

		if(stream) {
			getline(stream,line);
			getline(stream,line);
			while(!stream.eof()) {
				std::istringstream l(line);

				l >> resnum;
				l >> exposure;
				if(exposure>0) {
					LipidExposure_[resnum]+=exposure;
					NumberOfConstraints++;
					} else {
					LipidBurial_[resnum]+=std::abs(exposure);
					NumberOfConstraints++;
				}
				std::cout <<  resnum << " " << exposure << " " << LipidExposure_[resnum] << " " << LipidBurial_[resnum] << "\n";
				getline(stream,line);
			}
			stream.close();
			stream.clear();
			std::cout << NumberOfConstraints << " exposure constraints read!\n";
			if(NumberOfConstraints>0)
			{
				LipoDefined_=true;
			}
		}else{
				std::cout << "unable to open " << lipofile << "\n";
		}



	}



	init_=true;
	return;
}

void
MembraneTopology::shift_span( Size shift )
{

	for(Size i=1;i<=total_tmhelix_;++i)
		{
			span_(i,1)+=shift;
			span_(i,2)+=shift;
			full_span_(i,1)+=shift;
			full_span_(i,2)+=shift;
			if(full_span_(i,1)<=0)
			{
				utility_exit_with_message("Span out of bounds");
			}
		}
}

//void
//MembraneTopology::attach_to_pose(pose::Pose & pose)
//{
//	if ( pose.data().has( util::MEMBRANE_TOPOLOGY ) ) {
//
//		core::scoring::MembraneTopologyOP topology = *this; //new core::scoring::MembraneTopology;
//		return *( static_cast< core::scoring::MembraneTopology * >( pose.data().get_ptr( util::MEMBRANE_TOPOLOGY )() ));
//	}
	// else
//	core::scoring::MembraneTopologyOP topology = *this;
//	pose.data().set( util::MEMBRANE_TOPOLOGY, *this); //topology );

//}

void
MembraneTopology::print() const
{
	std::cout << "TOTAL TMH " << total_tmhelix_ << std::endl;
	for(Size i=1;i<=total_tmhelix_;++i)
		{
			std::cout << "SPAN " << i << " " << span_(i,1) << " " << span_(i,2) << std::endl;
		}
}


void
MembraneTopology::get_subset( utility::vector1< Size > & TMH_list, MembraneTopology & src)
{

	// Assume list is sorted for now...
	// Will add a sorter here later....

	if(TMH_list.size()>src.total_tmhelix_)
		{
			utility_exit_with_message("Too long TMH_list");
		}
	Size const len(TMH_list.size());
	span_.dimension(len,2);
	full_span_.dimension(len,2);
	relative_tmh_ori_.dimension(len,len);


	for(Size i=1;i<=len;++i)
		{
			span_(i,1)=src.span_(TMH_list[i],1);
			span_(i,2)=src.span_(TMH_list[i],2);
			full_span_(i,1)=src.full_span_(TMH_list[i],1);
			full_span_(i,2)=src.full_span_(TMH_list[i],2);
			helix_id_(i)=TMH_list[i];
			for(Size j=1;j<=TMH_list.size();++j)
				{
					relative_tmh_ori_(i,j)=src.relative_tmh_ori_(TMH_list[i],TMH_list[j]);
				}
		}



	total_tmhelix_=TMH_list.size();




}

/*
MembraneTopology &
MembraneTopology_from_pose( pose::Pose const & pose ) const
{
	return *( static_cast< MembraneTopology const * >( pose.data().get_const_ptr( util::MEMBRANE_TOPOLOGY )() ));
}
*/
}
}
