// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/ScoreFunction.cc
/// @brief  ScoreFunction class definition.
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)
/// @author Modified by Sergey Lyskov


// Unit headers
#include <core/scoring/ScoreFunction.hh>

// Package headers
#include <core/scoring/EnergiesCacheableDataType.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/scoring/ScoreFunctionInfo.hh>

#include <core/scoring/methods/EnergyMethod.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>
#include <core/scoring/methods/ContextIndependentLRTwoBodyEnergy.hh>
#include <core/scoring/methods/ContextIndependentTwoBodyEnergy.hh>
#include <core/scoring/methods/ContextDependentLRTwoBodyEnergy.hh>
#include <core/scoring/methods/ContextDependentTwoBodyEnergy.hh>
#include <core/scoring/methods/ContextIndependentOneBodyEnergy.hh>
#include <core/scoring/methods/ContextDependentOneBodyEnergy.hh>
#include <core/scoring/methods/TwoBodyEnergy.hh>
#include <core/scoring/methods/WholeStructureEnergy.hh>
#include <core/scoring/hbonds/hbonds.hh>
#include <core/scoring/LREnergyContainer.hh>
#include <core/conformation/symmetry/util.hh>
#include <core/mm/MMBondAngleResidueTypeParamSet.hh>

//Temporary!!! -- how temporary?
#include <core/scoring/constraints/ConstraintSet.hh>


// // Project headers
#include <core/pack/rotamer_set/RotamerSet.hh> // TMP HACK
#include <core/util/Tracer.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/util/prof.hh>

#include <core/io/database/open.hh>

// // Numeric headers
// #include <numeric/numeric.functions.hh>

/// Utility headers
#include <utility/io/izstream.hh>
#include <utility/string_util.hh>

/// ObjexxFCL headers
#include <ObjexxFCL/formatted.o.hh>

using namespace ObjexxFCL;
using namespace ObjexxFCL::fmt;

namespace core {
namespace scoring {



///////////////////////////////////////////////////////////////////////////////
ScoreFunction::ScoreFunction()
{
	reset();
}


ScoreFunction::~ScoreFunction() {}

///////////////////////////////////////////////////////////////////////////////
ScoreFunctionOP
ScoreFunction::clone() const
{
	ScoreFunctionOP new_score_function( new ScoreFunction( *this ) );
	return new_score_function;
}

///////////////////////////////////////////////////////////////////////////////
/// @details reset function that can be called either independently or from default
/// ScoreFunction creation
void
ScoreFunction::reset()
{
	score_function_info_current_ = true;
	score_function_info_ = new ScoreFunctionInfo;
	any_intrares_energies_ = false;
	energy_method_options_ = new methods::EnergyMethodOptions;
	initialize_methods_arrays();
	weights_.clear();
}

///////////////////////////////////////////////////////////////////////////////
/// read info from file
///

void
ScoreFunction::initialize_from_file( std::string const & filename )
{
	reset();

	add_weights_from_file( filename );
}


///////////////////////////////////////////////////////////////////////////////
/// read weights/etc from file. Does not clear weights beforehand.
///

void
ScoreFunction::add_weights_from_file( std::string const & filename )
{

	utility::io::izstream data( filename );
	if ( !data.good() ){
		utility_exit_with_message( "Unable to open weights file: "+filename );
	}

	std::string line,tag;
	ScoreType score_type;
	Real wt;
	while ( getline( data, line ) ) {
		std::istringstream l( line );
		l >> tag;
		if ( l.fail() || tag[0] == '#' ) continue;
		if ( tag == "ETABLE" ) {
			l >> tag;
			set_etable( tag );
		} else if ( tag == "METHOD_WEIGHTS" ) {
			l >> score_type;
			if ( l.fail() ) {
				utility_exit_with_message( "unrecognized score_type: "+line );
			}
			utility::vector1< Real > wts;
			while ( !l.fail() ) {
				l >> wt;
				if ( l.fail() ) break;
				wts.push_back( wt );
			}
			energy_method_options_->set_method_weights( score_type, wts );
		} else if ( tag == "STRAND_STRAND_WEIGHTS" ) {
			utility::vector1< int > values;
			int value;
			while ( !l.fail() ) {
				l >> value;
				if ( l.fail() ) break;
				values.push_back( value );
			}
			if ( values.size() != 2 )
				utility_exit_with_message( "incorrect number of arguments to STRAND_STRAND_WEIGHTS: " + line );
			energy_method_options_->set_strand_strand_weights( values[1], values[2] );
		} else if ( tag == "NO_HB_ENV_DEP" ) {
			energy_method_options_->use_hb_env_dep( false );
		} else if ( tag == "NO_HB_ENV_DEP_DNA" ) {
			energy_method_options_->use_hb_env_dep_DNA( false );
		} else if ( tag == "NO_SMOOTH_HB_ENV_DEP" ) {
			energy_method_options_->smooth_hb_env_dep( false );
		} else if ( tag == "BOND_ANGLE_CENTRAL_ATOMS_TO_SCORE" ) {
			utility::vector1<std::string> central_atoms;
			std::string central_atom;
			while ( !l.fail() ) {
				l >> central_atom;
				if ( l.fail() ) break;
				central_atoms.push_back( central_atom );
			}
			if ( ! energy_method_options_->bond_angle_residue_type_param_set() ) {
				energy_method_options_->bond_angle_residue_type_param_set( new core::mm::MMBondAngleResidueTypeParamSet() );
			}
			energy_method_options_->bond_angle_residue_type_param_set()->central_atoms_to_score( central_atoms );
		} else if ( tag == "BOND_ANGLE_USE_RESIDUE_TYPE_THETA0" ) {
			if ( ! energy_method_options_->bond_angle_residue_type_param_set() ) {
				energy_method_options_->bond_angle_residue_type_param_set( new core::mm::MMBondAngleResidueTypeParamSet() );
			}
			energy_method_options_->bond_angle_residue_type_param_set()->use_residue_type_theta0( true );
		} else {
			l.str( line );
			l >> score_type >> wt;
			if ( l.fail() ) {
				utility_exit_with_message( "bad line in file "+filename+":"+line );
			}
			set_weight( score_type, wt );
		}
	}
}



///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::set_energy_method_options(
	 methods::EnergyMethodOptions const &
	 energy_method_options_in )
{
	energy_method_options_
		= methods::EnergyMethodOptionsOP( new methods::EnergyMethodOptions( energy_method_options_in ) );
	//*energy_method_options_ = energy_method_options_in;

// Some of the energy methods only know about these options when
// they are constructed. So the safest thing is to destroy them and
// create them again.
	reset_energy_methods();

	score_function_info_current_ = false;
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::reset_energy_methods()
{
	//Note from rhiju: This is super hacky, but it was Andrew Leaver-Fay's idea.
	// Zero out the weights ... which
	// destroys the methods. Then return the weights to their
	// non-zero values to construct them again from scratch.
	EnergyMap weights_old( weights_ );
	for ( Size i = 1; i <= n_score_types; ++i ) {
		set_weight( ScoreType( i ), 0.0 );
	}
	for ( Size i = 1; i <= n_score_types; ++i ) {
		set_weight( ScoreType( i ), weights_old.get( ScoreType( i ) ) );
	}

}

void
ScoreFunction::apply_patch_from_file( std::string const & patch_tag ) {
	utility::io::izstream data1( patch_tag );
	if ( data1.good() ) {
		_apply_patch_from_file( patch_tag );
	} else {
		utility::io::izstream data2(  io::database::full_name( "scoring/weights/"+patch_tag+".wts_patch" )  );
		if ( data2.good() ) {
			_apply_patch_from_file( io::database::full_name( "scoring/weights/"+patch_tag+".wts_patch" ) );
		} else {
			utility_exit_with_message( "Unable to open weights. Neither  ./" + patch_tag + " nor " +
				io::database::full_name( "scoring/weights/"+patch_tag+".wts_patch" )  + "exist"  );
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
/// read info from file
///

void
ScoreFunction::_apply_patch_from_file( std::string const & filename )
{

	utility::io::izstream data( filename );
	if ( !data.good() ){
		utility_exit_with_message( "Unable to open weights-patch file: "+filename );
	}

	std::string line,operation;
	ScoreType score_type;
	Real value;
	while ( getline( data, line ) ) {
		if(line == "" | line[0] == '#') continue;
		std::istringstream l( line );
		l >> score_type >> operation >> value;
		if ( l.fail() ) continue;
		if ( operation == "*=" ) {
			set_weight( score_type, weights_[ score_type ]*value );

		} else if ( operation == "=" ) {
			set_weight( score_type, value );

		} else {
			utility_exit_with_message(
				"unrecognized scorefunction patch operation "+operation+" in file: "+filename
			);
		}
	}
}

/// @brief Given a filename (represented by a std::string), set the e_table for this ScoreFunction.
void
ScoreFunction::set_etable(
	std::string const & etable_name
)
{
	// ensure that this has been done before the relevant EnergyMethod is created
	// this needs to be fixed to be smarter
	//	assert( weights_[ fa_atr ] == 0.0 && weights_[ fa_rep ] == 0.0 && weights_[ fa_sol ] == 0.0 );
	energy_method_options_->etable_type( etable_name );
	reset_energy_methods();
	score_function_info_current_ = false;
}

///
void
ScoreFunction::set_method_weights(
	ScoreType const & t,
	utility::vector1< Real > const & wts
)
{
	//	assert( weights_[ t ] == 0.0 );
	energy_method_options_->set_method_weights( t, wts );
	reset_energy_methods();
	score_function_info_current_ = false;
}

///////////////////////////////////////////////////////////////////////////////
ScoreFunction &
ScoreFunction::operator=( ScoreFunction const & src )
{
	if ( this == &src ) return *this;

	// copy the weights
	weights_ = src.weights_;

	// deep copy of the energy method options
	energy_method_options_ = new methods::EnergyMethodOptions( * src.energy_method_options_ );

	// copy the methods:
	initialize_methods_arrays(); // clears & sizes the arrays

	for ( AllMethods::const_iterator it = src.all_methods_.begin(),
			ite = src.all_methods_.end(); it != ite; ++it ) {
		add_method( (*it)->clone() );
	}
	update_intrares_energy_status();
	assert( check_methods() );

	/// SL: Fri Jun 29 12:51:25 EDT 2007 @744 /Internet Time/
	score_function_info_current_ = src.score_function_info_current_;
	score_function_info_ = new ScoreFunctionInfo( *src.score_function_info_ );

	any_intrares_energies_ = src.any_intrares_energies_;

	return *this;
}

///////////////////////////////////////////////////////////////////////////////
ScoreFunction::ScoreFunction( ScoreFunction const & src ):
	ReferenceCount()
{
	*this = src;
}


///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::show( std::ostream & out ) const
{
	out << "ScoreFunction::show():\nweights:";
	for ( int i=1; i<= n_score_types; ++i ) {
		if ( weights_[ ScoreType(i) ] != 0.0 ) {
			out << " (" << ScoreType(i) << ' '<< weights_[ScoreType(i)] << ')';
		}
	}
	out << '\n';

	out << "energy_method_options: " << *energy_method_options_ << '\n';
}

///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::show( std::ostream & out,  pose::Pose & pose ) const
{
	(*this)(pose); //make sure scores are set
	out << "------------------------------------------------------------\n";
	out << " Scores                       Weight   Raw Score Wghtd.Score\n";
	out << "------------------------------------------------------------\n";
	float sum_weighted=0.0;
	for ( int i=1; i<= n_score_types; ++i ) {
		if ( weights_[ ScoreType(i) ] != 0.0 ) {
			out << ' ' << LJ(24,ScoreType(i)) << ' '<< F(9,3,weights_[ ScoreType(i) ]) << "   "
					<<  	 F(9,3,pose.energies().total_energies()[ ScoreType(i) ] ) << "   "
					<<  	 F(9,3, weights_[ ScoreType(i) ] * pose.energies().total_energies()[ ScoreType(i) ] )
					<< '\n';
			sum_weighted += weights_[ ScoreType(i) ] * pose.energies().total_energies()[ ScoreType(i) ];
		}
	}
	out << "---------------------------------------------------\n";
	out << " Total weighted score:                    " << F(9,3,sum_weighted) << '\n';
}

///////////////////////////////////////////////////////////////////////////////
void
show_detail( std::ostream & out, EnergyMap & energies,  EnergyMap weights )
{
	out << "------------------------------------------------------------\n";
	out << " Scores                       Weight   Raw Score Wghtd.Score\n";
	out << "------------------------------------------------------------\n";
	float sum_weighted=0.0;
	for ( int i=1; i<= n_score_types; ++i ) {
		if ( weights[ ScoreType(i) ] != 0.0 ) {
			out << ' ' << LJ(24,ScoreType(i)) << ' '<< F(9,3,weights[ ScoreType(i) ]) << "   "
					<<  	 F(9,3,energies[ ScoreType(i) ] ) << "   "
					<<  	 F(9,3, weights[ ScoreType(i) ] * energies[ ScoreType(i) ] )
					<< '\n';
			sum_weighted += weights[ ScoreType(i) ] * energies[ ScoreType(i) ];
		}
	}
	out << "---------------------------------------------------\n";
	out << " Total weighted score:                    " << F(9,3,sum_weighted) << '\n';
}

///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::show_line_headers( std::ostream & out ) const
{

	for ( int i=1; i<= n_score_types; ++i ) {
		if ( weights_[ ScoreType(i) ] != 0.0 ) {
			out << ' ' << LJ(16,ScoreType(i)) << ' ';
		}
	}

}
///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::show_line( std::ostream & out,  pose::Pose & pose ) const
{

	float sum_weighted=0.0;
	for ( int i=1; i<= n_score_types; ++i ) {
		if ( weights_[ ScoreType(i) ] != 0.0 ) {
			sum_weighted += weights_[ ScoreType(i) ] * pose.energies().total_energies()[ ScoreType(i) ];
		}
	}
	out << F(9,3,sum_weighted);

	for ( int i=1; i<= n_score_types; ++i ) {
		if ( weights_[ ScoreType(i) ] != 0.0 ) {
			// OL: added the " " to the output. Otherwise column separation can be missing in silent-files
			// OL: if large numbers come into play
			// OL: reduced F(9,3) to F(8,3) to account for the extra space.
			out << " " << F(8,3, weights_[ ScoreType(i) ] * pose.energies().total_energies()[ ScoreType(i) ] );
		}
	}

}


///////////////////////////////////////////////////////////////////////////////

// to start out, just thinking fullatom energies
//
// NOTE: no freakin rotamer trials inside scoring!

Real
ScoreFunction::operator()( pose::Pose & pose ) const
{
	// completely unnecessary temporary hack to force refold if nec. for profiling
	pose.residue( pose.total_residue() );

	//fpd fail if this is called on a symmetric pose
	runtime_assert( !core::conformation::symmetry::is_symmetric( pose ) );

	PROF_START( util::SCORE );
	//std::cout << "ScoreFunction::operator()\n";

	// notify the pose that we are starting a score evaluation.
	// also tells the cached energies object about the scoring
	// parameters, for the purposes of invalidating cached data
	// if necessary
	//
	// at this point the dof/xyz-moved information will be converted
	// to a domain map. Energy/neighbor links between pair-moved residues
	// will be deleted, and 1d energies of res-moved residues will be
	// cleared.
	//
	// further structure modification will be prevented until scoring is
	// completed
	//
	pose.scoring_begin( *this );
	//std::cout << "ScoreFunction::operator() 1\n";

	if ( pose.energies().total_energy() != 0.0 ) {
		std::cout << "STARTING SCORE NON-ZERO!" << std::endl;
	}

	// ensure that the total_energies are zeroed out -- this happens in Energies.scoring_begin()
	// unneccessary pose.energies().total_energies().clear();
	//std::cout << "ScoreFunction::operator() 2\n";

	// do any setup necessary
	setup_for_scoring( pose );
	//std::cout << "ScoreFunction::operator() 3\n";

	// evaluate the residue-residue energies that only exist between
	// neighboring residues
	PROF_START( util::NEIGHBOR_ENERGIES );

	eval_twobody_neighbor_energies( pose );

	PROF_STOP ( util::NEIGHBOR_ENERGIES );

	// evaluate the residue pair energies that exist between possibly-distant residues
	PROF_START( util::LONG_RANGE_ENERGIES );

	eval_long_range_twobody_energies( pose );

	PROF_STOP ( util::LONG_RANGE_ENERGIES );

	// evaluate the onebody energies -- rama, dunbrack, ...
	eval_onebody_energies( pose );

	// give energyfunctions a chance update/finalize energies
	// etable nblist calculation is performed here
	for ( AllMethods::const_iterator it=all_methods_.begin(),
			it_end = all_methods_.end(); it != it_end; ++it ) {
		(*it)->finalize_total_energy( pose, *this, pose.energies().finalized_energies() );
	}

	pose.energies().total_energies() += pose.energies().finalized_energies();

	// dot the weights with the scores
	pose.energies().total_energy() = pose.energies().total_energies().dot( weights_ );
	pose.energies().total_energies()[ total_score ] = pose.energies().total_energy();

	// notify that scoring is over
	pose.scoring_end( *this );

	PROF_STOP ( util::SCORE );

	return pose.energies().total_energy();
}

Real
ScoreFunction::score( pose::Pose & pose ) const {
	return (*this)(pose);
}

/*
/// @details This operates much like the regular score function evaluation, except
/// that it waits to compute the two-body energies until after the other components of the
/// score function evaluation has completed.  It relies on the mechanism already present in
/// the Energies object to update the component energies lazily.  This mechanism will
/// update both the EnergyGraph and the EnergyGraphLite objects.
Real
ScoreFunction::score_components( pose::Pose & pose ) const
{
	// completely unnecessary temporary hack to force refold if nec. for profiling
	pose.residue( pose.total_residue() );

	PROF_START( util::SCORE );
	//std::cout << "ScoreFunction::operator()\n";

	// notify the pose that we are starting a score evaluation.
	// also tells the cached energies object about the scoring
	// parameters, for the purposes of invalidating cached data
	// if necessary
	//
	// at this point the dof/xyz-moved information will be converted
	// to a domain map. Energy/neighbor links between pair-moved residues
	// will be deleted, and 1d energies of res-moved residues will be
	// cleared.
	//
	// further structure modification will be prevented until scoring is
	// completed
	//
	pose.scoring_begin( *this );
	//std::cout << "ScoreFunction::operator() 1\n";

	if ( pose.energies().total_energy() != 0.0 ) {
		std::cout << "STARTING SCORE NON-ZERO!" << std::endl;
	}

	// ensure that the total_energies are zeroed out -- this happens in Energies.scoring_begin()
	// unneccessary pose.energies().total_energies().clear();
	//std::cout << "ScoreFunction::operator() 2\n";

	// do any setup necessary
	setup_for_scoring( pose );

	// evaluate the residue pair energies that exist between possibly-distant residues
	PROF_START( util::LONG_RANGE_ENERGIES );

	eval_long_range_twobody_energies( pose );

	PROF_STOP ( util::LONG_RANGE_ENERGIES );

	// evaluate the onebody energies -- rama, dunbrack, ...
	eval_onebody_energies( pose );

	// give energyfunctions a chance update/finalize energies
	// etable nblist calculation is performed here
	for ( AllMethods::const_iterator it=all_methods_.begin(),
			it_end = all_methods_.end(); it != it_end; ++it ) {
		(*it)->finalize_total_energy( pose, *this, pose.energies().finalized_energies() );
	}

	// notify that scoring is over
	pose.scoring_end( *this );

	PROF_STOP ( util::SCORE );

	/// Now go back and score to the two-body energies -- triggered automatically accessing the
	/// energy graph.
	pose.energies().energy_graph();

	/// The above call forced the computation of the total_energies emap; the total score can
	/// now be computed.
	EnergyMap const & total_energies( pose.energies().total_energies() );

	pose.energies().total_energy() = total_energies.dot( weights_ );
	return pose.energies().total_energy();
}*/

///////////////////////////////////////////////////////////////////////////////
core::Real
ScoreFunction::get_sub_score_exclude_res(
	pose::Pose & pose,
	utility::vector1< core::Size > exclude_list
) const
{

	// This is slow currently and will be made faster soon by just grabbing the
	// hbonds rather than recalculating

//  methods::EnergyMethodOptions opts = sfxn_->energy_method_options();
//  opts.decompose_bb_hb_into_pair_energies( true );
//	set_energy_method_options( opts );
//  (*this)(pose);

	// cached energies object
	Energies & energies( pose.energies() );
	if ( !energies.energies_updated() ) (*this)(pose); //apply score
	// the neighbor/energy links
	EnergyGraph & energy_graph( energies.energy_graph() );

	// should be zeroed at the beginning of scoring
	Real total( 0.0 ); //( energies.totals() );
	bool exclude=false;
	for ( Size i=1, i_end = pose.total_residue(); i<= i_end; ++i ) {
		//conformation::Residue const & resl( pose.residue( i ) );

		exclude = false;
		for ( Size k = 1; k <= exclude_list.size(); k ++ ){
			if( exclude_list[k] == i ){ exclude = true; break; };
		}
		if( exclude ) continue;

		for ( graph::Graph::EdgeListIter
				iru  = energy_graph.get_node(i)->upper_edge_list_begin(),
				irue = energy_graph.get_node(i)->upper_edge_list_end();
				iru != irue; ++iru ) {
			EnergyEdge & edge( static_cast< EnergyEdge &> (**iru) );
			Size const j( edge.get_second_node_ind() );

			exclude = false;
			for ( Size k = 1; k <= exclude_list.size(); k ++ ){
				if( exclude_list[k] == j ){ exclude = true; break; };
			}
			if( exclude ) continue;

			total += edge.dot( weights_ );

		} // nbrs of i
	} // i=1,nres


	///////////////////////////////////////////////////////////////
	// context independent onebody energies

	for ( Size i=1; i<= pose.total_residue(); ++i ) {
		exclude = false;
		for ( Size k = 1; k <= exclude_list.size(); k ++ ){
			if( exclude_list[k] == i ){ exclude = true; break; };
		}
		if( exclude ) continue;

		EnergyMap & emap( energies.onebody_energies( i ) );

		total += weights_.dot( emap, ci_1b_types() );
		total += weights_.dot( emap, cd_1b_types() );


		// 2body energy methods are allowed to define 1body intxns ///////////////////////////
		if ( any_intrares_energies_ ) {
			// context independent:
			total += weights_.dot( emap, ci_2b_types() );
			total += weights_.dot( emap, ci_lr_2b_types() );
			total += weights_.dot( emap, cd_2b_types() );
			total += weights_.dot( emap, cd_lr_2b_types() );
		}
	}

	// add in hbonding
	if ( pose.energies().data().has( EnergiesCacheableDataType::HBOND_SET )) {

		hbonds::HBondSet const & hbond_set
			( static_cast< hbonds::HBondSet const & >
				( pose.energies().data().get( EnergiesCacheableDataType::HBOND_SET )));

		Real lr_bbE( 0.0 ), sr_bbE( 0.0 ), bb_scE( 0.0 ), scE( 0.0 );
		get_hbond_energies_excl( hbond_set, sr_bbE, lr_bbE, bb_scE, scE, exclude_list );

		total += weights_[ hbond_sr_bb ] * sr_bbE;
		total += weights_[ hbond_lr_bb ] * lr_bbE;
	}

	/// NO LONGER POSSIBLE -- IS THIS DESIRED?
	///show_detail( std::cout , totals,  weights_ );

	return total;
}




///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::eval_twobody_neighbor_energies(
	pose::Pose & pose
) const
{

	// cached energies object
	Energies & energies( pose.energies() );

	EnergyMap & total_energies( const_cast< EnergyMap & > ( energies.total_energies() ) );

	// the neighbor/energy links
	EnergyGraph & energy_graph( energies.energy_graph() );

	// are we using the atom-atom nblist?
	// renaming for true purpose: are we minimizing -- if so,
	// zero the energies stored on edges in the Energy graph, but do
	// not mark the edges as having had their energies computed
	bool const minimizing( energies.use_nblist() );

	TwoBodyEnergyMap tbemap;

	for ( Size i=1, i_end = pose.total_residue(); i<= i_end; ++i ) {
		conformation::Residue const & resl( pose.residue( i ) );
		for ( graph::Graph::EdgeListIter
				iru  = energy_graph.get_node(i)->upper_edge_list_begin(),
				irue = energy_graph.get_node(i)->upper_edge_list_end();
				iru != irue; ++iru ) {
			EnergyEdge & edge( static_cast< EnergyEdge & > (**iru) );

			Size const j( edge.get_second_node_ind() );
			conformation::Residue const & resu( pose.residue( j ) );
			// the pair energies cached in the link
			///TwoBodyEnergyMap & emap( energy_edge->energy_map() );
			tbemap.zero( cd_2b_types() );
			tbemap.zero( ci_2b_types() );

			// the context-dependent guys can't be cached, so they are always reevaluated
			eval_cd_2b( resl, resu, pose, tbemap );

			if ( edge.energies_not_yet_computed() ) {
				// energies not yet computed? <-> (during minimization w/ nblist) moving rsd pair
				/// TEMP zero portions;

				if ( minimizing ) {
					// confirm that this rsd-rsd interaction will be included
					// in the atom pairs on the nblist:
					//assert( ( pose.energies().domain_map(i) !=
					//					pose.energies().domain_map(j) ) ||
					//				( pose.energies().res_moved(i) ) );
					// ensure that these are zeroed, since we will hit them at the
					// end inside the nblist calculation
					// they almost certainly should be, since the energies have not
					// yet been computed...
					eval_ci_2b( resl, resu, pose, tbemap );
					edge.store_active_energies( tbemap );
					// do not mark energies as computed!!!!!!!!!!!!!
				} else {
					eval_ci_2b( resl, resu, pose, tbemap );
					edge.store_active_energies( tbemap );
					edge.mark_energies_computed();
				}
			} else {
				/// Read the CI energies from the edge, as they are still valid;
				for ( Size ii = 1; ii <= ci_2b_types().size(); ++ii ) {
					tbemap[ ci_2b_types()[ ii ]] = edge[ ci_2b_types()[ ii ] ];
				}

				/// Save the freshly computed CD energies on the edge
				edge.store_active_energies( tbemap, cd_2b_types() );
			}

			total_energies.accumulate( tbemap, ci_2b_types() );
			total_energies.accumulate( tbemap, cd_2b_types() );
		} // nbrs of i
	} // i=1,nres

}

void
ScoreFunction::eval_long_range_twobody_energies( pose::Pose & pose ) const
{
	EnergyMap & total_energies(const_cast< EnergyMap & > (pose.energies().total_energies()));

	for ( CI_LR_2B_Methods::const_iterator iter = ci_lr_2b_methods_.begin(),
			iter_end = ci_lr_2b_methods_.end(); iter != iter_end; ++iter ) {

		LREnergyContainerOP lrec = pose.energies().nonconst_long_range_container( (*iter)->long_range_type() );
		if ( !lrec || lrec->empty() ) continue; // only score non-emtpy energies.

		// Potentially O(N^2) operation...
		for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
			for ( ResidueNeighborIteratorOP
					rni = lrec->upper_neighbor_iterator_begin( ii ),
					rniend = lrec->upper_neighbor_iterator_end( ii );
					(*rni) != (*rniend); ++(*rni) ) {
				EnergyMap emap;
				if ( ! rni->energy_computed() ) {
					(*iter)->residue_pair_energy(
						pose.residue(ii),
						pose.residue( rni->upper_neighbor_id() ),
						pose, *this, emap );
					rni->save_energy( emap );

					/// DANGER DANGER DANGER.  use_nblist() now means "In the process of a minimization". There is
					/// no such thing as a non-neighborlist minimization.  This will confuse people at some point.
					/// We ought to have some "are we minimizing currently" flag who's meaning can be decoupled
					/// from the neighborlist idea.
					if ( ! pose.energies().use_nblist() ) {
						rni->mark_energy_computed();
					}
				} else {
					rni->retrieve_energy( emap ); // pbmod
				}
				total_energies += emap;
			}
		}

	}

	for ( CD_LR_2B_Methods::const_iterator iter = cd_lr_2b_methods_.begin(),
			iter_end = cd_lr_2b_methods_.end(); iter != iter_end; ++iter ) {

		LREnergyContainerOP lrec
			= pose.energies().nonconst_long_range_container( (*iter)->long_range_type() );

		// Potentially O(N^2) operation...
		for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
			for ( ResidueNeighborIteratorOP
					rni = lrec->upper_neighbor_iterator_begin( ii ),
					rniend = lrec->upper_neighbor_iterator_end( ii );
					(*rni) != (*rniend); ++(*rni) ) {

				EnergyMap emap;
				(*iter)->residue_pair_energy(
					pose.residue(ii),
					pose.residue( rni->upper_neighbor_id() ),
					pose, *this, emap );

				rni->save_energy( emap );
				//rni->mark_energy_computed();

				total_energies += emap;

			}
		}

	}
}

///////////////////////////////////////////////////////////////////////////////
/*
void
ScoreFunction::accumulate_residue_total_energies(
	pose::Pose & pose
) const
{
	// cached energies object
	Energies & energies( pose.energies() );

	// the neighbor/energy links
	EnergyGraph & energy_graph( energies.energy_graph() );

	// debug
	assert( !energies.use_nblist() && energies.energies_updated() );

	// zero out the scores we are going to accumulate
	for ( Size i=1, i_end = pose.total_residue(); i<= i_end; ++i ) {
		EnergyMap & emap( energies.residue_total_energies(i) );
		emap.zero();
		emap += energies.onebody_energies(i);
	}

	for ( Size i=1, i_end = pose.total_residue(); i<= i_end; ++i ) {
		for ( graph::Graph::EdgeListIter
				iru  = energy_graph.get_node(i)->upper_edge_list_begin(),
				irue = energy_graph.get_node(i)->upper_edge_list_end();
				iru != irue; ++iru ) {

			EnergyEdge * edge( static_cast< EnergyEdge *> (*iru) );

			Size const j( edge->get_second_node_ind() );

			// the pair energies cached in the link
			TwoBodyEnergyMap const & emap( edge->energy_map());

			assert( !edge->energies_not_yet_computed() );

			// accumulate energies
			energies.residue_total_energies(i).accumulate( emap, ci_2b_types(), 0.5 );
			energies.residue_total_energies(i).accumulate( emap, cd_2b_types(), 0.5 );

			energies.residue_total_energies(j).accumulate( emap, ci_2b_types(), 0.5 );
			energies.residue_total_energies(j).accumulate( emap, cd_2b_types(), 0.5 );
		} // nbrs of i
	} // i=1,nres

	Real tmp_score( 0.0 );

	for ( LR_2B_Methods::const_iterator iter = lr_2b_methods_.begin(),
			iter_end = lr_2b_methods_.end(); iter != iter_end; ++iter ) {
		// Why is this non-const?!
		LREnergyContainerOP lrec
			= pose.energies().nonconst_long_range_container( (*iter)->long_range_type() );
		assert( lrec );
		if ( lrec->empty() ) continue;
		ScoreTypes const & lrec_score_types = score_types_by_method_type_[ (*iter)->method_type() ];

		// Potentially O(N^2) operation...
		for ( Size i = 1; i <= pose.total_residue(); ++i ) {
			for ( ResidueNeighborConstIteratorOP
					rni = lrec->const_upper_neighbor_iterator_begin( i ),
					rniend = lrec->const_upper_neighbor_iterator_end( i );
					(*rni) != (*rniend); ++(*rni) ) {
				Size j = rni->upper_neighbor_id();
				EnergyMap emap;
				rni->retrieve_energy( emap );

				energies.residue_total_energies(i).accumulate( emap, lrec_score_types, 0.5 );
				energies.residue_total_energies(j).accumulate( emap, lrec_score_types, 0.5 );
			}
		}
	}

	for ( Size i=1, i_end = pose.total_residue(); i<= i_end; ++i ) {
		EnergyMap & emap( energies.residue_total_energies(i) );
		emap[ total_score ] = emap.dot( weights_ );
		tmp_score += emap[ total_score ];
	}

#ifdef NDEBUG
	return;
#endif

	/////////////////////////////////////////////////////////////////////////////
	// just for debugging, since this involves some extra calculation

	{ // extra scores
		EnergyMap emap;
		for ( AllMethods::const_iterator it=all_methods_.begin(),
				it_end = all_methods_.end(); it != it_end; ++it ) {
			(*it)->finalize_total_energy( pose, *this, emap );
		}
		//std::cout << "before: " << tmp_score << std::endl;
		tmp_score += emap.dot( weights_ );


		// this warning is only relevant if all energies are 1body or 2body,
		// ie no whole-structure energies which are not assigned to an
		// individual residue
		Real const expected_score( energies.total_energy() );
		//std::cout << "[ DEBUG ] score-check: " << tmp_score << ' ' <<
		//	expected_score << std::endl;
		if ( std::abs( tmp_score - expected_score ) > 1e-2 ) {
			std::cout << "[ WARNING ] score mismatch: " << tmp_score << ' ' <<
				expected_score << std::endl;
		}
	}
} // accumulate_residue_total_energies

*/

///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::eval_onebody_energies( pose::Pose & pose ) const
{
	// context independent onebody energies
	Energies & energies( pose.energies() );
	EnergyMap & totals( energies.total_energies() );

	for ( Size i=1; i<= pose.total_residue(); ++i ) {
		EnergyMap & emap( energies.onebody_energies( i ) );

		// 1body intxns ///////////////////////////
		if ( energies.res_moved( i ) ) {
			// have to recalculate
			emap.clear(); // should already have been done when the domain map was internalized
			eval_ci_1b( pose.residue(i), emap );
		}

		emap.zero( cd_1b_types() ); // cant be cached
		eval_cd_1b( pose.residue(i), pose, emap );


		// 2body energy methods are allowed to define 1body intxns ///////////////////////////
		if ( any_intrares_energies_ ) {
			// context independent:
			if ( energies.res_moved( i ) ) {
				eval_ci_intrares_energy( pose.residue(i), pose, emap );
			}
			// context dependent
			emap.zero( cd_2b_types() ); // cant be cached (here only intrares are relevant)
			emap.zero( cd_lr_2b_types() ); // cant be cached (here only intrares are relevant)
			eval_cd_intrares_energy( pose.residue(i), pose, emap );
		}

		totals += emap;

		energies.reset_res_moved( i ); // mark one body energies as having been calculated
		//std::cout << "totals: "<<  i  << totals;
	}
}

void
ScoreFunction::eval_intrares_energy(
	conformation::Residue const & rsd,
	pose::Pose const & pose,
	EnergyMap & emap
) const
{
	if ( ! any_intrares_energies_ ) return;
	for ( TWO_B_Methods::const_iterator iter = ci_2b_intrares_.begin(),
			iter_end = ci_2b_intrares_.end(); iter != iter_end; ++iter ) {
		(*iter)->eval_intrares_energy( rsd, pose, *this, emap );
	}
	for ( TWO_B_Methods::const_iterator iter = cd_2b_intrares_.begin(),
			iter_end = cd_2b_intrares_.end(); iter != iter_end; ++iter ) {
		(*iter)->eval_intrares_energy( rsd, pose, *this, emap );
	}
}

void
ScoreFunction::evaluate_rotamer_intrares_energies(
	pack::rotamer_set::RotamerSet const & set,
	pose::Pose const & pose,
	utility::vector1< pack::PackerEnergy > & energies
) const
{
	if ( ! any_intrares_energies_ ) return;
	for ( TWO_B_Methods::const_iterator iter = ci_2b_intrares_.begin(),
			iter_end = ci_2b_intrares_.end(); iter != iter_end; ++iter ) {
		(*iter)->evaluate_rotamer_intrares_energies( set, pose, *this, energies );
	}
	for ( TWO_B_Methods::const_iterator iter = cd_2b_intrares_.begin(),
			iter_end = cd_2b_intrares_.end(); iter != iter_end; ++iter ) {
		(*iter)->evaluate_rotamer_intrares_energies( set, pose, *this, energies );
	}
}

void
ScoreFunction::evaluate_rotamer_intrares_energy_maps(
	pack::rotamer_set::RotamerSet const & set,
	pose::Pose const & pose,
	utility::vector1< EnergyMap > & emaps
) const
{
	if ( ! any_intrares_energies_ ) return;
	for ( TWO_B_Methods::const_iterator iter = ci_2b_intrares_.begin(),
			iter_end = ci_2b_intrares_.end(); iter != iter_end; ++iter ) {
		(*iter)->evaluate_rotamer_intrares_energy_maps( set, pose, *this, emaps );
	}
	for ( TWO_B_Methods::const_iterator iter = cd_2b_intrares_.begin(),
			iter_end = cd_2b_intrares_.end(); iter != iter_end; ++iter ) {
		(*iter)->evaluate_rotamer_intrares_energy_maps( set, pose, *this, emaps );
	}
}

void
ScoreFunction::eval_ci_intrares_energy(
	conformation::Residue const & rsd,
	pose::Pose const & pose,
	EnergyMap & emap
) const
{
	if ( ! any_intrares_energies_ ) return;
	for ( TWO_B_Methods::const_iterator iter = ci_2b_intrares_.begin(),
			iter_end = ci_2b_intrares_.end(); iter != iter_end; ++iter ) {
		(*iter)->eval_intrares_energy( rsd, pose, *this, emap );
	}
}

void
ScoreFunction::eval_cd_intrares_energy(
	conformation::Residue const & rsd,
	pose::Pose const & pose,
	EnergyMap & emap
) const
{
	if ( ! any_intrares_energies_ ) return;
	for ( TWO_B_Methods::const_iterator iter = cd_2b_intrares_.begin(),
			iter_end = cd_2b_intrares_.end(); iter != iter_end; ++iter ) {
		(*iter)->eval_intrares_energy( rsd, pose, *this, emap );
	}
}


///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::eval_ci_1b(
	conformation::Residue const & rsd,
	EnergyMap & emap
) const
{
	for ( CI_1B_Methods::const_iterator iter = ci_1b_methods_.begin(),
			iter_end = ci_1b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->residue_energy( rsd, emap );
	}
}

void
ScoreFunction::eval_cd_1b(
	conformation::Residue const & rsd,
	pose::Pose const & pose,
	EnergyMap & emap
) const
{
	for ( CD_1B_Methods::const_iterator iter = cd_1b_methods_.begin(),
			iter_end = cd_1b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->residue_energy( rsd, pose, emap );
	}
}


///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::eval_ci_2b(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	TwoBodyEnergyMap & emap
) const
{
	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->residue_pair_energy( rsd1, rsd2, pose, *this, emap );
	}

}


void
ScoreFunction::eval_ci_2b_bb_bb(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	TwoBodyEnergyMap & emap
) const
{
	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->backbone_backbone_energy( rsd1, rsd2, pose, *this, emap );
	}

}


void
ScoreFunction::eval_ci_2b_bb_sc(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	TwoBodyEnergyMap & emap
) const
{
	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->backbone_sidechain_energy( rsd1, rsd2, pose, *this, emap );
	}

}

void
ScoreFunction::eval_ci_2b_sc_sc(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	TwoBodyEnergyMap & emap
) const
{
	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->sidechain_sidechain_energy( rsd1, rsd2, pose, *this, emap );
	}

}



///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::eval_cd_2b(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	TwoBodyEnergyMap & emap
) const
{
	for ( CD_2B_Methods::const_iterator
			iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->residue_pair_energy( rsd1, rsd2, pose, *this, emap );
	}
}

void
ScoreFunction::eval_cd_2b_bb_bb(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	TwoBodyEnergyMap & emap
) const
{
	for ( CD_2B_Methods::const_iterator iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->backbone_backbone_energy( rsd1, rsd2, pose, *this, emap );
	}

}


void
ScoreFunction::eval_cd_2b_bb_sc(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	TwoBodyEnergyMap & emap
) const
{
	for ( CD_2B_Methods::const_iterator iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->backbone_sidechain_energy( rsd1, rsd2, pose, *this, emap );
	}

}

void
ScoreFunction::eval_cd_2b_sc_sc(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	TwoBodyEnergyMap & emap
) const
{
	for ( CD_2B_Methods::const_iterator iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->sidechain_sidechain_energy( rsd1, rsd2, pose, *this, emap );
	}

}



/// @brief score the sidechain from rsd1 against the entirety of rsd2
///
/// low fidelity description of what it would be like to replace
/// the sidechain of rsd1 at a particular position -- only those scoring functions
/// that specify they should be included in the bump check will be evaluated.
/// The inaccuracy in this function is based on the change in LK solvation types
/// and/or charges for backbone atoms when residues are mutated from one amino
/// acid type to another (or from any other residue with backbone atoms)
void
ScoreFunction::bump_check_full(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	EnergyMap & emap
) const
{
	for ( CD_2B_Methods::const_iterator iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->bump_energy_full( rsd1, rsd2, pose, *this, emap );
	}
	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->bump_energy_full( rsd1, rsd2, pose, *this, emap );
	}

}


/// @details low fidelity description of what it would be like to replace
/// the sidechain of rsd1 at a particular position knowing that
/// rsd2's sidechain might also change sychronously.
/// only those scoring functions that specify they should be included in the
/// bump check will be evaluated.
/// The inaccuracy in this function is based on the change in LK solvation types
/// and/or charges for backbone atoms when residues are mutated from one amino
/// acid type to another (or from any other residue with backbone atoms)
void
ScoreFunction::bump_check_backbone(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	EnergyMap & emap
) const
{
	for ( CD_2B_Methods::const_iterator iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->bump_energy_backbone( rsd1, rsd2, pose, *this, emap );
	}
	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->bump_energy_backbone( rsd1, rsd2, pose, *this, emap );
	}

}

void
ScoreFunction::evaluate_rotamer_pair_energies(
	pack::rotamer_set::RotamerSet const & set1,
	pack::rotamer_set::RotamerSet const & set2,
	pose::Pose const & pose,
	FArray2D< pack::PackerEnergy > & energy_table
) const
{
	for ( CD_2B_Methods::const_iterator iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->evaluate_rotamer_pair_energies(
			set1, set2, pose, *this, weights(), energy_table );
	}
	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->evaluate_rotamer_pair_energies(
			set1, set2, pose, *this, weights(), energy_table );
	}

}

/// @details Determines whether two positions could have non-zero interaction energy
bool
ScoreFunction::are_they_neighbors(
	pose::Pose const & pose,
	Size const pos1,
	Size const pos2
) const
{
	conformation::Residue const & rsd1( pose.residue( pos1 ) );
	conformation::Residue const & rsd2( pose.residue( pos2 ) );

	Real const intxn_radius( rsd1.nbr_radius() + rsd2.nbr_radius() + max_atomic_interaction_cutoff() );

	return ( ( intxn_radius > 0 &&
						 rsd1.nbr_atom_xyz().distance_squared( rsd2.nbr_atom_xyz() ) < intxn_radius * intxn_radius ) ||
					 ( any_lr_residue_pair_energy( pose, pos1, pos2 ) ) );
}


bool
ScoreFunction::any_lr_residue_pair_energy(
	pose::Pose const & pose,
	Size res1,
	Size res2
) const
{
	for ( LR_2B_Methods::const_iterator iter = lr_2b_methods_.begin(),
			iter_end = lr_2b_methods_.end(); iter != iter_end; ++iter ) {
		if ( (*iter)->defines_residue_pair_energy( pose, res1, res2 )) {
			return true;
		}
	}

	return false;
}

ScoreFunction::AllMethodsIterator
ScoreFunction::all_energies_begin() const
{
	return all_methods_.begin();
}

ScoreFunction::AllMethodsIterator
ScoreFunction::all_energies_end() const
{
	return all_methods_.end();
}

ScoreFunction::LR_2B_MethodIterator
ScoreFunction::long_range_energies_begin() const
{
	return lr_2b_methods_.begin();
}

ScoreFunction::LR_2B_MethodIterator
ScoreFunction::long_range_energies_end() const
{
	return lr_2b_methods_.end();
}

ScoreFunction::TWO_B_MethodIterator
ScoreFunction::ci_2b_intrares_begin() const
{
	return ci_2b_intrares_.begin();
}

ScoreFunction::TWO_B_MethodIterator
ScoreFunction::ci_2b_intrares_end() const
{
	return ci_2b_intrares_.end();
}

ScoreFunction::TWO_B_MethodIterator
ScoreFunction::cd_2b_intrares_begin() const
{
	return cd_2b_intrares_.begin();
}

ScoreFunction::TWO_B_MethodIterator
ScoreFunction::cd_2b_intrares_end() const
{
	return cd_2b_intrares_.end();
}

ScoreFunction::CI_LR_2B_MethodIterator
ScoreFunction::ci_lr_2b_methods_begin() const
{
  return ci_lr_2b_methods_.begin();
}

ScoreFunction::CI_LR_2B_MethodIterator
ScoreFunction::ci_lr_2b_methods_end() const
{
  return ci_lr_2b_methods_.end();
}

ScoreFunction::CD_LR_2B_MethodIterator
ScoreFunction::cd_lr_2b_methods_begin() const
{
  return cd_lr_2b_methods_.begin();
}

ScoreFunction::CD_LR_2B_MethodIterator
ScoreFunction::cd_lr_2b_methods_end() const
{
  return cd_lr_2b_methods_.end();
}

ScoreFunction::WS_MethodIterator
ScoreFunction::ws_methods_begin() const
{
  return ws_methods_.begin();
}

ScoreFunction::WS_MethodIterator
ScoreFunction::ws_methods_end() const
{
  return ws_methods_.end();
}

/// @details Evaluate the energies between a set of rotamers and a single other (background) residue
void
ScoreFunction::evaluate_rotamer_background_energies(
	pack::rotamer_set::RotamerSet const & set1,
	conformation::Residue const & residue2,
	pose::Pose const & pose,
	utility::vector1< pack::PackerEnergy > & energy_vector
) const
{
	for ( CD_2B_Methods::const_iterator iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->evaluate_rotamer_background_energies(
			set1, residue2, pose, *this, weights(), energy_vector );
	}
	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->evaluate_rotamer_background_energies(
			set1, residue2, pose, *this, weights(), energy_vector );
	}
}


///////////////////////////////////////////////////////////////////////////////
/// @details set the weight
void
ScoreFunction::set_weight( ScoreType const & t, Real const & setting )
{
	using namespace methods;

	Real const old_weight( weights_[t] );
	weights_[t] = setting;

	if ( setting == Real(0.0) ) {
		if ( old_weight != Real(0.0) ) {

			EnergyMethodOP method( methods_by_score_type_[t] );
			assert( method );

			// check to see if there's a non-zero weight for this method
			bool has_nonzero_weight( false );
			for ( ScoreTypes::const_iterator
							iter     = method->score_types().begin(),
							iter_end = method->score_types().end(); iter != iter_end; ++iter ) {
				if ( weights_[ *iter ] != 0.0 ) {
					has_nonzero_weight = true;
					break;
				}
			}
			if ( ! has_nonzero_weight ) {
				score_function_info_current_ = false; // cached score_function_info_ object no longer up-to-date
				remove_method( method );
			}
		} else {
			// Nothing needs doing.  Early return.  Be careful of intrares status and check_methods() for early returns!!!
			return;
		}
	} else {
		if ( old_weight == Real(0.0) ) {
			// do we already have a method that evaluates this score??
			if ( ! methods_by_score_type_[ t ] != 0  &&
					( t != python ) )  { // sheffler

				score_function_info_current_ = false; // cached score_function_info_ object no longer up-to-date
				assert( std::abs( old_weight ) < 1e-2 ); // sanity

				// get pointer to the energyfunction that evaluates this score
				EnergyMethodOP method ( ScoringManager::get_instance()->energy_method( t, *energy_method_options_));
				add_method( method );
			} else {
				if ( t == python ) return; // escape for python defined scores -- intentially avoid update_intrares_status
			}
		} else {
			// only adjusted a weight that was already non-zero.  Nothing else needs doing.
			// Early return.  Be careful of intrares status and check_methods() for early returns!!!
			assert( old_weight != Real( 0.0 ) || weights_[ t ] != Real( 0.0 ) );
			return;
		}
	}

	// arriving here, we know that a weight went either to or from zero.
	// perform some neccessary bookkeeping.
	assert( old_weight == Real( 0.0 ) || weights_[ t ] == Real( 0.0 ) );

	update_intrares_energy_status();
	assert( check_methods() );
}


///////////////////////////////////////////////////////////////////////////////
/// @details get the weight
Real
ScoreFunction::get_weight( ScoreType const & t) const
{
	return  weights_[t];
}


///////////////////////////////////////////////////////////////////////////////
/// @details private
bool
ScoreFunction::check_methods() const
{
	using namespace methods;

	EnergyMap counter;
	// counter indexes scoretypes to the number of energy methods. Each scoretype should only
	// map to one energy method.
	for ( AllMethods::const_iterator it=all_methods_.begin(),
		it_end = all_methods_.end(); it != it_end; ++it ) {
		for ( ScoreTypes::const_iterator t=(*it)->score_types().begin(),
			t_end=(*it)->score_types().end(); t != t_end; ++t ) {
			//++counter[ *t ]; //++ not defined for floats, right?
			counter[ *t ] += 1;
		}
	}
	for ( EnergyMap::const_iterator it=counter.begin(), it_end=counter.end();
		it != it_end; ++it ) {
		// total hack!!!
		ScoreType const t( static_cast< ScoreType >( it - counter.begin() + 1));
		Real const count( *it );
		if ( count > 1.5 ) {
			utility_exit_with_message("multiple methods for same score type!" );
		} else if ( count < 0.5 && std::abs( weights_[ t ] > 1e-2 ) ) {
			utility_exit_with_message(
				"no method for scoretype " + name_from_score_type( t ) + " w/nonzero weight!"
			);
		}
	}
	return true;
}

void
ScoreFunction::update_intrares_energy_status()
{
	any_intrares_energies_ = false;
	ci_2b_intrares_.clear();
	cd_2b_intrares_.clear();
	for ( CD_2B_Methods::const_iterator iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		if ( (*iter)->defines_intrares_energy(weights())) {
			any_intrares_energies_ = true;
			cd_2b_intrares_.push_back( (*iter)() );
		}
	}
	for ( CD_LR_2B_Methods::const_iterator iter = cd_lr_2b_methods_.begin(),
			iter_end = cd_lr_2b_methods_.end(); iter != iter_end; ++iter ) {
		if ( (*iter)->defines_intrares_energy(weights())) {
			any_intrares_energies_ = true;
			cd_2b_intrares_.push_back( (*iter)() );
		}
	}

	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		if ( (*iter)->defines_intrares_energy( weights() )) {
			any_intrares_energies_ = true;
			ci_2b_intrares_.push_back( (*iter)() );
		}
	}
	for ( CI_LR_2B_Methods::const_iterator iter = ci_lr_2b_methods_.begin(),
			iter_end = ci_lr_2b_methods_.end(); iter != iter_end; ++iter ) {
		if ( (*iter)->defines_intrares_energy( weights() )) {
			any_intrares_energies_ = true;
			ci_2b_intrares_.push_back( (*iter)() );
		}
	}
}

void
ScoreFunction::setup_for_scoring(
	pose::Pose & pose
) const
{
	for ( AllMethods::const_iterator it=all_methods_.begin(),
			it_end = all_methods_.end(); it != it_end; ++it ) {
		(*it)->setup_for_scoring( pose, *this );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::setup_for_packing(
	pose::Pose & pose,
	pack::task::PackerTask const & task
) const
{
	for ( AllMethods::const_iterator iter=all_methods_.begin(),
			iter_end= all_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->setup_for_packing( pose, task );
	}
}

///
void
ScoreFunction::prepare_rotamers_for_packing(
	pose::Pose const & pose,
	pack::rotamer_set::RotamerSet & set
) const
{
	for ( AllMethods::const_iterator iter=all_methods_.begin(),
			iter_end= all_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->prepare_rotamers_for_packing( pose, set );
	}
}

///
void
ScoreFunction::update_residue_for_packing(
	pose::Pose & pose,
	Size resid
) const
{
	for ( AllMethods::const_iterator iter=all_methods_.begin(),
			iter_end= all_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->update_residue_for_packing( pose, resid );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::setup_for_derivatives(
	pose::Pose & pose
) const
{
	//std::cout << "ScoreFunction::setup_for_derivatives" << std::endl;
	for ( AllMethods::const_iterator iter=all_methods_.begin(),
			iter_end= all_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->setup_for_derivatives( pose, *this );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::finalize_after_derivatives(
	pose::Pose & pose
) const
{
	//std::cout << "ScoreFunction::setup_for_derivatives" << std::endl;
	for ( AllMethods::const_iterator iter=all_methods_.begin(),
			iter_end= all_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->finalize_after_derivatives( pose, *this );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::setup_for_minimizing(
	pose::Pose & pose,
	optimization::MinimizerMap const & min_map
) const
{
	for ( AllMethods::const_iterator iter=all_methods_.begin(),
			iter_end= all_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->setup_for_minimizing( pose, *this, min_map );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::eval_atom_derivative(
	id::AtomID const & atom_id,
	pose::Pose const & pose,
	kinematics::DomainMap const & domain_map,
	Vector & F1,
	Vector & F2
) const
{
	for ( AllMethods::const_iterator iter=all_methods_.begin(),
			iter_end= all_methods_.end(); iter != iter_end; ++iter ) {
		(*iter)->eval_atom_derivative( atom_id, pose, domain_map, *this, weights_, F1, F2 );
	}


}

///////////////////////////////////////////////////////////////////////////////
Real
ScoreFunction::eval_dof_derivative(
	id::DOF_ID const & dof_id,
	id::TorsionID const & torsion_id,
	pose::Pose const & pose
) const
{
	Real deriv(0.0);

	for ( AllMethods::const_iterator iter=all_methods_.begin(),
			iter_end= all_methods_.end(); iter != iter_end; ++iter ) {
// 	for ( CI_1B_Methods::const_iterator iter = ci_1b_methods_.begin(),
// 			iter_end = ci_1b_methods_.end(); iter != iter_end; ++iter ) {
		deriv += (*iter)->eval_dof_derivative( dof_id, torsion_id, pose, *this, weights_ );
	}
	return deriv;
}

///////////////////////////////////////////////////////////////////////////////
// void
// ScoreFunction::zero_energies(
// 	methods::EnergyMethodType const & t,
// 	EnergyMap & emap
// ) const
// {
// 	using namespace methods;

// 	for ( AllMethods::const_iterator iter= all_methods_.begin(),
// 					iter_end = all_methods_.end(); iter != iter_end; ++iter ) {
// 		if ( (*iter)->method_type() == t ) {
// 			for ( ScoreTypes::const_iterator it2 = (*iter)->score_types().begin(),
// 							it2_end = (*iter)->score_types().end(); it2 != it2_end; ++it2 ) {
// 				emap[ *it2 ] = 0.0;
// 			}
// 		}
// 	}
// }


///////////////////////////////////////////////////////////////////////////////
ScoreFunctionInfoOP
ScoreFunction::info() const
{
	if ( ! score_function_info_current_ ) {
		score_function_info_->initialize_from( *this );
		score_function_info_current_ = true;
	}
	return new ScoreFunctionInfo( *score_function_info_ );
}

///////////////////////////////////////////////////////////////////////////////
/// @details private -- handles setting the derived data

void
ScoreFunction::add_method( methods::EnergyMethodOP method )
{
	using namespace methods;

	all_methods_.push_back( method );

	// mapping from ScoreType to EnergyMethod
	for ( ScoreTypes::const_iterator
			iter     = method->score_types().begin(),
			iter_end = method->score_types().end(); iter != iter_end; ++iter ) {
		methods_by_score_type_[ *iter ] = method;
		score_types_by_method_type_[ method->method_type() ].push_back( *iter );
	}

	// PHIL replace these with utility::down_cast when it's working
	switch ( method->method_type() ) {
	case ci_2b:
		// there must be an easier way to cast these owning pointers!
		ci_2b_methods_.push_back
			( static_cast< ContextIndependentTwoBodyEnergy* >( method() ) );
		break;

	case cd_2b:
		cd_2b_methods_.push_back
			( static_cast< ContextDependentTwoBodyEnergy* >( method() ) );
		break;

	case ci_1b:
		ci_1b_methods_.push_back
			( static_cast< ContextIndependentOneBodyEnergy* >( method() ) );
		break;

	case cd_1b:
		cd_1b_methods_.push_back
			( static_cast< ContextDependentOneBodyEnergy* >( method() ) );
		break;

	case ci_lr_2b:
		ci_lr_2b_methods_.push_back(
			static_cast< ContextIndependentLRTwoBodyEnergy* >( method() ) );
		lr_2b_methods_.push_back(
			static_cast< LongRangeTwoBodyEnergy* >( method() ) );
		break;

	case cd_lr_2b:
		cd_lr_2b_methods_.push_back(
			static_cast< ContextDependentLRTwoBodyEnergy* >( method() ) );
		lr_2b_methods_.push_back(
			static_cast< LongRangeTwoBodyEnergy* >( method() ) );
		break;

	case ws:
		ws_methods_.push_back(
			static_cast< WholeStructureEnergy* >( method() ) );
		break;

	default:
		utility_exit_with_message( "unrecognized method type " );
	} // switch
}


/// @details  Add a scoring method that's not necessarily included in the core library
/// @note  Currently only works for methods associated with a single specific score_type
void
ScoreFunction::add_extra_method(
	ScoreType const & new_type,
	Real const new_weight,
	methods::EnergyMethod const & new_method
)
{
	if ( weights_[ new_type]  != Real( 0.0 ) ||
			 new_weight == Real(0.0) ||
			 new_method.score_types().size() != 1 ||
			 new_method.score_types().front() != new_type ) {
		utility_exit_with_message( "bad call to ScoreFunction::add_extra_method" );
	}
	weights_[ new_type ] = new_weight;
	add_method( new_method.clone() );
}


/// @details  Add a scoring method that's not necessarily included in the core library
///
void
ScoreFunction::add_extra_method(
	std::map< ScoreType, Real > const & new_weights,
	methods::EnergyMethod const & new_method
)
{
	for ( std::map< ScoreType, Real >::const_iterator it= new_weights.begin(); it != new_weights.end(); ++it ) {
		ScoreType const new_type( it->first );
		Real const new_weight( it->second );
		if ( ( weights_[ new_type ] != Real(0.0) ) ||
				 ( std::find( new_method.score_types().begin(), new_method.score_types().end(), new_type ) ==
					 new_method.score_types().end() ) ) {
			utility_exit_with_message( "bad call to ScoreFunction::add_extra_method" );
		}
		weights_[ new_type ] = new_weight;
	}
	add_method( new_method.clone() );
}


///////////////////////////////////////////////////////////////////////////////
/// private -- handles setting the derived data

// This can/should be moved to utility/vector1.hh
template< class T >
inline
void
vector1_remove( utility::vector1< T > & v, T const & t )
{
	assert( std::find( v.begin(), v.end(), t ) != v.end() );
	v.erase( std::find( v.begin(), v.end(), t ) );
}

void
ScoreFunction::remove_method( methods::EnergyMethodOP method )
{
	using namespace methods;
	vector1_remove( all_methods_, method );

	// mapping from ScoreType to EnergyMethod
	for ( ScoreTypes::const_iterator
			iter     = method->score_types().begin(),
			iter_end = method->score_types().end(); iter != iter_end; ++iter ) {
		methods_by_score_type_[ *iter ] = 0;
		vector1_remove( score_types_by_method_type_[ method->method_type() ], *iter );
	}

	bool rebuild_lr_methods( false );
	// PHIL replace these with utility::down_cast when it's working
	switch ( method->method_type() ) {
	case ci_2b:
		// there must be an easier way to cast these owning pointers!
		vector1_remove( ci_2b_methods_,
			ContextIndependentTwoBodyEnergyOP(( static_cast< ContextIndependentTwoBodyEnergy* >( method() ) )));
		break;

	case cd_2b:
		vector1_remove( cd_2b_methods_,
			ContextDependentTwoBodyEnergyOP(( static_cast< ContextDependentTwoBodyEnergy* >( method() ) ) ));
		break;

	case ci_1b:
		vector1_remove( ci_1b_methods_,
			ContextIndependentOneBodyEnergyOP(( static_cast< ContextIndependentOneBodyEnergy* >( method() ) )));
		break;

	case cd_1b:
		vector1_remove( cd_1b_methods_,
			ContextDependentOneBodyEnergyOP(( static_cast< ContextDependentOneBodyEnergy* >( method() ) )));
		break;

	case ci_lr_2b:
		rebuild_lr_methods = true;
		vector1_remove( ci_lr_2b_methods_,
			ContextIndependentLRTwoBodyEnergyOP(static_cast< ContextIndependentLRTwoBodyEnergy* >( method() )));
		break;

	case cd_lr_2b:
		rebuild_lr_methods = true;
		vector1_remove( cd_lr_2b_methods_,
			ContextDependentLRTwoBodyEnergyOP( static_cast< ContextDependentLRTwoBodyEnergy* >( method() )));
		break;

	case ws:
		vector1_remove( ws_methods_,
			WholeStructureEnergyOP( static_cast< WholeStructureEnergy* >( method() )));
		break;

	default:
		utility_exit_with_message( "unrecognized method type " );
	} // switch

	if ( rebuild_lr_methods )  {
		lr_2b_methods_.clear();
		for ( CD_LR_2B_Methods::const_iterator iter = cd_lr_2b_methods_.begin(),
				iter_end = cd_lr_2b_methods_.end(); iter != iter_end; ++iter ) {
			lr_2b_methods_.push_back( *iter );
		}
		for ( CI_LR_2B_Methods::const_iterator iter = ci_lr_2b_methods_.begin(),
				iter_end = ci_lr_2b_methods_.end(); iter != iter_end; ++iter ) {
			lr_2b_methods_.push_back( *iter );
		}
	}

}

///////////////////////////////////////////////////////////////////////////////
void
ScoreFunction::initialize_methods_arrays()
{
	// clear (may not be necessary)
	all_methods_.clear();
	methods_by_score_type_.clear();
	score_types_by_method_type_.clear();

	ci_2b_methods_.clear();
	cd_2b_methods_.clear();
	ci_1b_methods_.clear();
	cd_1b_methods_.clear();
	ci_lr_2b_methods_.clear();
	cd_lr_2b_methods_.clear();
	lr_2b_methods_.clear();
	ws_methods_.clear();

	// some need resizing
	methods_by_score_type_.resize( n_score_types, 0 );
	//std::fill( methods_by_score_type_.begin(), methods_by_score_type_.end(),  ); // wipe old pointers
	for ( Size ii = 1; ii <= n_score_types; ++ii ) { methods_by_score_type_[ ii ] = 0; }
	score_types_by_method_type_.resize( methods::n_energy_method_types );
	for ( Size ii = 1; ii <= methods::n_energy_method_types; ++ii ) {
		score_types_by_method_type_[ ii ].clear();
	}
}

/// @details determines the furthest-reach of the short-range two body energies
/// as well as the whole-structure energies, which are allowed to require
/// that the EnergyGraph have edges of a minimum length.
Distance
ScoreFunction::max_atomic_interaction_cutoff() const {
	Distance max_cutoff = 0;

	for ( CD_2B_Methods::const_iterator iter = cd_2b_methods_.begin(),
			iter_end = cd_2b_methods_.end(); iter != iter_end; ++iter ) {
		if ( (*iter)->atomic_interaction_cutoff() > max_cutoff ) {
			max_cutoff = (*iter)->atomic_interaction_cutoff();
		}
	}
	for ( CI_2B_Methods::const_iterator iter = ci_2b_methods_.begin(),
			iter_end = ci_2b_methods_.end(); iter != iter_end; ++iter ) {
		if ( (*iter)->atomic_interaction_cutoff() > max_cutoff ) {
			max_cutoff = (*iter)->atomic_interaction_cutoff();
		}
	}
	for ( WS_Methods::const_iterator iter = ws_methods_.begin(),
			iter_end = ws_methods_.end(); iter != iter_end; ++iter ) {
		if ( (*iter)->atomic_interaction_cutoff() > max_cutoff ) {
			max_cutoff = (*iter)->atomic_interaction_cutoff();
		}
	}

	return max_cutoff;
}

///@brief scoring function fills in the context graphs that its energy methods require
///
/// input vector should be false and have num_context_graph_types slots.  Each method
/// ors its required context graphs
void
ScoreFunction::indicate_required_context_graphs(
	utility::vector1< bool > & context_graphs_required
) const
{
	 for ( AllMethods::const_iterator it=all_methods_.begin(),
			it_end = all_methods_.end(); it != it_end; ++it ) {
		(*it)->indicate_required_context_graphs(context_graphs_required);
	}
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////




} // namespace scoring
} // namespace core
