// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/ScoreFunctionInfo.cc
/// @brief  Score function class descriptor
/// @author Phil Bradley

// Unit Headers
#include <core/scoring/ScoreFunctionInfo.hh>

// Package Headers
#include <core/scoring/ContextGraph.hh>
#include <core/scoring/ContextGraphTypes.hh>
#include <core/scoring/ContextGraphFactory.hh>
#include <core/scoring/ScoreFunction.hh>


namespace core {
namespace scoring {

ScoreFunctionInfo::ScoreFunctionInfo( ScoreFunction const & scorefxn )
:
	context_graphs_required_( num_context_graph_types, false )
{
	initialize_from( scorefxn );
}

/// @brief initializes three peices of data that describe the score function,
/// the atomic interaction distance, the context neighbor distance, and the
/// context graphs required by the scoring function to be properly evaluated
///
/// now also including the EnergyMethodOptions object, which holds eg the etable
/// name and (currently) the reference energy aa-weights
///
void
ScoreFunctionInfo::initialize_from( ScoreFunction const & scorefxn ) {

	max_atomic_interaction_distance_ = scorefxn.max_atomic_interaction_cutoff();

	std::fill( context_graphs_required_.begin(), context_graphs_required_.end(), false );
	scorefxn.indicate_required_context_graphs( context_graphs_required_ );

	max_context_neighbor_cutoff_ = 0;
	for ( uint ii = 1; ii <= num_context_graph_types; ++ii ) {
		if ( context_graphs_required_[ ii ] )
		{
			ContextGraphOP example_graph = ContextGraphFactory::create_context_graph( ContextGraphType(ii) );
			if ( example_graph->neighbor_cutoff() > max_context_neighbor_cutoff_ ) {
				max_context_neighbor_cutoff_ = example_graph->neighbor_cutoff();
			}
		}
	}
	for ( uint ii = 1; ii <= n_score_types; ++ii ) {
		scores_present_[ ScoreType ( ii ) ] = scorefxn.has_zero_weight( ScoreType( ii ) ) ? 0 : 1;
	}

	energy_method_options_ = scorefxn.energy_method_options();
}

bool
ScoreFunctionInfo::requires_context_graph( ContextGraphType cgt ) const {
	return context_graphs_required_[ cgt ];
}


}
}
