// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/ScoreType.hh
/// @brief  Score type enumeration
/// @author Phil Bradley


#ifndef INCLUDED_core_scoring_ScoreType_HH
#define INCLUDED_core_scoring_ScoreType_HH

// Project Headers
#include <core/types.hh>

// Utility Headers
#include <utility/vector1.hh>

// STL Headers
#include <cassert>
#include <iostream>


namespace core {
namespace scoring {

	/////////////////////////////////////////////////////////////////////////////////
	/////// WARNING WARNING WARNING
	///////
	/////// if you add a new scoretype please also add its string name in ScoreTypeManager.cc
	///////
	/////// WARNING WARNING WARNING
	/////////////////////////////////////////////////////////////////////////////////

/// @brief  Type for looking up cached energies
/// I guess we could get rid of the fa_ prefix, except maybe for
/// fa_pair, to distinguish from std::pair and the centroid pair score...
enum ScoreType {

	/// begin short ranged ci2b scores -- these guys are cached
	/// in the energy graph -- when appropriate --
	/// they are reused between rounds of scoring.
	fa_atr = 1, //enumeration starts at 1 for indexing utility::vector1
	fa_rep,
	fa_sol,
	fa_intra_atr,
	fa_intra_rep,
	fa_intra_sol,
	lk_hack,
	coarse_fa_atr,
	coarse_fa_rep,
	coarse_fa_sol,
	coarse_beadlj,
	mm_twist,      // could be lr 2benergy and not in energy graph
	mm_bend,       // could be lr 2benergy and not in energy graph
	mm_stretch,    // could be lr 2benergy and not in energy graph
	lk_costheta,
	lk_polar,
	lk_nonpolar,
//	csd_torsion, //commenting out until it is implemented
	hack_elec,
	hack_elec_bb_bb,
	hack_elec_bb_sc,
	hack_elec_sc_sc,
	h2o_hbond,
	dna_dr,
	dna_bp,
	dna_bs,
	peptide_bond,
	pcs, //Pseudocontact Shift Energy

	//Split out hack_elec for RNA.
	hack_elec_rna_phos_phos,
	hack_elec_rna_phos_sugr,
	hack_elec_rna_phos_base,
	hack_elec_rna_sugr_sugr,
	hack_elec_rna_sugr_base,
	hack_elec_rna_base_base,

	rna_fa_atr_base,
	rna_fa_rep_base,

	/// proline closure energy
	pro_close,
	rama2b,

	vdw,     // centroid
	cenpack, // centroid

	hybrid_vdw,     // hybrid centroid+fa

	fa_cust_pair_dist,  // custom short range 2b

	// for in-python runtime-defined methods: most inclusive positioning in the
	// score type enum: user may define any kind of energy method
	// However, the user may define only a single runtime defined method
	python,

	n_ci_2b_score_types = python, /// keep this guy at the end of the ci2b scores
	//end ci2b scores

	// Begin short-ranged, context dependent two-body energy method types.
	// These are also cached in the edges of the EnergyGraph.
	fa_pair, /// == fa_pair_pol_pol
	fa_pair_aro_aro,
	fa_pair_aro_pol,
	fa_pair_pol_pol,
	fa_plane,
	hbond_sr_bb,
	hbond_lr_bb,
	hbond_bb_sc,
	hbond_sr_bb_sc,
	hbond_lr_bb_sc,
	hbond_sc,

	//Carbon hydrogen bonds?
	ch_bond,
	ch_bond_bb_bb,
	ch_bond_sc_sc,
	ch_bond_bb_sc,
	// Geometric solvation
	geom_sol,
	occ_sol_pw,
	occ_sol_exact,

	pair, // centroid
	Mpair,
	// sucker atom energy
	suck,

	//RNA stuff
	//Low resolution
	rna_rg,           // Radius of gyration for RNA
	rna_vdw,          // low res clash check for RNA
	rna_base_backbone,          // Bases to 2'-OH, phosphates, etc.
	rna_backbone_backbone,      // 2'-OH to 2'-OH, phosphates, etc.
	rna_repulsive,              // mainly phosphate-phosphate repulsion
	rna_base_pair_pairwise,    // Base-base interactions (Watson-Crick and non-Watson-Crick)
	rna_base_axis_pairwise,    // Force base normals to be parallel
	rna_base_stagger_pairwise, // Force base pairs to be in same plane.
	rna_base_stack_pairwise,   // Stacking interactions
	rna_base_stack_axis_pairwise,   // Stacking interactions should involve parallel bases.


	//RNA stuff
	// This is a filtered version of the pairwise RNA low resolution terms above,
	//  disallows a base edge to form  more than one base pair, and
	//  disallows two bases to both stack and pair.
	// THIS IS NOT REALLY PAIR-WISE, but is calculated in a finalize_energy
	//  step at the end of a 2-body  score function (RNA_PairwiseLowResolutionEnergy.cc)
	rna_base_pair,    // Base-base interactions (Watson-Crick and non-Watson-Crick)
	rna_base_axis,    // Force base normals to be parallel
	rna_base_stagger, // Force base pairs to be in same plane.
	rna_base_stack,   // Stacking interactions
	rna_base_stack_axis,   // Stacking interactions should involve parallel bases.


	// High resolution
	rna_torsion,       // RNA torsional potential.
	rna_sugar_close,   // constraints to keep RNA sugar closed, and with reasonably ideal geometry
	fa_stack,          // stacking interaction modeled as pairwise atom-atom interactions

	// centroid interchain 1b (docking) scores
	interchain_pair,
	interchain_vdw,

	//
	// end short ranged two body scores
	n_shortranged_2b_score_types = interchain_vdw, // keep this guy at the end of the sr ci/cd 2b scores
	// 30 as of 1/7/2007 -- don't ever code using the literal "30", this is just a helpful count

	gb_elec,
	//Full atom disulfide terms
	dslf_ss_dst,
	dslf_cs_ang,
	dslf_ss_dih,
	dslf_ca_dih,
	dslf_cbs_ds,
	//Centroid disulfide terms
	dslfc_cen_dst,
	dslfc_cb_dst,
	dslfc_ang,
	dslfc_cb_dih,
	dslfc_bb_dih,

	atom_pair_constraint,
	constant_constraint,
	coordinate_constraint,
	angle_constraint,
	dihedral_constraint,
	big_bin_constraint,
	rna_bond_geometry, // deviations from ideal geometry


	rama,
	omega,
	fa_dun,
	p_aa_pp,
	h2o_intra,
	ref,
	envsmooth,

	/// Whole structure energies
	/// centroid whole structure energies
	env,
	cbeta,
	Menv,
	Mcbeta,
	Menv_hbond,
	Menv_termini,
	Menv_tm_proj,
	Mlipo,
	rg, // radius of gyration
	co, // contact order
	hs_pair,
	ss_pair,
	rsigma,
	sheet,
	rdc,//NMR residual dipolar coupling energy
	rdc_rohl,
  // end centroid whole structure energies
	holes,
	holes_decoy,
	holes_resl,
	holes_min,

	dab_sasa, // classic 1.4A probe solvant accessible surface area
	dab_sev,  // solvent excluded volume -- volume of atoms inflated by 1.4A

	// centroid interhcain 1b (docking) scores -- Demonstrate that these improve performance or remove them
	interchain_env,
	interchain_contact,
	//

	chainbreak,
	linear_chainbreak,
	overlap_chainbreak,
	distance_chainbreak,
	dof_constraint,


	//Neighbor Vector solvation approximation
	neigh_vect,
	neigh_count,
	neigh_vect_raw,

	//Symmetry bonus
	symE_bonus,

	// Other energies.

	// packing score energy
	pack_stat,

	// model-quality metrics.
	rms,


	// for ResidueConstraint
	res_type_constraint,

	// for BackboneStubConstraint
	backbone_stub_constraint,

	surface,
	p_aa,
	unfolded,

	// fit-to-density scores
	elec_dens_window,
	elec_dens_whole_structure_ca,
	elec_dens_whole_structure_allatom,

	// etc etc
	// Why is there a total score?
	total_score,

	/// This element marks the end of the active score types.  Elements in the enumeration
	/// up to this point will have space allocated for them in the EnergyMap object.  Elements
	/// past this point are considered inactive and will not have space allocated for them.
	/// If you wish to use an inactive score type, you must move that score type into its appropriate
	/// position in the ScoreType enumeration (described above) and then recompile.
	/// Inactive score types must still have their names included in the ScoreTypeManager's
	/// string-to-score-type map.
	n_score_types = total_score,

	/// This element marks the very end of the score type enumeration.  Elements between
	/// the n_score_types element and this element are considered inactive.  They may not
	/// be used by any EnergyMethod or they will result in an out-of-bounds write and
	/// unexpected behavior.  To use an inactived score type, the score type must be moved
	/// to an earlier position in this enumeration, and the program must be recompiled.
	/// Keep this guy last.
	end_of_score_type_enumeration = total_score
};


typedef utility::vector1< ScoreType > ScoreTypes;

ScoreType
score_type_from_name( std::string const & name );

std::string
name_from_score_type( ScoreType  score_type );

	/// @brief input operator for ScoreType enum type
std::istream & operator >>( std::istream & is, ScoreType & t );
	/// @brief output operator for ScoreType enum type
std::ostream & operator <<( std::ostream & os, ScoreType const & t );

} // namespace scoring
} // namespace core


#endif
