// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 sw=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/ScoringManager.cc
/// @brief  Scoring manager class
/// @author Andrew Leaver-Fay (leaverfa@email.unc.edu)

// Unit headers
#include <core/scoring/ScoringManager.hh>

#include <core/options/keys/score.OptionKeys.gen.hh>

#include <core/scoring/carbon_hbonds/CarbonHBondPotential.hh>
#include <core/scoring/PairEPotential.hh>
#include <core/scoring/EnvPairPotential.hh>
#include <core/scoring/MembranePotential.hh>
#include <core/scoring/InterchainPotential.hh>
#include <core/scoring/SecondaryStructurePotential.hh>
#include <core/scoring/dunbrack/RotamerLibrary.hh>
#include <core/scoring/Ramachandran.hh>
#include <core/scoring/Ramachandran2B.hh>
#include <core/scoring/OmegaTether.hh>
#include <core/scoring/GenBornPotential.hh>
#include <core/scoring/AtomVDW.hh>
#include <core/scoring/rna/RNA_AtomVDW.hh>
#include <core/scoring/geometric_solvation/DatabaseOccSolEne.hh>
#include <core/scoring/dna/DNA_BasePotential.hh>
#include <core/scoring/rna/RNA_LowResolutionPotential.hh>
#include <core/scoring/rna/RNA_TorsionPotential.hh>
#include <core/scoring/dna/DirectReadoutPotential.hh>
#include <core/scoring/P_AA.hh>
#include <core/scoring/WaterAdductHBondPotential.hh>
#include <core/scoring/disulfides/FullatomDisulfidePotential.hh>
#include <core/scoring/disulfides/CentroidDisulfidePotential.hh>
#include <core/scoring/UnfoldedStatePotential.hh>
//#include <core/scoring/methods/PCS/PseudocontactShiftEnergy.hh>

// Package headers
#include <core/scoring/methods/EnergyMethodCreator.hh>
#include <core/scoring/etable/Etable.hh>
#include <core/coarse/CoarseEtable.hh>

//#include <core/scoring/hbonds/HBondEnergy.hh>

#include <core/mm/MMTorsionLibrary.hh>
#include <core/mm/MMBondAngleLibrary.hh>



//#include <core/scoring/carbon_hbonds/CarbonHBondEnergy.hh>
//#include <core/scoring/methods/ChainbreakEnergy.hh>
//#include <core/scoring/methods/LinearChainbreakEnergy.hh>
//#include <core/scoring/methods/DistanceChainbreakEnergy.hh>
//#include <core/scoring/methods/CSD_TorsionEnergy.hh>
//#include <core/scoring/methods/DNA_BaseEnergy.hh>
//#include <core/scoring/methods/DunbrackEnergy.hh>
//#include <core/scoring/methods/CenPairEnergy.hh>
//#include <core/scoring/methods/EnvEnergy.hh>
//#include <core/scoring/methods/MembraneEnvEnergy.hh>
//#include <core/scoring/methods/MembraneEnvPenalties.hh>
//#include <core/scoring/methods/MembraneCbetaEnergy.hh>
//#include <core/scoring/methods/MembraneCenPairEnergy.hh>
//#include <core/scoring/methods/MembraneLipo.hh>
//#include <core/scoring/methods/EnvSmoothEnergy.hh>
//#include <core/scoring/methods/DirectReadoutEnergy.hh>
//#include <core/scoring/methods/GenBornEnergy.hh>
//#include <core/scoring/geometric_solvation/GeometricSolEnergy.hh>
//#include <core/scoring/geometric_solvation/OccludedHbondSolEnergy.hh>
//#include <core/scoring/geometric_solvation/ExactOccludedHbondSolEnergy.hh>
//#include <core/scoring/hackelec/HackElecEnergy.hh>
//#include <core/scoring/hackelec/RNAHackElecEnergy.hh>
//#include <core/scoring/methods/LK_hack.hh>
//#include <core/scoring/methods/LK_CosThetaEnergy.hh>
//#include <core/scoring/methods/InterchainEnvEnergy.hh>
//#include <core/scoring/methods/InterchainPairEnergy.hh>
//#include <core/scoring/methods/MMBondAngleEnergy.hh>
//#include <core/scoring/methods/MMTorsionEnergy.hh>
//#include <core/scoring/methods/OmegaTetherEnergy.hh>
//#include <core/scoring/methods/P_AA_pp_Energy.hh>

//#include <core/scoring/NV/NVscore.hh>
#include <core/scoring/NV/NVlookup.hh>

#include <core/scoring/symE/symE.hh>
//#include <core/scoring/methods/P_AA_Energy.hh>
//#include <core/scoring/methods/PairEnergy.hh>
//#include <core/scoring/methods/ProClosureEnergy.hh>
//#include <core/scoring/methods/RamachandranEnergy.hh>
//#include <core/scoring/methods/RamachandranEnergy2B.hh>
//#include <core/scoring/rna/RNA_FullAtomStackingEnergy.hh>
//#include <core/scoring/rna/RNA_PairwiseLowResolutionEnergy.hh>
//#include <core/scoring/rna/RNA_TorsionEnergy.hh>
//#include <core/scoring/rna/RNA_LJ_BaseEnergy.hh>
//#include <core/scoring/rna/RNA_VDW_Energy.hh>
//#include <core/scoring/rna/RG_Energy_RNA.hh>
//#include <core/scoring/methods/RG_Energy_Fast.hh>
//#include <core/scoring/methods/ContactOrderEnergy.hh>
//#include <core/scoring/methods/ReferenceEnergy.hh>
//#include <core/scoring/methods/RMS_Energy.hh>
//#include <core/scoring/methods/SecondaryStructureEnergy.hh>
//#include <core/scoring/methods/SuckerEnergy.hh>
//#include <core/scoring/methods/WaterAdductHBondEnergy.hh>
//#include <core/scoring/methods/WaterAdductIntraEnergy.hh>
//#include <core/scoring/methods/VDW_Energy.hh>
//#include <core/scoring/methods/HybridVDW_Energy.hh>
//#include <core/scoring/methods/PackStatEnergy.hh>
//#include <core/scoring/methods/SurfaceEnergy.hh>
//#include <core/scoring/methods/UnfoldedStateEnergy.hh>
//#include <core/scoring/methods/ResidualDipolarCouplingEnergy.hh>
//#include <core/scoring/methods/ResidualDipolarCouplingEnergy_Rohl.hh>
//#include <core/scoring/packing/HolesEnergy.hh>
//#include <core/scoring/packing/SurfVolEnergy.hh>
//#include <core/scoring/methods/PeptideBondEnergy.hh>

//#include <core/scoring/electron_density/ElecDensEnergy.hh>
//#include <core/scoring/electron_density/ElecDensCenEnergy.hh>
//#include <core/scoring/electron_density/ElecDensAllAtomCenEnergy.hh>

//#include <core/scoring/constraints/ConstraintsEnergy.hh>

//#include <core/scoring/disulfides/FullatomDisulfideEnergy.hh>
//#include <core/scoring/disulfides/CentroidDisulfideEnergy.hh>

//#include <core/scoring/custom_pair_distance/FullatomCustomPairDistanceEnergy.hh>

#include <core/scoring/EnergyMap.hh>
#include <core/scoring/types.hh>
#include <core/scoring/ScoreType.hh>

#include <core/chemical/ChemicalManager.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/io/database/open.hh>


// Utility headers
//#include <utility/vector1.hh>
//#include <utility/pointer/ReferenceCount.hh>
#include <utility/string_util.hh>

namespace core {
namespace scoring {


///////////////////////////////////////////////////////////////////////////////
ScoringManager* ScoringManager::instance_( 0 );

///////////////////////////////////////////////////////////////////////////////
//singleton class
/// SAFE singleton initialization; static from within a function ensures
/// proper load-time behavior
ScoringManager * ScoringManager::get_instance()
{
	if ( instance_ == 0 ) {
		 instance_ = new ScoringManager();
	}
	return instance_;
}


ScoringManager::~ScoringManager() {}

///////////////////////////////////////////////////////////////////////////////
ScoringManager::ScoringManager() :
	pairE_potential_( 0 ),
	rotamer_Library_( 0 ),
	rama_( 0 ),
	rama2b_( 0 ),
	omega_( 0 ),
	env_pair_potential_( 0 ),
	interchain_potential_( 0 ),
	secondary_structure_potential_( 0 ),
	atom_vdw_(),
	rna_atom_vdw_( 0 ),
	occ_hbond_sol_database_( 0 ),
	dna_dr_potential_( 0 ),
	mm_torsion_library_( 0 ),
	mm_bondangle_library_( 0 ),
	DNA_base_potential_( 0 ),
	carbon_hbond_potential_( 0 ),
	rna_low_resolution_potential_( 0 ),
	rna_torsion_potential_( 0 ),
	p_aa_( 0 ),
	water_adduct_hbond_potential_( 0 ),
	gen_born_potential_( 0 ),
	fa_disulfide_potential_( 0 ),
	cen_disulfide_potential_( 0 ),
	membrane_potential_( 0 ),
	unf_state_( 0 ),
	NV_lookup_table_(0),
	method_creator_map_( n_score_types, 0 )
{}

///////////////////////////////////////////////////////////////////////////////
PairEPotential const &
ScoringManager::get_PairEPotential() const
{
	if (pairE_potential_ == 0 )
	{
		pairE_potential_ = new PairEPotential();
	}
	return *pairE_potential_;
}

/// @details The ScoringManager acts as an EnergyMethodFactory.  All EnergyMethods must
/// create a helper class, an EnergyMethodCreator class, that will respond to a call to
/// its create_energy_method by returning a new instance of that EnergyMethod its helping.
/// This Creator class must also register itself with the ScoringManager at load time and
/// hand an instance of itself to the singleton ScoringManager instance.
void
ScoringManager::factory_register( methods::EnergyMethodCreatorOP creator )
{
	ScoreTypes sts = creator->score_types_for_method();
	for ( Size ii = 1; ii <= sts.size(); ++ii ) {
		///std::cout << "Registering " << (int) sts[ ii ] << " to " << creator() << std::endl;
		// make sure no two EnergyMethodCreators lay claim to the same ScoreType
		if ( method_creator_map_[ sts[ ii ] ] != 0 ) {
			utility_exit_with_message( "Cannot register a term to two different EnergyMethodCreators. Term " + utility::to_string( sts[ ii ] ) + " has already been registered!" );
		}
		method_creator_map_[ sts[ ii ] ] = creator;
	}
}


///////////////////////////////////////////////////////////////////////////////
dna::DNA_BasePotential const &
ScoringManager::get_DNA_BasePotential() const
{
	if (DNA_base_potential_ == 0 )
	{
		DNA_base_potential_ = new dna::DNA_BasePotential();
	}
	return *DNA_base_potential_;
}

///////////////////////////////////////////////////////////////////////////////
EnvPairPotential const &
ScoringManager::get_EnvPairPotential() const
{
	if (env_pair_potential_ == 0 )
	{
		env_pair_potential_ = new EnvPairPotential();
	}
	return *env_pair_potential_;
}

///////////////////////////////////////////////////////////////////////////////
MembranePotential const &
ScoringManager::get_MembranePotential() const
{
	if (membrane_potential_ == 0 )
	{
		membrane_potential_ = new MembranePotential();
	}
	return *membrane_potential_;
}

///////////////////////////////////////////////////////////////////////////////
InterchainPotential const &
ScoringManager::get_InterchainPotential() const
{
	if (interchain_potential_ == 0 )
	{
		interchain_potential_ = new InterchainPotential();
	}
	return *interchain_potential_;
}

///////////////////////////////////////////////////////////////////////////////
SecondaryStructurePotential const &
ScoringManager::get_SecondaryStructurePotential() const
{
	if (secondary_structure_potential_ == 0 )
	{
		secondary_structure_potential_ = new SecondaryStructurePotential();
	}
	return *secondary_structure_potential_;
}

///////////////////////////////////////////////////////////////////////////////
GenBornPotential const &
ScoringManager::get_GenBornPotential() const
{
	if (gen_born_potential_ == 0 )
	{
		gen_born_potential_ = new GenBornPotential();
	}
	return *gen_born_potential_;
}

///////////////////////////////////////////////////////////////////////////////
AtomVDW const &
ScoringManager::get_AtomVDW( std::string const & atom_type_set_name ) const
{
	if ( atom_vdw_.count( atom_type_set_name ) == 0 ) {
		atom_vdw_[ atom_type_set_name ] = new AtomVDW( atom_type_set_name );
	}
	return * ( atom_vdw_[ atom_type_set_name ] );
}

///////////////////////////////////////////////////////////////////////////////
carbon_hbonds::CarbonHBondPotential const &
ScoringManager::get_CarbonHBondPotential() const
{
	if (carbon_hbond_potential_ == 0 )
	{
		carbon_hbond_potential_ = new carbon_hbonds::CarbonHBondPotential();
	}
	return *carbon_hbond_potential_;
}

///////////////////////////////////////////////////////////////////////////////
rna::RNA_AtomVDW const &
ScoringManager::get_RNA_AtomVDW() const
{
	if (rna_atom_vdw_ == 0 )
	{
		rna_atom_vdw_ = new rna::RNA_AtomVDW();
	}
	return *rna_atom_vdw_;
}

///////////////////////////////////////////////////////////////////////////////
geometric_solvation::DatabaseOccSolEne const &
ScoringManager::get_DatabaseOccSolEne( std::string const & atom_type_set_name, Real const & min_occ_energy ) const
{
	if (occ_hbond_sol_database_ == 0 )
	{
		occ_hbond_sol_database_ = new geometric_solvation::DatabaseOccSolEne( atom_type_set_name, min_occ_energy );
	}
	return *occ_hbond_sol_database_;
}

///////////////////////////////////////////////////////////////////////////////
rna::RNA_LowResolutionPotential const &
ScoringManager::get_RNA_LowResolutionPotential() const
{
	if (rna_low_resolution_potential_ == 0 )
	{
		rna_low_resolution_potential_ = new rna::RNA_LowResolutionPotential();
	}
	return *rna_low_resolution_potential_;
}

///////////////////////////////////////////////////////////////////////////////
rna::RNA_TorsionPotential const &
ScoringManager::get_RNA_TorsionPotential() const
{
	if (rna_torsion_potential_ == 0 )
	{
		rna_torsion_potential_ = new rna::RNA_TorsionPotential();
	}
	return *rna_torsion_potential_;
}

///////////////////////////////////////////////////////////////////////////////
dna::DirectReadoutPotential const &
ScoringManager::get_DirectReadoutPotential() const
{
	if (dna_dr_potential_ == 0 )
	{
		dna_dr_potential_ = new dna::DirectReadoutPotential();
	}
	return *dna_dr_potential_;
}

P_AA const &
ScoringManager::get_P_AA() const
{
	if ( p_aa_ == 0 ) {
		p_aa_ = new P_AA;
	}
	return *p_aa_;
}

WaterAdductHBondPotential const &
ScoringManager::get_WaterAdductHBondPotential() const
{
	if ( water_adduct_hbond_potential_ == 0 ) {
		water_adduct_hbond_potential_ = new WaterAdductHBondPotential;
	}
	return *water_adduct_hbond_potential_;
}


/////////////////////////////////////
//////////////////////////////
////////////
ScoringManager::RotamerLibrary &
ScoringManager::get_RotamerLibrary() const
{
	if (rotamer_Library_ == 0 )
	{
		rotamer_Library_ = new RotamerLibrary();
		read_dunbrack_library( *rotamer_Library_ );
	}
	return *rotamer_Library_;
}

///////////////////////////////////////////////////////////////////////////////
Ramachandran const &
ScoringManager::get_Ramachandran() const
{
	if ( rama_ == 0 )
	{
		rama_ =  new Ramachandran;
	}
	return *rama_;
}

///////////////////////////////////////////////////////////////////////////////
Ramachandran2B const &
ScoringManager::get_Ramachandran2B() const
{
	if ( rama2b_ == 0 )
	{
		rama2b_ =  new Ramachandran2B;
	}
	return *rama2b_;
}

///////////////////////////////////////////////////////////////////////////////
OmegaTether const &
ScoringManager::get_OmegaTether() const
{
	if ( omega_ == 0 )
	{
		omega_ =  new OmegaTether();
	}
	return *omega_;
}


///////////////////////////////////////////////////////////////////////////////
mm::MMTorsionLibrary const &
ScoringManager::get_MMTorsionLibrary() const
{
	if ( mm_torsion_library_ == 0 )
		{
			mm_torsion_library_ = new MMTorsionLibrary
				( io::database::full_name( "chemical/mm_atom_type_sets/fa_standard/mm_torsion_params.txt" ),
					chemical::ChemicalManager::get_instance()->mm_atom_type_set( chemical::FA_STANDARD ) );
		}
	return *mm_torsion_library_;
}

disulfides::FullatomDisulfidePotential &
ScoringManager::get_FullatomDisulfidePotential() const
{
	if ( fa_disulfide_potential_ == 0 ) {
		fa_disulfide_potential_ = new disulfides::FullatomDisulfidePotential;
	}
	return *fa_disulfide_potential_;
}

disulfides::CentroidDisulfidePotential &
ScoringManager::get_CentroidDisulfidePotential() const
{
	if ( cen_disulfide_potential_ == 0 ) {
		cen_disulfide_potential_ = new disulfides::CentroidDisulfidePotential;
	}
	return *cen_disulfide_potential_;
}

///////////////////////////////////////////////////////////////////////////////


NV::NVlookup const &
ScoringManager::get_NVLookupTable() const
{
	if(NV_lookup_table_ == 0)
	{
		using namespace core::options;
		using namespace core::options::OptionKeys;
		NV_lookup_table_ = new NV::NVlookup(io::database::full_name(option[score::NV_table]()));
		//NV_lookup_table_ = new nv::NVlookup(io::database::full_name("neighbor_vector_score.histogram"));
	}
	return *NV_lookup_table_;
}

///////////////////////////////////////////////////////////////////////////////
mm::MMBondAngleLibrary const &
ScoringManager::get_MMBondAngleLibrary() const
{
	if ( mm_bondangle_library_ == 0 )
		{
			mm_bondangle_library_ = new MMBondAngleLibrary
				( io::database::full_name( "chemical/mm_atom_type_sets/fa_standard/par_all27_prot_na.prm" ),
					chemical::ChemicalManager::get_instance()->mm_atom_type_set( chemical::FA_STANDARD ) );
		}
	return *mm_bondangle_library_;
}

///////////////////////////////////////////////////////////////////////////////
UnfoldedStatePotential const &
ScoringManager::get_UnfoldedStatePotential() const
{
	if ( unf_state_ == 0 ) {
		unf_state_ = new UnfoldedStatePotential();
	}
	return *unf_state_;
}

///////////////////////////////////////////////////////////////////////////////
void
ScoringManager::add_etable( std::string const & name, etable::EtableOP etable )
{
	assert( etables_.count(name) == 0 );
	etables_[ name ] = etable;
}

///////////////////////////////////////////////////////////////////////////////
void
ScoringManager::add_coarse_etable( std::string const &name, coarse::CoarseEtableOP etable )
{
	assert( coarse_etables_.count(name) == 0);
	coarse_etables_ [name] = etable;
}

///////////////////////////////////////////////////////////////////////////////
etable::EtableCAP
ScoringManager::etable( std::string const & table_id ) const
{
	using namespace etable;

	if ( etables_.find( table_id ) == etables_.end() ) {
		// try to build if possible
		if ( table_id == FA_STANDARD_DEFAULT ) {
			EtableOP etable_ptr
				( new Etable( chemical::ChemicalManager::get_instance()->atom_type_set( chemical::FA_STANDARD ),
											EtableOptions() ) );
			etables_[ table_id ] = etable_ptr;
		} else if ( table_id == FA_STANDARD_SOFT ) {
			// soft rep etable: modified radii and also change to lj_switch_dis2sigma
			EtableOptions options;
			options.lj_switch_dis2sigma = 0.91;
			EtableOP etable_ptr
				( new Etable( chemical::ChemicalManager::get_instance()->atom_type_set( chemical::FA_STANDARD ),
											options, "SOFT" ) );
			etables_[ table_id ] = etable_ptr;

		} else if ( table_id.substr(0, FA_STANDARD_DEFAULT.size() + 1 ) == FA_STANDARD_DEFAULT+"_" ) {
			// note we check for soft rep 1st since that would match this as well -- confusing??
			// apply a modification of the radii/wdepths
			std::string const alternate_parameters( table_id.substr( FA_STANDARD_DEFAULT.size() + 1 ) );
			EtableOP etable_ptr
				( new Etable( chemical::ChemicalManager::get_instance()->atom_type_set( chemical::FA_STANDARD ),
											EtableOptions(), alternate_parameters ) );
			etables_[ table_id ] = etable_ptr;

		} else {
			std::cout << "1:" << table_id.substr(0, FA_STANDARD_DEFAULT.size() + 1 ) << '\n' <<
				"2:" << FA_STANDARD_DEFAULT+"_" << std::endl;
			utility_exit_with_message("unrecognized etable: "+table_id );
		}
	}
	return (etables_.find( table_id )->second)();
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
coarse::CoarseEtableCAP
ScoringManager::coarse_etable( std::string const & table_id ) const
{
	using namespace etable;
	if ( coarse_etables_.find( table_id ) == coarse_etables_.end() ) {
		// try to build if possible
		if ( table_id == chemical::COARSE_TWO_BEAD ) {
			coarse::CoarseEtableOP etable_ptr
				( new coarse::CoarseEtable(
							chemical::ChemicalManager::get_instance()->
							atom_type_set( chemical::COARSE_TWO_BEAD ), chemical::COARSE_TWO_BEAD )
				);
			coarse_etables_[ table_id ] = etable_ptr;
		} else {
			std::string msg = "unrecognized etable: "+table_id;
			utility_exit_with_message( msg );
		}
	}
	return (coarse_etables_.find( table_id )->second)();
}


///////////////////////////////////////////////////////////////////////////////

/// @details When a ScoreFunction the weight for a particular ScoreType set from 0
/// to some non-zero value, it will request an instance of the EnergyMethod class
/// that is responsible for calculating that ScoreType.  The ScoringManager responds
/// to that request by asking the EnergyMethodCreator that has claimed responsibility
/// for this ScoreType for a new instance.  EnergyMethodCreators must first have
/// registered themselves with the ScoringManager.  This should have been done at
/// load time, using a static-variable-initialization function call.
/// See src/core/scoring/etable/EtableEnergy.cc for an example of how the
/// EtableEnergyCreator class registers itself with the ScoringManager.
methods::EnergyMethodOP
ScoringManager::energy_method(
	ScoreType const & score_type,
	methods::EnergyMethodOptions const & options
) const
{
	if ( score_type > n_score_types ) {
		/// Inactive score type requested.  The program must be recompiled such that the desired score type
		/// appears before the n_score_types element in the ScoreType enumeration.  The program must now quit
		/// or it will later produce a segmentation fault when the EnergyMethod responsible for this term
		/// attempts to write into an EnergyMap object at a position that the EnergyMap has not allocated space
		/// for.
		std::cerr << "Critical error in ScoringManager::energy_method().\nRequested an inactive score_type '" << score_type;
		std::cerr << "' defined at position " << (int) score_type << " in the ScoreType enumeration.\n";
		std::cerr << "Active score types must appear before the n_score_types element ";
		std::cerr << "(at position " << (int) n_score_types << ") as this element marks the end of the active score types.\n";
		std::cerr << "Rosetta must be recompiled after src/core/scoring/ScoreType.hh is modified to include " << score_type;
		std::cerr << " as an active score type." << std::endl;
		utility_exit_with_message( "ERROR: Attempted to use an inactive score type" );
	}

	if ( score_type == python ) return 0; /// python special case; this could now be changed...

	if ( method_creator_map_[ score_type ] == 0 ) {
		utility_exit_with_message( "Requested ScoreType " + utility::to_string( score_type ) + " does not have a registered EnergyMethodCreator." );
	}

	return method_creator_map_[ score_type ]->create_energy_method( options );

	/* OLD WAY
	using namespace methods;
	switch( score_type ) {
	case fa_atr:
	case fa_rep:
	case fa_sol:
	case fa_intra_atr:
	case fa_intra_rep:
	case fa_intra_sol:
		return new etable::EtableEnergy( *etable( options.etable_type() ), options );
	case lk_hack:
		return new methods::LK_hack( *etable( options.etable_type() ));
	case lk_costheta:
	case lk_polar:
	case lk_nonpolar:
		return new methods::LK_CosThetaEnergy( *etable( options.etable_type() ));
	case coarse_fa_atr:
	case coarse_fa_rep:
	case coarse_fa_sol:
	case coarse_beadlj:
		return new etable::CoarseEtableEnergy( *etable( options.etable_type() ), options,
			coarse_etable( chemical::COARSE_TWO_BEAD ) );
	case hbond_lr_bb:
	case hbond_sr_bb:
	case hbond_bb_sc:
	case hbond_sr_bb_sc:
	case hbond_lr_bb_sc:
	case hbond_sc:
		return new hbonds::HBondEnergy( options );
	case ch_bond:
	case ch_bond_sc_sc:
	case ch_bond_bb_sc:
	case ch_bond_bb_bb:
		return new hbonds::CarbonHBondEnergy();
	case geom_sol:
		return new hbonds::GeometricSolEnergy( options );
	case occ_sol_pw:
		return new geometric_solvation::OccludedHbondSolEnergy( options );
	case occ_sol_exact:
		return new geometric_solvation::ExactOccludedHbondSolEnergy(
			options::option[options::OptionKeys::score::exact_occ_pairwise],
			options::option[options::OptionKeys::score::exact_occ_split_between_res] );
	case dslf_ss_dst:
	case dslf_cs_ang:
	case dslf_ss_dih:
	case dslf_ca_dih:
	case dslf_cbs_ds:
		return new disulfides::FullatomDisulfideEnergy( get_FullatomDisulfidePotential() );
	case dslfc_cen_dst:
	case dslfc_cb_dst:
	case dslfc_ang:
	case dslfc_cb_dih:
	case dslfc_bb_dih:
		return new disulfides::CentroidDisulfideEnergy( get_CentroidDisulfidePotential() );
	case neigh_count:
	case neigh_vect:
	case neigh_vect_raw:
		return new nv::NVscore();
	case symE_bonus:
		return new symDesign::symE();
	case gb_elec:
		return new methods::GenBornEnergy( options );
	case fa_pair:
	case fa_pair_aro_aro:
	case fa_pair_aro_pol:
	case fa_pair_pol_pol:
		return new PairEnergy;
	case fa_dun:
		return new DunbrackEnergy;
	case p_aa_pp:
		return new P_AA_pp_Energy;
	case pro_close:
		return new ProClosureEnergy;
	case h2o_intra:
		return new WaterAdductIntraEnergy;
	case h2o_hbond:
		return new WaterAdductHBondEnergy;
	case ref:
		if ( options.has_method_weights( ref ) ) {
			return new ReferenceEnergy( options.method_weights( ref ) );
		} else {
			return new ReferenceEnergy;
		}
	case rama:
		return new RamachandranEnergy;
	case rama2b:
		return new RamachandranEnergy2B;
	case omega:
		return new OmegaTetherEnergy;
	case chainbreak:
		return new ChainbreakEnergy;
	case linear_chainbreak:
	case overlap_chainbreak:
		return new LinearChainbreakEnergy;
	case distance_chainbreak:
		return new DistanceChainbreakEnergy;
	case envsmooth:
		return new EnvSmoothEnergy;
	case env:
	case cbeta:
		return new EnvEnergy;
	case pair:
	case cenpack:
		return new CenPairEnergy;
	case Menv:
			return new MembraneEnvEnergy;
	case Menv_hbond:
	case Menv_termini:
	case Menv_tm_proj:
		return new MembraneEnvPenalties;
	case Mcbeta:
			return new MembraneCbetaEnergy;
	case Mpair:
		return new MembraneCenPairEnergy;
	case Mlipo:
		return new MembraneLipo;
	case interchain_env:
	case interchain_contact:
		return new InterchainEnvEnergy;
	case interchain_pair:
	case interchain_vdw:
		return new InterchainPairEnergy;
	case hs_pair:
	case ss_pair:
	case rsigma:
	case sheet:
		return new SecondaryStructureEnergy;
	case rdc:
		return new ResidualDipolarCouplingEnergy;
	case rdc_rohl:
		return new ResidualDipolarCouplingEnergy_Rohl;
	case holes:
	case holes_decoy:
	case holes_resl:
	case holes_min:
		return new packing::HolesEnergy;
	case dab_sasa:
	case dab_sev:
		return new packing::SurfVolEnergy;
	case vdw:
		return new VDW_Energy( options );
	case hybrid_vdw:
		return new HybridVDW_Energy();
	case rna_rg:
		return new rna::RG_Energy_RNA;
	case rna_vdw:
		return new rna::RNA_VDW_Energy;
	case rna_base_pair:
	case rna_base_axis:
	case rna_base_stagger:
	case rna_base_stack:
	case rna_base_stack_axis:
	case rna_base_pair_pairwise:
	case rna_base_axis_pairwise:
	case rna_base_stagger_pairwise:
	case rna_base_stack_pairwise:
	case rna_base_stack_axis_pairwise:
	case rna_base_backbone:
	case rna_backbone_backbone:
	case rna_repulsive:
		return new rna::RNA_PairwiseLowResolutionEnergy;
	case fa_stack:
		return new rna::RNA_FullAtomStackingEnergy;
	case rna_torsion:
	case rna_sugar_close:
		return new rna::RNA_TorsionEnergy;
	case rna_fa_atr_base:
	case rna_fa_rep_base:
		return new rna::RNA_LJ_BaseEnergy( *etable( options.etable_type() ) );
	case rg:
		return new RG_Energy_Fast;
	case co:
		return new ContactOrderEnergy;
	case rms:
		return new RMS_Energy;
	case mm_twist:
		return new MMTorsionEnergy;
	case mm_bend:
		return new MMBondAngleEnergy( options );
//	case csd_torsion:
//		return new CSD_TorsionEnergy();
	case fa_cust_pair_dist:
		return new custom_pair_distance::FullatomCustomPairDistanceEnergy;
	case python:
		return NULL;
	case hack_elec:
	case hack_elec_bb_bb:
	case hack_elec_bb_sc:
	case hack_elec_sc_sc:
		return new hackelec::HackElecEnergy( options );
	case hack_elec_rna_phos_phos:
	case hack_elec_rna_phos_sugr:
	case hack_elec_rna_phos_base:
	case hack_elec_rna_sugr_sugr:
	case hack_elec_rna_sugr_base:
	case hack_elec_rna_base_base:
		return new hackelec::RNAHackElecEnergy( options );
	case dna_bp:
	case dna_bs:
		return new DNA_BaseEnergy;
	case dna_dr:
		return new DirectReadoutEnergy;
	case atom_pair_constraint:
	case angle_constraint:
	case dihedral_constraint:
	case constant_constraint:
	case coordinate_constraint:
	case dof_constraint:
	case res_type_constraint:
	case backbone_stub_constraint:
	case big_bin_constraint:
	case rna_bond_geometry:
		return new constraints::ConstraintsEnergy;
	case suck:
		return new SuckerEnergy;
	case pack_stat:
		return new PackStatEnergy;
	case surface:
		return new SurfaceEnergy;
	case p_aa:
		return new P_AA_Energy;
	case unfolded:
		if ( options.has_method_weights( unfolded ) ) {
			return new UnfoldedStateEnergy( options.method_weights( unfolded ) );
		} else {
			return new UnfoldedStateEnergy;
		}
	case elec_dens_window:
		return new ElecDensEnergy;
	case elec_dens_whole_structure_ca:
		return new ElecDensCenEnergy;
	case elec_dens_whole_structure_allatom:
		return new ElecDensAllAtomCenEnergy;
	case peptide_bond:
		return new PeptideBondEnergy;
	case pcs:
		return new PCS::PCS_Energy;
	default:
		utility_exit_with_message(
			"no energymethod for this score_type " + name_from_score_type( score_type )
		);
	}
	return 0;
	*/
}

/// global etable_id
std::string const FA_STANDARD_DEFAULT( "FA_STANDARD_DEFAULT" );
std::string const FA_STANDARD_SOFT   ( "FA_STANDARD_SOFT" );


} // namespace core
} // namespace scoring

