// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   src/core/scoring/SurfacePotential.cc
/// @brief  Class which keeps reads the residue hydrophobic ASA database file and calculates surface residue energies.
/// @author Ron Jacak

#include <core/io/database/open.hh>
#include <core/scoring/SurfacePotential.hh>
#include <core/types.hh>
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/TenANeighborGraph.hh>

// ObjexxFCL Headers

// Numeric Headers

// Utility Headers
#include <core/util/Tracer.hh>
#include <utility/io/izstream.hh>

// C++ Headers

namespace core {
namespace scoring {

static core::util::Tracer TR("core.scoring.SurfacePotential");

const core::Size SurfacePotential::MAX_PATCH_SURFACE_AREA = 800;
const core::Real SurfacePotential::MAX_SURFACE_ENERGY = 50.0;
const core::Size SurfacePotential::SURFACE_EXPOSED_CUTOFF = 16;
const core::Real SurfacePotential::INTERACTION_RADIUS = 8.0;

/// @brief set initial value as no instance
SurfacePotential* SurfacePotential::instance_( 0 );


/// @brief static function to get the instance of (pointer to) this singleton class
SurfacePotential* SurfacePotential::get_instance() {
	if ( instance_ == 0 )
		instance_ = new SurfacePotential();
	return instance_;
}

/// @brief private constructor to guarantee the singleton
SurfacePotential::SurfacePotential() {
	read_average_res_hASA_database_file();
	read_hASA_score_database_file();
}


/// @brief Reads in the database file which contains average residue hydrophobic accessible surface areas.
void SurfacePotential::read_average_res_hASA_database_file() {

	utility::io::izstream residue_hASA_ifstream;
	core::io::database::open( residue_hASA_ifstream, "average_hASA_by_res_and_neighbor.txt" );

	std::string str_restype;
	Real lte10_asa = 0.0;
	Real lte13_asa = 0.0;
	Real lte16_asa = 0.0;

	res_to_average_hASA_.resize( chemical::num_canonical_aas );

	while ( !residue_hASA_ifstream.eof() ) {
		residue_hASA_ifstream >> str_restype >> lte10_asa >> lte13_asa >> lte16_asa;

		// store the mean hASA values in a vector for faster lookup
		// This way, rather than constructing a key each time we need to get the information on a particular AA type
		// we can just index right into the vector. We waste some memory on storing the same values in the vector
		// but this class is a Singleton and it's only 48 floats and the size of the vectors.
		chemical::AA aa_type = chemical::aa_from_name( str_restype );
		res_to_average_hASA_[ aa_type ].resize( 16, 0.0 );

		for ( Size ii=1; ii <= 10; ++ii ) {  // for nbs 1-10
			res_to_average_hASA_[ aa_type ][ ii ] = lte10_asa;
		}
		for ( Size ii=11; ii <= 13; ++ii ) { // for nbs 11-13
			res_to_average_hASA_[ aa_type ][ ii ] = lte13_asa;
		}
		for ( Size ii=14; ii <= 16; ++ii ) {  // for nbs 14-16
			res_to_average_hASA_[ aa_type ][ ii ] = lte16_asa;
		}

	}

#ifndef NDEBUG
	// quick check to make sure we stored the right things
	for ( Size ii=1; ii <= chemical::num_canonical_aas; ++ii ) {
		TR << chemical::name_from_aa( (chemical::AA)ii ) << ": [ ";
		for ( Size jj=1; jj <= res_to_average_hASA_[ ii ].size(); ++jj ) {
			TR << res_to_average_hASA_[ ii ][ jj ] << ", ";
		}
		TR << "]" << std::endl;
	}
#endif

}


/// @brief Reads in the database file which contains the scores for a distribution of patch sizes.
void SurfacePotential::read_hASA_score_database_file() {

	utility::io::izstream hASA_score_ifstream;
	core::io::database::open( hASA_score_ifstream, "hASA_score.txt" );

	Real amount_hASA = 0.0;
	Real energy = 0.0;

	hASA_to_energy_.resize( 33, 0.0 );

	while ( !hASA_score_ifstream.eof() ) {
		hASA_score_ifstream >> amount_hASA >> energy;

		Size index = (Size)(amount_hASA / 25);  // should only be the whole part, not the remainder
		hASA_to_energy_[ index ] = energy;
	}

#ifndef NDEBUG
	TR << "patch energies: [ ";
	for ( Size ii=0; ii < hASA_to_energy_.size(); ++ii ) {
		TR << hASA_to_energy_[ ii ] << ", ";
	}
	TR << "]" << std::endl;
#endif
}


///
/// @begin SurfacePotential::average_residue_hASA
///
/// @brief
/// Returns the average surface energy for the given residue type and number of neighbors.
///
Real SurfacePotential::average_residue_hASA( chemical::AA aa_type, Size num_nbs ) {

	runtime_assert( num_nbs <= SURFACE_EXPOSED_CUTOFF );
	return res_to_average_hASA_[ aa_type ][ num_nbs ];

}

///
/// @begin SurfacePotential::hASA_patch_energy
///
/// @brief
/// Returns the energy for a given patch size.  The calling function must ensure that an out-of-bounds error will not occur.
///
Real SurfacePotential::hASA_patch_energy( Real patch_area ) {

	runtime_assert( patch_area <= MAX_PATCH_SURFACE_AREA );
	return hASA_to_energy_[ (Size) (patch_area / 25) ];
}


///
/// @begin SurfacePotential::compute_residue_surface_energy
///
/// @brief
/// Calculates the surface energy for a single residue within a Pose object. Used only by the RotamerSet_::compute_one_body_energy_maps
/// function (which, in turn, is only used by the optE protocol).  Nowhere else in mini is this function used.
///
void SurfacePotential::compute_residue_surface_energy( conformation::Residue const & rsd, pose::Pose const & pose, EnergyMap & emap,
	Size resid, utility::vector1< Size > num_neighbors_ ) {

	// our definition of surface residue is that the residue has fewer than 16 neighbors
	if ( !( pose.energies().residue_neighbors_updated() ) || num_neighbors_[ resid ] > SURFACE_EXPOSED_CUTOFF ) {
		emap[ surface ] = 0.0;
		return;
	}

	// check if the current rotamer is a hydrophobic one. if it is, add its hASA to the total since we want to count all the hydrophobics
	Real total_hASA = 0.0;

	if ( !(rsd.is_polar()) ) {
		total_hASA += average_residue_hASA( rsd.aa(), num_neighbors_[ resid ] );
#ifndef NDEBUG
		TR << "Residue is hydrophobic. Adding self hASA " << average_residue_hASA( rsd.aa(), num_neighbors_[ resid ] )
			<< " to total. aa type: " << rsd.aa() << ", num_nbs: " << num_neighbors_[ resid ] << std::endl;
#endif
	}

	// now add the hASA of all se hp residues within 8A (nbr_atom - nbr_atom) distance. Use the nbr_atom coordinates of the passed
	// in (being considered) rotamer, not of the wild type sequence rotamer.  In a small percent of the cases, using the wild type
	// nbr_atom will give a different count than when using the new rotamer nbr_atom position.
	Real distanceBetweenAtoms = 0.0;
	for ( Size res2_position = 1; res2_position < pose.total_residue(); ++res2_position ) {

		if ( resid == res2_position ) { continue; }
		conformation::Residue const & rsd2 = pose.residue( res2_position );

		distanceBetweenAtoms = rsd.xyz( rsd.nbr_atom() ).distance( rsd2.xyz( rsd2.nbr_atom() ) );

		// first, have to check again if these two residues are neighbors
		if ( distanceBetweenAtoms <= INTERACTION_RADIUS ) {

			// now we can check whether that nb'ing residue is nonpolar and exposed
			if ( !(pose.residue( res2_position ).is_polar()) ) {

				// ok, so it's hydrophobic, but is it surface-exposed, too?
				//if ( tenA_neighbor_graph.get_node( res2_position )->num_neighbors_counting_self() >= SURFACE_EXPOSED_CUTOFF ) {
				if ( num_neighbors_[ res2_position ] > SURFACE_EXPOSED_CUTOFF ) {
					continue;
				}
				// passed all checks
				total_hASA += average_residue_hASA( rsd2.aa(), num_neighbors_[ res2_position ] );
#ifndef NDEBUG
				TR << "Found se hp neighbor. Adding hASA " << average_residue_hASA( rsd2.aa(), num_neighbors_[ res2_position ] )
					<< " to total. aa type: " << rsd2.aa() << ", num_nbs: " << num_neighbors_[ res2_position ] << std::endl;
#endif
			}
		}
	}

	// now that we know how many surface-exposed, hphobic neighbors res1 has, get the surface energy for that value
	if ( total_hASA > MAX_PATCH_SURFACE_AREA ) {
		emap[ surface ] = MAX_SURFACE_ENERGY;
	} else {
		emap[ surface ] = hASA_patch_energy( total_hASA );
	}

#ifndef NDEBUG
	TR << "compute_residue_surface_energy: calculated surface energy: " << emap[ surface ] << ", total_hASA: "
		<< total_hASA << std::endl;
#endif

}


///
///@begin compute_pose_surface_energy
///
///@brief helper method for computing surface score. in the optE protocol we don't care about the total vs. residue
/// level surface scores. so just call that function but discard the values for those variables.
///
void SurfacePotential::compute_pose_surface_energy( pose::Pose const & pose, Real & surface_energy_ ) {

	utility::vector1< Size > num_neighbors_;
	utility::vector1< Real > res_level_energies_;

	compute_pose_surface_energy( pose, surface_energy_, res_level_energies_ );

	return;
}

void SurfacePotential::compute_pose_surface_energy( pose::Pose const & pose, Real & total_surface_energy_,
	utility::vector1< Real > & residue_surface_energy_ ) {

	// the pose has to have been scored at this point for this method to work.  since I can't force a score eval here,
	// return 0.0 for everything if it hasn't been.
	// if the size of the pose is really small, also return zeros
	if ( !( pose.energies().residue_neighbors_updated() ) || ( pose.n_residue() <= 4 ) ) {
		total_surface_energy_ = 0.0;
		residue_surface_energy_.clear();
		return;
	}

	// resize the per-residue surface energy vector
	residue_surface_energy_.clear();
	residue_surface_energy_.resize( pose.n_residue(), 0.0 );

	utility::vector1< Size > num_neighbors_( pose.n_residue(), 0 );

	// first, we need to init the num neighbors array (either using the tenA nb graph or by counting manually)
	for ( core::Size res1_position = 1; res1_position <= pose.n_residue(); ++res1_position ) {

		core::scoring::TenANeighborGraph const & tenA_neighbor_graph( pose.energies().tenA_neighbor_graph() );

		// set the number of neighbors vector for later output
		num_neighbors_[ res1_position ] = tenA_neighbor_graph.get_node( res1_position )->num_neighbors_counting_self();

		{ //  manual way of counting neighbors

			//conformation::Residue const & rsd1 = pose.residue( res1_position );
			//float distanceBetweenAtoms = 0.0;
			//int num_neighbors_counting_self = 0;

			//for ( Size ii=1; ii < pose.total_residue(); ++ii ) {
			//	if ( ii == res1_position ) { continue; }
			//	conformation::Residue const & rsd2 = pose.residue( ii );

			//	distanceBetweenAtoms = rsd1.xyz( rsd1.nbr_atom() ).distance( rsd2.xyz( rsd2.nbr_atom() ) );
			//	if ( distanceBetweenAtoms <= INTERACTION_RADIUS ) {
			//		num_neighbors_counting_self++;
			//	}
			//}

			// finally, count ourselves...
			//num_neighbors_counting_self++;
			//num_neighbors_[ res1_position ] = num_neighbors_counting_self;
		}
	}

	// now, we have to loop over all residues and find exposed hydrophobics (and we can use the num neighbors array
	// to determine which ones are surface exposed)

	for ( core::Size res1_position = 1; res1_position <= pose.n_residue(); ++res1_position ) {

		// reset the counter
		Real total_hASA = 0.0;

		// our definition of surface residue is that the residue has fewer than 16 neighbors
		if ( num_neighbors_[ res1_position ] > SURFACE_EXPOSED_CUTOFF ) {
			continue;
		}

		// passed the surface-exposed check...

		// check if this residue is hydrophobic
		if ( !(pose.residue( res1_position ).is_polar()) ) {
			total_hASA += average_residue_hASA( pose.residue( res1_position ).aa(), num_neighbors_[ res1_position ] );
#ifndef NDEBUG
			TR << "compute_pose_surface_energy: residue is hydrophobic. Adding self hASA "
				<< average_residue_hASA( pose.residue( res1_position ).aa(), num_neighbors_[ res1_position ] )
				<< " to total. aa type: " << pose.residue( res1_position ).aa() << ", num_nbs: " << num_neighbors_[ res1_position ] << std::endl;
#endif
		}

		//TR << "Neighbors of residue " << pose.residue( res1_position ).name3() << " " << res1_position << " include " << std::endl;

		// for every Edge in the neighbor graph, figure out if that residue is surface exposed *and* hydrophobic
		//for ( core::graph::EdgeListConstIterator eli = tenA_neighbor_graph.get_node( res1_position )->const_edge_list_begin(),
		//	eli_end = tenA_neighbor_graph.get_node( res1_position )->const_edge_list_end(); eli != eli_end; ++eli ) {

			// save the value to simplify code ahead
			//int res2_position = (*eli)->get_other_ind( res1_position );

			// get the other node for this edge, so pass in the res1 node to this method
			//TR << pose.residue( res2_position ).name3() << " " << res2_position << std::endl;

		conformation::Residue const & rsd1 = pose.residue( res1_position );
		Real distanceBetweenAtoms = 0.0;

		for ( Size res2_position = 1; res2_position < pose.total_residue(); ++res2_position ) {

			if ( res2_position == res1_position ) { continue; }
			conformation::Residue const & rsd2 = pose.residue( res2_position );

			distanceBetweenAtoms = rsd1.xyz( rsd1.nbr_atom() ).distance( rsd2.xyz( rsd2.nbr_atom() ) );

			// first, have to check again if these two residues are neighbors
			if ( distanceBetweenAtoms <= INTERACTION_RADIUS ) {

				// now we can check whether that nb'ing residue is nonpolar and exposed
				if ( !(pose.residue( res2_position ).is_polar()) ) {

					// ok, so it's hydrophobic, but is it surface-exposed, too?
					//if ( tenA_neighbor_graph.get_node( res2_position )->num_neighbors_counting_self() >= SURFACE_EXPOSED_CUTOFF ) {
					if ( num_neighbors_[ res2_position ] > SURFACE_EXPOSED_CUTOFF ) {
						continue;
					}
					// passed all checks
					total_hASA += average_residue_hASA( rsd2.aa(), num_neighbors_[ res2_position ] );
#ifndef NDEBUG
					TR << "Found se hp neighbor. Adding hASA " << average_residue_hASA( rsd2.aa(), num_neighbors_[ res2_position ] )
						<< " to total. aa type: " << rsd2.aa() << ", num_nbs: " << num_neighbors_[ res2_position ] << std::endl;
#endif
				}
			}
		}

		// now that we know how many surface-exposed, hphobic neighbors res1 has, get the surface energy for that value
		if ( total_hASA > MAX_PATCH_SURFACE_AREA ) {
			residue_surface_energy_[ res1_position ] = MAX_SURFACE_ENERGY;
		} else {
			residue_surface_energy_[ res1_position ] = hASA_patch_energy( total_hASA );
		}

#ifndef NDEBUG
		TR << "compute_pose_surface_energy: calculated residue surface energy: " << residue_surface_energy_[ res1_position ]
			<< ", total_hASA: " << total_hASA << std::endl;
#endif

	} // end second res1 loop


	total_surface_energy_ = 0.0;
	for ( Size ii=1; ii < residue_surface_energy_.size(); ++ii ) {
		total_surface_energy_ += residue_surface_energy_[ii];
	}

#ifndef NDEBUG
	TR << "compute_pose_surface_energy: calculated surface energy: " << total_surface_energy_ << ", residue_surfaceE: [ ";
	for ( Size ii=1; ii <= residue_surface_energy_.size(); ++ii ) {
		TR << residue_surface_energy_[ ii ] << ", ";
	}
	TR << "]" << std::endl;
#endif

}


} // namespace scoring
} // namespace core

