// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/UnfoldedStatePotential.cc
/// @brief  Unfolded state energies based on energies of residues in fragments, definition file
/// @author Ron Jacak (ronj@email.unc.edu)

// Unit headers
#include <core/scoring/UnfoldedStatePotential.hh>

// Project headers
#include <core/io/database/open.hh>
#include <core/chemical/AA.hh>
#include <core/conformation/Residue.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/pose/Pose.hh>

// Numeric headers

// ObjexxFCL headers
#include <ObjexxFCL/formatted.i.hh>

// Utility headers
#include <utility/io/izstream.hh>

// C++ headers
#include <iostream>


namespace core {
namespace scoring {


UnfoldedStatePotential::UnfoldedStatePotential() {
	read_database_file();
}

UnfoldedStatePotential::~UnfoldedStatePotential() {}


void
UnfoldedStatePotential::read_database_file() {

	using namespace core::chemical;
	using namespace core::scoring;

	// Read the energies file and load the array

	// The format of the file is as follows (ugly, ugly Perl):
	// printf "%s %1.5f %1.5f %1.5f %1.5f %1.5f %1.5f %1.5f %1.5f %1.5f %1.5f %1.5f %1.5f %1.5f %1.5f\n",
	//	( $key, $avgAtrE, $avgRepE, $avgSolvE, $avgIntrarepE, $avgProCloseE, $avgPairE, $avgHbond_sr_bb, $avgHbond_lr_bb,
	//     $avgHbond_bb_sc, $avgHbond_sc, $avgRamaE, $avgOmegaE, $avgDunE, $avgPaappE );

	std::string aa_str;
	Real fa_atr = 0.0;
	Real fa_rep = 0.0;
	Real fa_sol = 0.0;
	Real fa_intra_rep = 0.0;
	Real pro_close = 0.0;
	Real fa_pair = 0.0;
	Real hbond_sr_bb = 0.0;
	Real hbond_lr_bb = 0.0;
	Real hbond_bb_sc = 0.0;
	Real hbond_sc = 0.0;
	Real rama = 0.0;
	Real omega = 0.0;
	Real fa_dun = 0.0;
	Real p_aa_pp = 0.0;

	utility::io::izstream stream;
	io::database::open( stream, "unfolded_state_residue_energies" );

	unfolded_energy_.resize( num_canonical_aas );

	while ( stream ) {
		using namespace ObjexxFCL::fmt;
		stream >> bite( 3, aa_str ) >> skip( 1 )
				>> bite( 7, fa_atr ) >> skip( 1 )
				>> bite( 7, fa_rep ) >> skip( 1 )
				>> bite( 7, fa_sol ) >> skip( 1 )
				>> bite( 7, fa_intra_rep ) >> skip( 1 )
				>> bite( 7, pro_close ) >> skip( 1 )
				>> bite( 7, fa_pair ) >> skip( 1 )
				>> bite( 7, hbond_sr_bb ) >> skip( 1 )
				>> bite( 7, hbond_lr_bb ) >> skip( 1 )
				>> bite( 7, hbond_bb_sc ) >> skip( 1 )
				>> bite( 7, hbond_sc ) >> skip( 1 )
				>> bite( 7, rama ) >> skip( 1 )
				>> bite( 7, omega ) >> skip( 1 )
				>> bite( 7, fa_dun ) >> skip( 1 )
				>> bite( 7, p_aa_pp ) >> skip;

		if ( stream ) {
			AA aa = aa_from_name( aa_str );
			assert( ( aa >= 1 ) && ( aa <= num_canonical_aas ) );

			// create an EnergyMap that holds the above values
			EnergyMap emap;
			emap[ scoring::fa_atr ] = fa_atr;
			emap[ scoring::fa_rep ] = fa_rep;
			emap[ scoring::fa_sol ] = fa_sol;
			emap[ scoring::fa_intra_rep ] = fa_intra_rep;
			emap[ scoring::pro_close ] = pro_close;
			emap[ scoring::fa_pair ] = fa_pair;
			emap[ scoring::hbond_sr_bb ] = hbond_sr_bb;
			emap[ scoring::hbond_lr_bb ] = hbond_lr_bb;
			emap[ scoring::hbond_bb_sc ] = hbond_bb_sc;
			emap[ scoring::hbond_sc ] = hbond_sc;
			emap[ scoring::rama ] = rama;
			emap[ scoring::omega ] = omega;
			emap[ scoring::fa_dun ] = fa_dun;
			emap[ scoring::p_aa_pp ] = p_aa_pp;

			unfolded_energy_[ aa ] = emap;
		}
	}
	stream.close();

	// add some possible error handling here!  how do you check for errors when reading in energies though?

}



void
UnfoldedStatePotential::unfolded_state_energy( chemical::AA const & aa, scoring::EnergyMap & e ) const {

	using namespace core::chemical;

	// Need to decide if/how/where to include NCAAs!
	if ( aa > chemical::num_canonical_aas )
		return;

	// the energies stored in the database file aren't probabilities; don't take the log of them, that doesn't make sense!
	// return unfolded_energy_[ res.aa() ]; // old way of returning unfolded state energies

	e = unfolded_energy_[ aa ];

}


void
UnfoldedStatePotential::pose_unfolded_state_energy( pose::Pose const & pose, scoring::EnergyMap & e ) const {

	EnergyMap unf_total;

	for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
		if ( ! pose.residue(ii).is_protein() ) {
			// Need to decide if/how/where to include NCAAs!
			// Alternately, can also do if ( aa > chemical::num_canonical_aas )
			continue;
		}

		// EnergyMap's know how to add, so we can take advantage of that feature to come up with a total unfolded state
		// energy (broken down by score type anyway).
		unf_total += unfolded_energy_[ pose.residue(ii).aa() ];
	}

	e = unf_total;

}



} // namespace scoring
} // namespace core

