// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/carbon_hbonds/CarbonHBondEnergy.fwd.hh
/// @brief  Hydrogen bond energy method forward declaration
/// @author Phil Bradley
/// @author Andrew Leaver-Fay
/// @author Rhiju Das


// Unit Headers
#include <core/scoring/carbon_hbonds/CarbonHBondEnergy.hh>
#include <core/scoring/carbon_hbonds/CarbonHBondEnergyCreator.hh>

// Package headers
#include <core/scoring/Energies.hh>
#include <core/scoring/EnergyGraph.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/scoring/carbon_hbonds/CarbonHBondPotential.hh>
#include <core/scoring/hbonds/HBondSet.hh>
#include <core/scoring/hbonds/hbonds.hh>
#include <core/scoring/hbonds/hbonds_geom.hh>
#include <core/scoring/hbonds/constants.hh>

// Project headers
#include <ObjexxFCL/formatted.o.hh>
#include <core/options/option.hh>
#include <core/options/keys/score.OptionKeys.gen.hh>

#include <core/pose/Pose.hh>
#include <core/util/Tracer.hh>
#include <core/util/prof.hh>

#include <numeric/xyzVector.hh>
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyz.io.hh>
#include <numeric/deriv/angle_deriv.hh>

static core::util::Tracer tr( "core.scoring.carbon_hbonds.CarbonHBondEnergy" );


namespace core {
namespace scoring {
namespace carbon_hbonds {


/// @details This must return a fresh instance of the CarbonHBondEnergy class,
/// never an instance already in use
methods::EnergyMethodOP
CarbonHBondEnergyCreator::create_energy_method(
	methods::EnergyMethodOptions const &
) const {
	return new CarbonHBondEnergy;
}

ScoreTypes
CarbonHBondEnergyCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( ch_bond );
	sts.push_back( ch_bond_sc_sc );
	sts.push_back( ch_bond_bb_sc );
	sts.push_back( ch_bond_bb_bb );
	return sts;
}


///@brief copy c-tor
CarbonHBondEnergy::CarbonHBondEnergy() :
	parent( new CarbonHBondEnergyCreator ),
	carbon_hbond_potential_( ScoringManager::get_instance()->get_CarbonHBondPotential() ),
	max_dis_( carbon_hbond_potential_.max_dis() ),
	max_dis2_( max_dis_*max_dis_ ),
	path_dist_cutoff_( 4 ),
	orientation_dep_rna_ch_o_bonds_( ! core::options::option[ core::options::OptionKeys::score::disable_orientation_dependent_rna_ch_o_bonds ]),
	verbose_( false )
{}

/// copy ctor
CarbonHBondEnergy::CarbonHBondEnergy( CarbonHBondEnergy const & src ):
	parent( src ),
	carbon_hbond_potential_( ScoringManager::get_instance()->get_CarbonHBondPotential() ),
	max_dis_( carbon_hbond_potential_.max_dis() ),
	max_dis2_( max_dis_*max_dis_ ),
	path_dist_cutoff_( src.path_dist_cutoff_ ),
	orientation_dep_rna_ch_o_bonds_( 	src.orientation_dep_rna_ch_o_bonds_ ),
	verbose_( src.verbose_ )
{}

/// clone
methods::EnergyMethodOP
CarbonHBondEnergy::clone() const
{
	return new CarbonHBondEnergy( *this );
}

///
void
CarbonHBondEnergy::setup_for_scoring( pose::Pose & /*pose*/, ScoreFunction const & ) const
{
	// nothing for now.

}

/////////////////////////////////////////////////////////////////////////////
void
CarbonHBondEnergy::setup_for_derivatives( pose::Pose & /*pose*/, ScoreFunction const & sfxn ) const
{
	wbb_bb_ = sfxn.weights()[ ch_bond ] + sfxn.weights()[ ch_bond_bb_bb ];
	wbb_sc_ = sfxn.weights()[ ch_bond ] + sfxn.weights()[ ch_bond_bb_sc ];
	wsc_sc_ = sfxn.weights()[ ch_bond ] + sfxn.weights()[ ch_bond_sc_sc ];
}


/////////////////////////////////////////////////////////////////////////////
// scoring
/////////////////////////////////////////////////////////////////////////////

/// Everything in here.
void
CarbonHBondEnergy::residue_pair_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const &,
	ScoreFunction const &,
	TwoBodyEnergyMap & emap
) const
{

	Real bb_bb(0.0);
	Real bb_sc(0.0);
	Real sc_sc(0.0);
	Real ch_bond_E =
		res_res_carbon_hbond_one_way( rsd1, rsd2, bb_bb, bb_sc, sc_sc) +
		res_res_carbon_hbond_one_way( rsd2, rsd1, bb_bb, bb_sc, sc_sc ) ;
	emap[ ch_bond_bb_bb ] += bb_bb;
	emap[ ch_bond_bb_sc ] += bb_sc;
	emap[ ch_bond_sc_sc ] += sc_sc;
	// store the energies
 	emap[ ch_bond ] += ch_bond_E;
	//	std::cout << "RUNNING SUM: " << rsd1.seqpos() <<  " " << rsd2.seqpos() << " " << ch_bond_E << " " << emap[ ch_bond ] << std::endl;

}


void
CarbonHBondEnergy::backbone_backbone_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const &,
	ScoreFunction const &,
	TwoBodyEnergyMap & emap
) const
{
	Real ch_bond_E =
		bb_bb_carbon_hbond_one_way( rsd1, rsd2 ) +
		bb_bb_carbon_hbond_one_way( rsd2, rsd1 ) ;

	// store the energies
	emap[ ch_bond ] += ch_bond_E;
	emap[ ch_bond_bb_bb ] += ch_bond_E;
	//	std::cout << "RUNNING SUM: " << rsd1.seqpos() <<  " " << rsd2.seqpos() << " " << ch_bond_E << " " << emap[ ch_bond ] << std::endl;
}


void
CarbonHBondEnergy::backbone_sidechain_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const &,
	ScoreFunction const &,
	TwoBodyEnergyMap & emap
) const
{
	Real ch_bond_E =
		bb_sc_carbon_hbond_one_way( rsd1, rsd2 ) +
		sc_bb_carbon_hbond_one_way( rsd2, rsd1 ) ;

	// store the energies
	emap[ ch_bond ] += ch_bond_E;
	emap[ ch_bond_bb_sc ] += ch_bond_E;

	//	std::cout << "RUNNING SUM: " << rsd1.seqpos() <<  " " << rsd2.seqpos() << " " << ch_bond_E << " " << emap[ ch_bond ] << std::endl;
}


void
CarbonHBondEnergy::sidechain_sidechain_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & ,
	ScoreFunction const & ,
	TwoBodyEnergyMap & emap
) const
{
	Real ch_bond_E =
		sc_sc_carbon_hbond_one_way( rsd1, rsd2 ) +
		sc_sc_carbon_hbond_one_way( rsd2, rsd1 ) ;

	// store the energies
	emap[ ch_bond ] += ch_bond_E;
	emap[ ch_bond_sc_sc ] += ch_bond_E;
	//	std::cout << "RUNNING SUM: " << rsd1.seqpos() <<  " " << rsd2.seqpos() << " " << ch_bond_E << " " << emap[ ch_bond ] << std::endl;
}


///////////////////////////////////////////////////////////////////////////////
// Look more than four atoms away.
bool
CarbonHBondEnergy::path_distance_OK(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	Size const ii,
	Size const jj
) const
{
	Size const & i( rsd1.seqpos() );
	Size const & j( rsd2.seqpos() );
	//	std::cout << i << " " << j << " " << ii << " " << jj << std::endl;
	if ( i == j && Size( rsd1.path_distance(ii,jj) ) <= path_dist_cutoff_) return false;
	else if ( rsd1.is_bonded( rsd2 ) ) {
		Size const path_size =
			rsd1.path_distance( ii, rsd1.connect_atom( rsd2 ) ) +
			rsd2.path_distance( jj, rsd2.connect_atom( rsd1 ) ) + 1;
		if ( path_size <= path_dist_cutoff_ ) return false;
	}
	return true;
}


/////////////////////////////////////////////////////////////////////
Real
CarbonHBondEnergy::res_res_carbon_hbond_one_way(
	conformation::Residue const & don_rsd,
	conformation::Residue const & acc_rsd,
	Real & bb_bb,
	Real & bb_sc,
	Real & sc_sc
) const
{

	Real res_res_energy( 0.0 ), energy( 0.0 );


	// Here we go -- cycle through non-polar hydrogens in don_aa, and all acceptors.
	for ( chemical::AtomIndices::const_iterator
			hnum  = don_rsd.Hpos_apolar().begin(),
			hnume = don_rsd.Hpos_apolar().end(); hnum != hnume; ++hnum ) {
		Size const don_h_atm( *hnum );

		//		std::cout << "Apolar hydrogen: " << don_rsd.atom_name( don_h_atm ) << " in " <<  don_rsd.name1() << don_rsd.seqpos()	<< std::endl;

		for ( chemical::AtomIndices::const_iterator
				anum  = acc_rsd.accpt_pos().begin(),
				anume = acc_rsd.accpt_pos().end(); anum != anume; ++anum ) {

			//check here whether is backbone
			Size const acc_atm( *anum );

			if ( get_atom_atom_carbon_hbond_energy( don_h_atm, don_rsd,
					acc_atm, acc_rsd, energy ) ) {
				if (don_rsd.atom_is_backbone(don_h_atm) && acc_rsd.atom_is_backbone(acc_atm)){
					//emap[ch_bond_bb_bb]+=energy;
					bb_bb +=energy;
				} else if (!don_rsd.atom_is_backbone(don_h_atm) && !acc_rsd.atom_is_backbone(acc_atm)){
					//emap[ch_bond_sc_sc]+=energy;
					sc_sc +=energy;
				} else {
					//emap[ch_bond_bb_sc]+=energy;
					bb_sc +=energy;
				}
				res_res_energy += energy;
			}
		}
	}

	return res_res_energy;
}

Real
CarbonHBondEnergy::bb_bb_carbon_hbond_one_way(
	conformation::Residue const & don_rsd,
	conformation::Residue const & acc_rsd
) const
{

	Real res_res_energy( 0.0 ), energy( 0.0 );

	// Here we go -- cycle through non-polar hydrogens in don_aa, and all acceptors.
	for ( chemical::AtomIndices::const_iterator
					hnum  = don_rsd.Hpos_apolar().begin(),
					hnume = don_rsd.Hpos_apolar().end(); hnum != hnume; ++hnum ) {
		Size const don_h_atm( *hnum );
		if ( don_h_atm >= don_rsd.first_sidechain_hydrogen() ) continue;
		//		std::cout << "Apolar hydrogen: " << don_rsd.atom_name( don_h_atm ) << " in " <<  don_rsd.name1() << don_rsd.seqpos()	<< std::endl;

		for ( chemical::AtomIndices::const_iterator
						anum  = acc_rsd.accpt_pos().begin(),
						anume = acc_rsd.accpt_pos().end(); anum != anume; ++anum ) {

			Size const acc_atm( *anum );
			if ( acc_atm > acc_rsd.last_backbone_atom() ) continue;

			get_atom_atom_carbon_hbond_energy( don_h_atm, don_rsd,
				acc_atm,   acc_rsd,
				energy );
			res_res_energy += energy;
		}
	}

	return res_res_energy;
}

Real
CarbonHBondEnergy::sc_bb_carbon_hbond_one_way(
	conformation::Residue const & don_rsd, // sidechain atoms on donor
	conformation::Residue const & acc_rsd  // backbone atoms on acceptor
) const
{

	Real res_res_energy( 0.0 ), energy( 0.0 );

	// Here we go -- cycle through non-polar hydrogens in don_aa, and all acceptors.
	for ( chemical::AtomIndices::const_iterator
					hnum  = don_rsd.Hpos_apolar().begin(),
					hnume = don_rsd.Hpos_apolar().end(); hnum != hnume; ++hnum ) {
		Size const don_h_atm( *hnum );
		if ( don_h_atm < don_rsd.first_sidechain_hydrogen() ) continue;
		//		std::cout << "Apolar hydrogen: " << don_rsd.atom_name( don_h_atm ) << " in " <<  don_rsd.name1() << don_rsd.seqpos()	<< std::endl;

		for ( chemical::AtomIndices::const_iterator
						anum  = acc_rsd.accpt_pos().begin(),
						anume = acc_rsd.accpt_pos().end(); anum != anume; ++anum ) {

			Size const acc_atm( *anum );
			if ( acc_atm > acc_rsd.last_backbone_atom() ) continue;

			get_atom_atom_carbon_hbond_energy( don_h_atm, don_rsd,
				acc_atm,   acc_rsd,
				energy );
			res_res_energy += energy;
		}
	}

	return res_res_energy;
}

Real
CarbonHBondEnergy::bb_sc_carbon_hbond_one_way(
	conformation::Residue const & don_rsd, // backbone atoms on donor
	conformation::Residue const & acc_rsd  // sidechain atoms on acceptor
) const
{

	Real res_res_energy( 0.0 ), energy( 0.0 );

	// Here we go -- cycle through non-polar hydrogens in don_aa, and all acceptors.
	for ( chemical::AtomIndices::const_iterator
					hnum  = don_rsd.Hpos_apolar().begin(),
					hnume = don_rsd.Hpos_apolar().end(); hnum != hnume; ++hnum ) {
		Size const don_h_atm( *hnum );
		if ( don_h_atm >= don_rsd.first_sidechain_hydrogen() ) continue;
		//		std::cout << "Apolar hydrogen: " << don_rsd.atom_name( don_h_atm ) << " in " <<  don_rsd.name1() << don_rsd.seqpos()	<< std::endl;

		for ( chemical::AtomIndices::const_iterator
						anum  = acc_rsd.accpt_pos().begin(),
						anume = acc_rsd.accpt_pos().end(); anum != anume; ++anum ) {

			Size const acc_atm( *anum );
			if ( acc_atm <= acc_rsd.last_backbone_atom() ) continue;

			get_atom_atom_carbon_hbond_energy( don_h_atm, don_rsd,
				acc_atm,   acc_rsd,
				energy );
			res_res_energy += energy;
		}
	}

	return res_res_energy;
}

Real
CarbonHBondEnergy::sc_sc_carbon_hbond_one_way(
	conformation::Residue const & don_rsd,
	conformation::Residue const & acc_rsd
) const
{

	Real res_res_energy( 0.0 ), energy( 0.0 );

	// Here we go -- cycle through non-polar hydrogens in don_aa, and all acceptors.
	for ( chemical::AtomIndices::const_iterator
					hnum  = don_rsd.Hpos_apolar().begin(),
					hnume = don_rsd.Hpos_apolar().end(); hnum != hnume; ++hnum ) {
		Size const don_h_atm( *hnum );
		if ( don_h_atm < don_rsd.first_sidechain_hydrogen() ) continue;
		//		std::cout << "Apolar hydrogen: " << don_rsd.atom_name( don_h_atm ) << " in " <<  don_rsd.name1() << don_rsd.seqpos()	<< std::endl;

		for ( chemical::AtomIndices::const_iterator
						anum  = acc_rsd.accpt_pos().begin(),
						anume = acc_rsd.accpt_pos().end(); anum != anume; ++anum ) {

			Size const acc_atm( *anum );
			if ( acc_atm <= acc_rsd.last_backbone_atom() ) continue;

			get_atom_atom_carbon_hbond_energy( don_h_atm, don_rsd,
				acc_atm,   acc_rsd,
				energy );
			res_res_energy += energy;
		}
	}

	return res_res_energy;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
bool
CarbonHBondEnergy::get_atom_atom_carbon_hbond_energy(
	Size const don_h_atm,
	conformation::Residue const & don_rsd,
	Size const acc_atm,
	conformation::Residue const & acc_rsd,
	Real & energy,
	bool const update_deriv /*= false*/,
	bool const is_don_h,
	Vector & f1 /*=ZERO_VECTOR*/,
	Vector & f2 /*=ZERO_VECTOR*/
) const
{

	energy = 0.0;
	f1 = 0.0;
	f2 = 0.0;

	Size const don_atm( don_rsd.atom_base( don_h_atm ) );
	Size const base_atm( acc_rsd.atom_base( acc_atm ) );

	if ( !path_distance_OK( don_rsd, acc_rsd, don_atm, acc_atm ) ) return false; // Look more than four atoms away.

	if ( std::abs( acc_rsd.atomic_charge( acc_atm ) ) < 1e-2 ) return false; //Acceptor atom is virtual!

	Vector const & don_h_atm_xyz( don_rsd.atom( don_h_atm ).xyz() );
	Vector const & don_atm_xyz( don_rsd.atom( don_atm ).xyz() );
	Vector const & acc_atm_xyz( acc_rsd.atom( acc_atm ).xyz() );
	Vector const & base_atm_xyz( acc_rsd.atom( base_atm ).xyz() );

	Real d2 = don_h_atm_xyz.distance_squared( acc_atm_xyz );
	if ( d2 > max_dis2_ ) return false;

	//Distance dist_H_A( std::sqrt( d2 ) );
	Vector H_A_vector = acc_atm_xyz - don_h_atm_xyz;
	Vector D_H_vector = don_h_atm_xyz - don_atm_xyz;
	Vector B_A_vector = acc_atm_xyz - base_atm_xyz;

	if ( orientation_dep_rna_ch_o_bonds_ && don_rsd.is_RNA() && acc_rsd.is_RNA() ) {

		Vector deriv_vector( 0.0 );


		Vector const r_H_A( acc_atm_xyz  - don_h_atm_xyz );
		Vector const z_D_H( ( don_h_atm_xyz - don_atm_xyz ).normalize() );

		energy = carbon_hbond_potential_.get_potential_RNA( r_H_A, z_D_H, update_deriv, deriv_vector );

		if ( update_deriv ) {
			f2 = -1.0 * deriv_vector;
			f1 = cross( f2, acc_atm_xyz );
		}

		if ( verbose_ && energy < -0.05 ) {
			Real const angle_DH_A = numeric::conversions::degrees( angle_radians( don_atm_xyz, don_h_atm_xyz, acc_atm_xyz ) );
			std::cout <<"CHbond [RNA]: "<< don_rsd.name1() << I(3,don_rsd.seqpos())<<
				" atom "<< don_rsd.atom_name( don_h_atm )<< " [ " <<
				don_rsd.atom_name( don_atm) <<
				" ] bonded to acc_res " <<
				acc_rsd.name1()<< I(3, acc_rsd.seqpos()) <<
				" atom "<< acc_rsd.atom_name( acc_atm ) <<
				" with energy "<< F(8,3,energy) << " [" << F(8,3,H_A_vector.length()) << " Angstroms; "
								<< angle_DH_A << " degrees ]" << std::endl;
		}


	} else {

		Vector deriv_vector;
		energy = carbon_hbond_potential_.get_potential( don_rsd.atom_type_index( don_atm ), H_A_vector, D_H_vector, B_A_vector, deriv_vector);

		if ( update_deriv ) {
			if (is_don_h) {
				f2 = deriv_vector;
				f1 = cross( f2, don_h_atm_xyz );
			}
			else {
				f2 = deriv_vector;
				f1 = cross( f2, acc_atm_xyz );
			}
		}

		if ( verbose_ && energy < -0.05 ) {
			std::cout <<"CHbond: "<< don_rsd.name1() << I(3,don_rsd.seqpos())<<
				" atom "<< don_rsd.atom_name( don_h_atm )<< " [ " <<
			don_rsd.atom_name( don_atm) <<
				" ] bonded to acc_res " <<
				acc_rsd.name1()<< I(3, acc_rsd.seqpos()) <<
				" atom "<< acc_rsd.atom_name( acc_atm ) <<
				" with energy "<< F(8,3,energy) << " [" << F(8,3,H_A_vector.length()) << "]" << std::endl;
		}

	}


	return true;
}

//////////////////////////////////////////////////////////////////////////////////////
// Stupid helper function
// These should probably live inside conformation::Residue.
//
bool
CarbonHBondEnergy::atom_is_apolar_h( conformation::Residue const & rsd, Size const atm ) const
{
	for ( chemical::AtomIndices::const_iterator
					hnum  = rsd.Hpos_apolar().begin(),
					hnume = rsd.Hpos_apolar().end(); hnum != hnume; ++hnum ) {
		Size const don_h_atm( *hnum );
		if ( don_h_atm == atm ) return true;
	}
	return false;
}
//////////////////////////////////////////////////////////////////////////////
// Stupid helper function
// These should probably live inside conformation::Residue.
bool
CarbonHBondEnergy::atom_is_acceptor( conformation::Residue const & rsd, Size const atm ) const
{
	for ( chemical::AtomIndices::const_iterator
					anum  = rsd.accpt_pos().begin(),
					anume = rsd.accpt_pos().end(); anum != anume; ++anum ) {
		Size const acc_atm( *anum );
		if ( acc_atm == atm ) return true;
	}
	return false;
}



//////////////////////////////////////////////////////////////////////////////
void
CarbonHBondEnergy::get_deriv_acceptor(
	conformation::Residue const & current_rsd,
	Size const current_atm,
	conformation::Residue const & other_rsd,
	Vector & F1,
	Vector & F2
) const
{

	Real dummy_energy( 0.0 );
	Vector f1( 0.0 ), f2( 0.0 );
	for ( chemical::AtomIndices::const_iterator
			anum  = other_rsd.accpt_pos().begin(),
			anume = other_rsd.accpt_pos().end(); anum != anume; ++anum ) {
		Size const acc_atm ( *anum );
		get_atom_atom_carbon_hbond_energy( current_atm, current_rsd,
			acc_atm, other_rsd,
			dummy_energy, true /*update_deriv*/, false /* is_don */, f1, f2 );
		Real w = get_deriv_weight_for_atom_pair( current_rsd, current_atm, other_rsd, acc_atm );
		F1 += w * f1;
		F2 += w * f2;
	}
}


//////////////////////////////////////////////////////////////////////////////
void
CarbonHBondEnergy::get_deriv_donor(
	conformation::Residue const & current_rsd,
	Size const current_atm,
	conformation::Residue const & other_rsd,
	Vector & F1,
	Vector & F2
) const
{

	Real dummy_energy( 0.0 );
	Vector f1( 0.0 ), f2( 0.0 );
	for ( chemical::AtomIndices::const_iterator
		hnum  = other_rsd.Hpos_apolar().begin(),
		hnume = other_rsd.Hpos_apolar().end(); hnum != hnume; ++hnum ) {

		Size const don_h_atm( *hnum );

		get_atom_atom_carbon_hbond_energy( don_h_atm, other_rsd,
			current_atm, current_rsd,
			dummy_energy, true /*update_deriv*/, true /* is_don */, f1, f2 );
		Real w = get_deriv_weight_for_atom_pair( current_rsd, current_atm, other_rsd, don_h_atm );

		F1 -= w*f1;
		F2 -= w*f2;
	}

}

Real
CarbonHBondEnergy::get_deriv_weight_for_atom_pair(
	conformation::Residue const & rsd1,
	Size const at1,
	conformation::Residue const & rsd2,
	Size const at2
) const
{
	if ( rsd1.atom_is_backbone( at1 ) ) {
		if ( rsd2.atom_is_backbone( at2 ) ) {
			return wbb_bb_;
		} else {
			return wbb_sc_;
		}
	} else if ( rsd2.atom_is_backbone(at2) ) {
		return wbb_sc_;
	} else {
		return wsc_sc_;
	}
}


//////////////////////////////////////////////////////////////////////////////
// Note that this computes every interaction *twice* -- three times if you
//  note that the score calculation above does most of the computation already.
// Oh well -- we currently assume derivative calculation doesn't happen too often!
//
void
CarbonHBondEnergy::eval_atom_derivative(
	id::AtomID const & atom_id,
	pose::Pose const & pose,
	kinematics::DomainMap const &,
	ScoreFunction const &,
	EnergyMap const & /*weights*/,
	Vector & F1,
	Vector & F2
) const
{
	// 	Real energy( 0.0 ); //not actually returned.
 	hbonds::HBond::Deriv deriv;

	EnergyGraph const & energy_graph( pose.energies().energy_graph() );

	Size const i( atom_id.rsd() );
 	conformation::Residue const & current_rsd( pose.residue( i ) );
	Size const current_atm( atom_id.atomno() );

	// 	Size const nres = pose.total_residue();
	// 	static bool const update_deriv( true );

	Vector f1( 0.0 ),f2( 0.0 ); // Accumulates!

	if ( atom_is_apolar_h( current_rsd, current_atm ) ){

		// Loop over all potential acceptors in the pose -- go over neighbors.
		for( graph::Graph::EdgeListConstIter
				iter = energy_graph.get_node( i )->const_edge_list_begin();
				iter != energy_graph.get_node( i )->const_edge_list_end();
				++iter ){
			Size j( (*iter)->get_other_ind( i ) );
			get_deriv_acceptor(  current_rsd, current_atm, pose.residue(j), f1, f2 );
		}

		//Intra residue...
		get_deriv_acceptor(  current_rsd, current_atm, pose.residue(i), f1, f2 );

	} else if ( atom_is_acceptor( current_rsd, current_atm ) ) {

		// Loop over all potential carbon hydrogen-bond donors in the pose  -- including within the same residue.
		for( graph::Graph::EdgeListConstIter
				iter = energy_graph.get_node( i )->const_edge_list_begin();
				iter != energy_graph.get_node( i )->const_edge_list_end();
				++iter ){
			Size j( (*iter)->get_other_ind( i ) );
			get_deriv_donor(  current_rsd, current_atm, pose.residue(j), f1, f2 );
		}

		//Intra residue...
		get_deriv_donor(  current_rsd, current_atm, pose.residue(i), f1, f2 );

	}


	F1 += /* already weighted -- weights[ ch_bond ] * */ f1;
	F2 += /* already weighted --  weights[ ch_bond ] * */ f2;

}


////////////////////////////////////////////////////////////////////////////////////////////
Distance
CarbonHBondEnergy::atomic_interaction_cutoff() const
{
	return hbonds::MAX_R + 1.35; // MAGIC NUMBER
}

////////////////////////////////////////////////////////////////////////////////////////////
bool
CarbonHBondEnergy::defines_intrares_energy( EnergyMap const & /*weights*/ ) const
{
	return true;
}

void
CarbonHBondEnergy::eval_intrares_energy(
	conformation::Residue const & rsd,
	pose::Pose const & ,
	ScoreFunction const & ,
	EnergyMap & emap
) const {

	Real bb_bb(0.0);
	Real bb_sc(0.0);
	Real sc_sc(0.0);
	Real const res_energy = res_res_carbon_hbond_one_way( rsd, rsd, bb_bb, bb_sc, sc_sc );
	emap[ ch_bond ] += res_energy;
	emap[ ch_bond_bb_bb ] += bb_bb;
	emap[ ch_bond_bb_sc ] += bb_sc;
	emap[ ch_bond_sc_sc ] += sc_sc;

	//	std::cout << "INTRARES" << rsd.seqpos() << " " << res_energy << " " << emap[ch_bond] << std::endl;

}

///@brief CarbonHBondEnergy is not context sensitive
void
CarbonHBondEnergy::indicate_required_context_graphs(
	utility::vector1< bool > & /*context_graphs_required*/
) const
{
	/*nothing*/
}


} // carbon_hbonds
} // scoring
} // core
