// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/CarbonHBondEnergy.hh
/// @brief  Hydrogen bond energy method class declaration
/// @author Phil Bradley
/// @author Andrew Leaver-Fay


#ifndef INCLUDED_core_scoring_carbon_hbonds_CarbonHBondEnergy_HH
#define INCLUDED_core_scoring_carbon_hbonds_CarbonHBondEnergy_HH

#include <core/types.hh>

// Unit Headers
#include <core/scoring/carbon_hbonds/CarbonHBondEnergy.fwd.hh>

// Package headers
#include <core/scoring/methods/ContextIndependentTwoBodyEnergy.hh>
#include <core/scoring/carbon_hbonds/CarbonHBondPotential.hh>
#include <numeric/xyzVector.hh>

// Project headers
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/EnergyMap.hh>

namespace core {
namespace scoring {
namespace carbon_hbonds {

static core::Vector DUMMY_VECTOR;

///
class CarbonHBondEnergy : public methods::ContextIndependentTwoBodyEnergy  {
public:
	typedef methods::ContextIndependentTwoBodyEnergy  parent;

public:

	///
	CarbonHBondEnergy();

	///@brief copy c-tor
	CarbonHBondEnergy( CarbonHBondEnergy const & src );

	/// clone
	virtual
	methods::EnergyMethodOP
	clone() const;

	///
	virtual
	void
	setup_for_scoring( pose::Pose & pose, ScoreFunction const & ) const;

	///
	virtual
	void
	setup_for_derivatives( pose::Pose & pose, ScoreFunction const & ) const;

	/////////////////////////////////////////////////////////////////////////////
	// scoring
	/////////////////////////////////////////////////////////////////////////////

	/// This evaluates everything for now,
	/// but eventually may want to split this
	/// based on backbone/backbone vs. others,
	/// as is carried out in HBondEnergy.cc
	virtual
	void
	residue_pair_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const &,
		TwoBodyEnergyMap & emap
	) const;


	/// @brief Splitting out based on bb/bb for the OnTheFly IGs
	virtual
	void
	backbone_backbone_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		TwoBodyEnergyMap & emap
	) const;


	virtual
	void
	backbone_sidechain_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		TwoBodyEnergyMap & emap
	) const;

	virtual
	void
	sidechain_sidechain_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		TwoBodyEnergyMap & emap
	) const;

	/// f1 and f2 are zeroed
	virtual
	void
	eval_atom_derivative(
		 id::AtomID const & atom_id,
		 pose::Pose const & pose,
		 kinematics::DomainMap const &,
		 ScoreFunction const &,
		 EnergyMap const & weights,
		 Vector & F1,
		 Vector & F2
	) const;

	///
	//	virtual
	//	void
	//	finalize_total_energy(
	//		pose::Pose & pose,
	//		ScoreFunction const &,
	//		EnergyMap & totals
	//	) const;

	virtual
	bool
	defines_intrares_energy( EnergyMap const & /*weights*/ ) const;

	virtual
	void
	eval_intrares_energy(
		conformation::Residue const & rsd,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap & emap
	) const;

	virtual
	Distance
	atomic_interaction_cutoff() const;

	Real
	hydrogen_interaction_cutoff2() const;

	///@brief CarbonHBondEnergy is context sensitive
	virtual
	void indicate_required_context_graphs(
		utility::vector1< bool > & context_graphs_required ) const;

private:

	bool
	path_distance_OK(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		Size const ii,
		Size const jj
	) const;

	Real
	res_res_carbon_hbond_one_way(
		conformation::Residue const & polar_rsd,
		conformation::Residue const & occ_rsd,
		Real & bb_bb,
		Real & bb_sc,
		Real & sc_sc
	) const;

	Real
	bb_bb_carbon_hbond_one_way(
		conformation::Residue const & don_rsd,
		conformation::Residue const & acc_rsd
	) const;

	Real
	sc_bb_carbon_hbond_one_way(
		conformation::Residue const & don_rsd, // sidechain atoms on donor
		conformation::Residue const & acc_rsd  // backbone atoms on acceptor
	) const;

	Real
	bb_sc_carbon_hbond_one_way(
		conformation::Residue const & don_rsd, // backbone atoms on donor
		conformation::Residue const & acc_rsd  // sidechain atoms on acceptor
	) const;

	Real
	sc_sc_carbon_hbond_one_way(
		conformation::Residue const & don_rsd,
		conformation::Residue const & acc_rsd
	) const;


	bool
	atom_is_apolar_h( conformation::Residue const & rsd, Size const atm ) const;

	bool
	atom_is_acceptor( conformation::Residue const & rsd, Size const atm ) const;

	/// @details Atom pair interaction energy; returns true if the interaction energy
	/// is nonzero.
	bool
	get_atom_atom_carbon_hbond_energy(
	  Size const don_h_atm,
		conformation::Residue const & don_rsd,
		Size const acc_atm,
		conformation::Residue const & acc_rsd,
		Real & energy,
		bool const update_deriv = false,
		bool const is_don_h = false,
		Vector & f1 = DUMMY_VECTOR,
		Vector & f2 = DUMMY_VECTOR
	) const;

private:

	void
	get_deriv_acceptor(
		conformation::Residue const & current_rsd,
		Size const current_atm,
		conformation::Residue const & other_rsd,
		Vector & F1,
		Vector & F2
	) const;

	void
	get_deriv_donor(
		conformation::Residue const & current_rsd,
		Size const current_atm,
		conformation::Residue const & other_rsd,
		Vector & F1,
		Vector & F2
	) const;

	Real
	get_deriv_weight_for_atom_pair(
		conformation::Residue const & rsd1,
		Size const at1,
		conformation::Residue const & rsd2,
		Size const at2
	) const;

private:

	/////////////////////////////////////////////////////////////////////////////
	// data
	/////////////////////////////////////////////////////////////////////////////

	// const-ref to scoring database
	carbon_hbonds::CarbonHBondPotential const & carbon_hbond_potential_;
	Real const max_dis_;
	Real const max_dis2_;

	Size const path_dist_cutoff_;

	bool const orientation_dep_rna_ch_o_bonds_;
	bool const verbose_;

	mutable Real wbb_bb_;
	mutable Real wbb_sc_;
	mutable Real wsc_sc_;
};

} // carbon_hbonds
} // scoring
} // core

#endif

