// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief


#ifndef INCLUDED_core_scoring_constraints_AngleConstraint_HH
#define INCLUDED_core_scoring_constraints_AngleConstraint_HH

#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/Func.hh>
#include <core/scoring/constraints/XYZ_Func.hh>
#include <core/scoring/constraints/FuncFactory.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/ScoreType.hh>
#include <core/pose/Pose.fwd.hh>

#include <core/id/AtomID.hh>
#include <core/util/Tracer.hh>

//Utility Headers

// C++ Headers

namespace core {
namespace scoring {
namespace constraints {

static core::util::Tracer TRACER("core.io.constraints");

/// @brief An Angular Constraint.
class AngleConstraint : public Constraint {
public:

	virtual std::string type() const {
		return "Angle";
	}

	virtual ConstraintOP clone() const {
		return new AngleConstraint( atom1_, atom2_, atom3_, func_, score_type() );
	}

	/// @brief read in constraint defiinition
	void read_def( std::istream& data, pose::Pose const& pose, FuncFactory const& func_factory );

	/// @brief compute score
	Real
	score(
		Vector const & xyz1,
		Vector const & xyz2,
		Vector const & xyz3
	) const;

	/// @brief compute score
	void
	score( XYZ_Func const & xyz, EnergyMap const &, EnergyMap & emap ) const
	{
		emap[ this->score_type() ] += score( xyz( atom1_ ), xyz( atom2_ ), xyz( atom3_ ) );
	}

	/// @brief compute atom deriv
	void
	fill_f1_f2(
		AtomID const & atom,
		conformation::Conformation const & conformation,
		Vector & F1,
		Vector & F2,
		EnergyMap const & weights
	) const;


	/// @brief Constructor
	AngleConstraint(
		AtomID const & a1,
		AtomID const & a2,
		AtomID const & a3,
		FuncOP func_in, // we take ownership of this guy
		ScoreType scotype = angle_constraint
	):
		Constraint( scotype ),
		atom1_(a1),
		atom2_(a2),
		atom3_(a3),
		func_( func_in )
	{}

	/// @brief Constructor without atom IDs -- if you create an AngleConstraint with
	/// this constructor, you must never call its score( XYZFunc ) method!  Dangerous and stupid!
	AngleConstraint(
		FuncOP func_in,
		ScoreType scoretype = angle_constraint
	):
		Constraint( scoretype ),
		func_( func_in )
	{}


	/// @brief number of atoms --- always 3 for angles
	Size
	natoms() const
	{
		return 3;
	}

	virtual
	ConstraintOP
	remap_resid( core::sequence::SequenceMapping const &seqmap ) const;

	/// @brief return AtomID for atom 1,2,3
	AtomID const &
	atom( Size const n ) const
	{
		switch( n ) {
		case 1:
			return atom1_;
		case 2:
			return atom2_;
		case 3:
			return atom3_;
		default:
			utility_exit_with_message( "AngleConstraint::atom() bad argument" );
		}
		return atom1_;
	}

	/// @brief output violation of constraint to out - returns 1 if violated ( i.e., func.show_violations() > 0 )
	Size show_violations( std::ostream& out, pose::Pose const& pose, Size verbose_level, core::Real threshold = 1  ) const;

	virtual void show(std::ostream& out ) const;

//private: /*functions*/

public:
	/// Previously private member functions made public so that in the absence of atom_ids,
	/// these functions could still be called externally.

	/// @brief evaluate func at theta
	Real
	func( Real const theta ) const
	{
		return func_->func( theta );
	}

	/// @brief evaluate dfunc at theta
	Real
	dfunc( Real const theta ) const
	{
		return func_->dfunc( theta );
	}

	static
	void
	p1_theta_deriv(
		Vector const & p1,
		Vector const & p2,
		Vector const & p3,
		Vector & f1,
		Vector & f2
	);


	static
	void
	helper(
		Vector const & M,
		Vector const & w,
		Vector & F1,
		Vector & F2
	);


	void
	p1_deriv(
		Vector const & p1,
		Vector const & p2,
		Vector const & p3,
		Vector & F1,
		Vector & F2
	) const;


	void
	p2_deriv(
		Vector const & p1,
		Vector const & p2,
		Vector const & p3,
		Vector & F1,
		Vector & F2
	) const;

private:
	// data
	AtomID atom1_, atom2_, atom3_;
	FuncOP func_;
};

} // constraints
} // scoring
} // core

#endif
