// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief


#ifndef INCLUDED_core_scoring_constraints_ConstantConstraint_HH
#define INCLUDED_core_scoring_constraints_ConstantConstraint_HH

#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/Func.hh>
#include <core/scoring/constraints/XYZ_Func.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/ScoreType.hh>

#include <core/id/AtomID.hh>

//Utility Headers

// C++ Headers

namespace core {
namespace scoring {
namespace constraints {

/// @brief A Constant Constraint.
class ConstantConstraint : public Constraint {
public:

	/// @brief compute score
	Real
	score() const {
		return func(0);
	}

	virtual std::string type() const {
		return "Constant";
	}

	/// @brief compute score
	void
	score( XYZ_Func const &, EnergyMap const &, EnergyMap & emap ) const
	{
		emap[ this->score_type() ] += score();
	}

	Real score( pose::Pose const& ) const {
		return func(0);
	}

	/// @brief compute atom deriv
	void
	fill_f1_f2(
		AtomID const & ,
		conformation::Conformation const & ,
		Vector & F1,
		Vector & F2,
		EnergyMap const &
	) const
	{ F1 = 0; F2 = 0; return; }


	/// @brief Constructor
	ConstantConstraint(
		FuncOP func_in, // we take ownership of this guy
		ScoreType scotype = constant_constraint
	):
		Constraint( scotype ),
		func_( func_in )
	{}

	/// @brief number of atoms --- zero
	Size
	natoms() const
	{
		return 0;
	}

	AtomID const &
	atom( Size const n ) const
	{
		switch( n ) {
		default:
			utility_exit_with_message( "ConstantConstraint::atom() - no atoms exist" );
		}
		return *(new AtomID); // never happens
	}

	/// @brief output violation of constraint (none!)
	Size show_violations( std::ostream& , pose::Pose const& , Size, core::Real  ) const { return 0; };

	void show( std::ostream& out ) const {
		out << "ConstantConstraint" << std::endl;
		func_->show( out );
	}

private:
	// functions

	/// @brief evaluate func at theta
	Real
	func( Real const theta ) const
	{
		return func_->func( theta );
	}

	/// @brief evaluate dfunc at theta
	Real
	dfunc( Real const theta ) const
	{
		return func_->dfunc( theta );
	}

	// data
	FuncOP func_;
};

} // constraints
} // scoring
} // core

#endif
