// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief

#include <core/scoring/constraints/ConstraintForest.hh>
#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/BigBinConstraint.hh>
#include <core/scoring/constraints/ConstraintIO.hh>

#include <core/pose/Pose.hh>
#include <core/util/Tracer.hh>
// Utility Headers

// C++ Headers
#include <vector>
#include <sstream>



namespace core {
namespace scoring {
namespace constraints {

using core::Real;

static util::Tracer tr("core.scoring.ConstraintForest");

static numeric::random::RandomGenerator cf_RG(63);

ConstraintForest::ConstraintForest() {
}

void ConstraintForest::read_forest(std::istream &in, core::pose::Pose &pose) {
  std::string name;
  Real prob;
  ConstraintBranchOP branch;
  ConstraintTreeOP tree;

  while(in.good()) {
    in >> name;
		if(!in.good())
			break;

		tr << "Tree " << name << std::endl;

    if(name == "}" || name == "")
      break;

    if(trees.count(name) > 0)
      tree = trees[name];
    else {
      tree = new ConstraintTree();
      trees[name] = tree;
    }

    branch = new ConstraintBranch();
    in >> prob;
//		tr << "Prob " << prob << std::endl;
    branch->read_branch(in, pose);
    tree->add_branch(prob, branch);
  }

  for(std::map<std::string, ConstraintTreeOP>::iterator it = trees.begin();
      it != trees.end();
      it++)
    it->second->normalize();
}

std::ostream & operator<<(std::ostream &out, const ConstraintForest &forest) {
  for(std::map<std::string, ConstraintTreeOP>::const_iterator it = forest.trees.begin();
      it != forest.trees.end();
      it++) {
		std::vector<Real>::iterator pit = it->second->probs.begin();
		std::vector<ConstraintBranchOP>::iterator bit = it->second->branches.begin();

		for(;
				pit != it->second->probs.end() && bit != it->second->branches.end();
				pit++, bit++)
			out << it->first << ' ' << *pit << ' ' << **bit << std::endl;
	}
	return out;
}

std::ostream & operator<<(std::ostream &out, const ConstraintBranch &branch) {
  for( protocols::jumping::PairingsList::const_iterator it = branch.pairings.begin();
       it != branch.pairings.end();
       it++ ) {
		protocols::jumping::Pairing p(*it);
		out << "JUMP " << p.pos1 << ' ' << p.pos2 << ' ' << p.orientation << ' ' << p.pleating << ' ';
	}
  for( ConstraintOPs::const_iterator it = branch.constraints.begin();
       it != branch.constraints.end();
       it++ ) {
		const core::scoring::constraints::ConstraintCOP p(*it);
    p->show(out);
	}
	out << " {\n" << branch.forest << "}";
	return out;
}

void ConstraintForest::generate_random_sample() {
	constraints.clear();
	pairings.clear();
  for(std::map<std::string, ConstraintTreeOP>::iterator it = trees.begin();
      it != trees.end();
      it++) {
		tr.Debug << "generate random sample for ConstraintTree: " << it->first << std::endl;
    it->second->generate_random_sample(constraints, pairings);
	}
}

ConstraintOPs ConstraintForest::get_constraints() {
	return constraints;
}

protocols::jumping::PairingsList ConstraintForest::get_pairings() {
	return pairings;
}

ConstraintTree::ConstraintTree() {
}

void ConstraintTree::add_branch(Real prob, ConstraintBranchOP branch) {
  branches.push_back(branch);
  probs.push_back(prob);
}

void ConstraintTree::normalize() {
  Real total = 0.0;

  for(std::vector<Real>::iterator it = probs.begin();
      it != probs.end();
      it++)
    total += *it;

  if(total > 1.0) {
    for(std::vector<Real>::iterator it = probs.begin();
        it != probs.end();
        it++)
      *it /= total;
  }
}

void ConstraintTree::generate_random_sample(ConstraintOPs &constraints, PairingsList &pairings) {
  Real finger = cf_RG.uniform();
  std::vector<Real>::iterator pit = probs.begin();
  std::vector<ConstraintBranchOP>::iterator bit = branches.begin();

  for(;
      pit != probs.end() && bit != branches.end();
      pit++, bit++) {
		tr.Trace << "finger: " << finger << " " << *pit << std::endl;
    finger -= *pit;

    if(finger < 0.0) {
      (*bit)->get_constraints(constraints, pairings);
			break;
		}
  }
	// Didn't choose any branch
}

ConstraintBranch::ConstraintBranch() {
}


void ConstraintBranch::read_branch(std::istream &in, core::pose::Pose &pose) {
  std::string type;
	ConstraintOP cons;
	std::string line;
	if(!in.good())
		return;

	//std::getline(in, line);
  //std::stringstream sin(line);

  while( in.good() ) {
    in >> type;
//		tr << "Branch " << type << std::endl;
		if(type == "BB_BIG_BIN") {
			cons = new BigBinConstraint();
      cons->read_constraint(in, pose);
			constraints.push_back(cons);
		} else if(type == "JUMP") {
			int res1, res2, orient, pleat;
			in >> res1 >> res2 >> orient >> pleat;
			pairings.push_back(Pairing(res1, res2, orient, pleat));
		} else if(type == "{") {
			if(in.good())
				forest.read_forest(in, pose);
			break;
		} else {
			cons = ConstraintIO::read_individual_constraint_new( in, pose, ConstraintIO::get_func_factory(), type );
			tr.Debug << "finished with constraint" << std::endl;
			if ( cons ) {
				constraints.push_back( cons );
			} else {
				utility_exit_with_message("UNRECOGNIZED CONSTRAINT TYPE\n");
			}
			break;
    }
  }
}

void ConstraintBranch::get_constraints(ConstraintOPs &cons, PairingsList &pairs) {
	tr.Trace <<" add Branch to constraints nr_const in Branch " << constraints.size() << std::endl;
	cons.insert(cons.end(), constraints.begin(), constraints.end());
	pairs.insert(pairs.end(), pairings.begin(), pairings.end());
	forest.generate_random_sample();
	ConstraintOPs fcons = forest.get_constraints();
	PairingsList fpairs = forest.get_pairings();
	cons.insert(cons.end(), fcons.begin(), fcons.end());
	pairs.insert(pairs.end(), fpairs.begin(), fpairs.end());
	tr.Trace << "final nr constraints: " << cons.size() << " from sub-forest: " << fcons.size() << std::endl;
}

} // constraints
} // scoring
} // core
