// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief

#include <core/scoring/constraints/Constraints.hh>

#include <core/scoring/EnergyMap.hh>
#include <core/pose/Pose.fwd.hh>

// Utility Headers

// C++ Headers
#include <algorithm>

namespace core {
namespace scoring {
namespace constraints {

Constraints::Constraints()
{}

Constraints::Constraints( Constraints const & other ) :
	ReferenceCount()
{
	copy_from( other );
}



ConstraintsOP
Constraints::clone() const
{

	ConstraintsOP new_constraints = new Constraints();
	new_constraints->copy_from( *this );
	return new_constraints;
}

Constraints const &
Constraints::operator = ( Constraints const & rhs )
{
	copy_from( rhs );
	return *this;
}


void
Constraints::copy_from( Constraints const & other ) {
	for ( ConstraintCOPs::const_iterator
			iter = other.constraints_.begin(), iter_end = other.constraints_.end();
			iter != iter_end; ++iter ) {
		// All Constraints are now immutable, and so do not ever need to be cloned.
		//constraints_.push_back( (*iter)->clone() );
		constraints_.push_back( (*iter) );
	}
}

// call the setup_for_derivatives for each constraint
void
Constraints::setup_for_scoring( pose::Pose &pose, ScoreFunction const &scfxn ) const {
	for ( ConstraintCOPs::const_iterator it= constraints_.begin(), ite=constraints_.end();
				it != ite; ++it ) {
		(*it)->setup_for_scoring( pose , scfxn );
	}
}

// call the setup_for_derivatives for each constraint
void
Constraints::setup_for_derivatives( pose::Pose &pose, ScoreFunction const &scfxn ) const {
	for ( ConstraintCOPs::const_iterator it= constraints_.begin(), ite=constraints_.end();
				it != ite; ++it ) {
		(*it)->setup_for_derivatives( pose , scfxn );
	}
}

void
Constraints::eval_atom_derivative(
	id::AtomID const & atom_id,
	conformation::Conformation const & conformation,
	EnergyMap const & weights,
	Vector & F1,
	Vector & F2
) const
{
	for ( ConstraintCOPs::const_iterator it= constraints_.begin(), ite=constraints_.end();
				it != ite; ++it ) {
		Constraint const & cst( **it );
		Vector f1(0.0), f2(0.0);
		cst.fill_f1_f2( atom_id, conformation, f1, f2, weights );
		F1 += f1;
		F2 += f2;
	}
}


/// private
/// does not zero the emap entries before accumulating
///
void
Constraints::energy(
	XYZ_Func const & xyz_func,
	EnergyMap const & weights,
	EnergyMap & emap
) const
{
	for ( ConstraintCOPs::const_iterator it=constraints_.begin(), ite = constraints_.end(); it != ite; ++it ) {
		Constraint const & cst( **it );
		cst.score( xyz_func, weights, emap );
	}
}

/// will fail if Residues dont contain all the necessary atoms
void
Constraints::residue_pair_energy(
	Residue const & rsd1,
	Residue const & rsd2,
	EnergyMap const & weights,
	EnergyMap & emap
) const
{
	ResiduePairXYZ const xyz_func( rsd1, rsd2 );
	energy( xyz_func, weights, emap );
}

/// will fail if Residues dont contain all the necessary atoms
void
Constraints::intra_residue_energy(
	Residue const & rsd,
	EnergyMap const & weights,
	EnergyMap & emap
) const
{
	ResidueXYZ const xyz_func( rsd );
	energy( xyz_func, weights, emap );
}

///
void
Constraints::conformation_energy(
	Conformation const & conformation,
	EnergyMap const & weights,
	EnergyMap & emap
) const
{
	ConformationXYZ const xyz_func( conformation );
	energy( xyz_func, weights, emap );
}

///
void
Constraints::add_constraint( ConstraintCOP cst )
{
	constraints_.push_back( cst );
}

/// @details If this list contains the same constraint multiple times,
/// only one copy of it is removed.
/// I can't imagine *why* that scenario would ever come up, but still...
bool
Constraints::remove_constraint( ConstraintCOP cst )
{
	ConstraintCOPs::iterator where = std::find( constraints_.begin(), constraints_.end(), cst );
	if( where == constraints_.end() ) return false;
	constraints_.erase( where );
	return true;
}

void
Constraints::show( std::ostream & out ) {
	for ( ConstraintCOPs::const_iterator it=constraints_.begin(), ite = constraints_.end();
				it != ite;
				++it ) {
		Constraint const & cst( **it );
		cst.show( out );
	}
}

void
Constraints::show_definition(
	std::ostream & out,
	pose::Pose const & pose
) const {
	for ( ConstraintCOPs::const_iterator it=constraints_.begin(), ite = constraints_.end();
				it != ite;
				++it ) {
		Constraint const & cst( **it );
		cst.show_def( out, pose );
	}
}

Size
Constraints::show_violations(
	std::ostream& out,
	pose::Pose const & pose,
	Size verbose_level,
	Real threshold
) {
	Size total_viol( 0 );
	Size total_cst( 0 );
	for ( ConstraintCOPs::const_iterator it=constraints_.begin(), ite = constraints_.end();
				it != ite;
				++it ) {
		Constraint const & cst( **it );
		total_viol+=cst.show_violations( out, pose, verbose_level, threshold );
		total_cst++;
	}
 	if ( verbose_level > 60 ) out << " of total: " << total_cst << " ";
	return total_viol;
}

} // constraints
} // scoring
} // core
